#include "TCPNetworkConnector.h"

#include <QHostAddress>
#include <QTcpSocket>

class TCPNetworkConnectorPrivate
{
public:
    int reconnectTimerId;

    QString address;
    quint16 port;

    QTcpSocket *socket;
    LoggingServicePerformer *performer;
    LoggingServicePerformerFactory *factory;
};

TCPNetworkConnector::TCPNetworkConnector(const QString &address, quint16 port, LoggingServicePerformerFactory *factory, QObject *parent)
    : NetworkServiceProvider(parent)
{
    this->d = new TCPNetworkConnectorPrivate;

    d->reconnectTimerId = -1;
    d->address = address;
    d->port = port;
    d->factory = factory;
    d->performer = NULL;
}

TCPNetworkConnector::~TCPNetworkConnector()
{
    if(d->socket->state() == QTcpSocket::ConnectedState)
    {
        qDebug() << "TCPNetworkConnector - Disconnecting from remote.";
        d->socket->close();
    }

    delete this->d;
}

bool TCPNetworkConnector::initialize()
{
    qDebug() << "TCPNetworkConnector - Instantiating TCP service connector.";
    d->socket = new QTcpSocket(this);

    qDebug() << "TCPNetworkConnector - Registering event handlers";
    QObject::connect(d->socket, SIGNAL(connected()), this, SLOT(onConnect()));
    QObject::connect(d->socket, SIGNAL(disconnected()), this, SLOT(onDisconnect()));
    QObject::connect(d->socket, SIGNAL(stateChanged(QAbstractSocket::SocketState)),
                     this, SLOT(onSocketStateChanged(QAbstractSocket::SocketState)));

    qDebug() << "TCPNetworkConnector - Attempting connect to remote.";
    d->socket->connectToHost(QHostAddress(d->address), d->port, QTcpSocket::WriteOnly);

    emit this->statusChanged("Attempting to connect.");
    return true;
}

void TCPNetworkConnector::timerEvent(QTimerEvent*)
{
    if(d->socket->state() == QAbstractSocket::UnconnectedState)
    {
        qDebug() << "TCPNetworkConnector - Attempting reconnect";
        d->socket->connectToHost(QHostAddress(d->address), d->port, QAbstractSocket::WriteOnly);
    }
}

void TCPNetworkConnector::onSocketStateChanged(QAbstractSocket::SocketState state)
{
    switch(state)
    {
    case QAbstractSocket::ConnectedState:
        emit this->statusChanged(QString("Connected to %1:%2").arg(d->socket->peerAddress().toString(), QString::number(d->socket->peerPort())));
        break;

    case QAbstractSocket::UnconnectedState:
        emit this->statusChanged(tr("Disconnected"));

        if(d->reconnectTimerId == -1)
        {
            d->reconnectTimerId = this->startTimer(2000);
        }
        break;

    default:
        break;
    }
}

void TCPNetworkConnector::onConnect()
{
    if(d->reconnectTimerId != -1)
    {
        this->killTimer(d->reconnectTimerId);
        d->reconnectTimerId = -1;
    }

    qDebug() << "TCPNetworkConnector - Connected to remote"
             << QString("%1:%2").arg(d->socket->peerAddress().toString(), QString::number(d->socket->peerPort()));

    qDebug() << "TCPNetworkConnector - Creating service performer.";
    d->performer = d->factory->createInstance(d->socket, this);
}

void TCPNetworkConnector::onDisconnect()
{
    qDebug() << "TCPNetworkConnector - Handling disconnect from"
             << QString("%1:%2").arg(d->socket->peerAddress().toString(), QString::number(d->socket->peerPort()));

    d->factory->destroyInstance(d->performer);
    d->performer = NULL;
}
