#include "NetworkServerPlugin.h"
#include "NetworkServerConfig.h"
#include "NetworkOptionsDialog.h"

#include "NetworkServiceProvider.h"

#include "TCPNetworkAcceptor.h"
#include "TCPNetworkConnector.h"
#include "UDPNetworkConnector.h"

#include <rx/columbus/ColumbusController.h>

#include <QDebug>
#include <QSettings>
#include <QHostAddress>
#include <QMaemo5ValueButton>

Q_EXPORT_PLUGIN2(network-plugin, NetworkServerPlugin)

class NetworkServerPluginPrivate
{
public:
    NetworkServerConfig *config;
    NetworkServerConfig *dialogConfig;

    PluginManager *manager;
    NetworkServiceProvider *provider;
    LoggingServicePerformerFactory *factory;

    QMaemo5ValueButton *control;
    NetworkOptionsDialog *dialog;
};

NetworkServerPlugin::NetworkServerPlugin(QObject *parent)
    : QObject(parent)
{
    qDebug() << "NetworkServerPlugin: ctor()";

    this->d = new NetworkServerPluginPrivate;
    d->config       = new NetworkServerConfig(this);
    d->dialogConfig = new NetworkServerConfig(this);
    d->manager      = NULL;
    d->provider     = NULL;
    d->factory      = NULL;

    d->control      = NULL;
    d->dialog       = NULL;
}

NetworkServerPlugin::~NetworkServerPlugin()
{
    qDebug() << "NetworkServerPlugin: dtor()";
    this->shutdown();
    delete this->d;
}

QString NetworkServerPlugin::id() const
{
    static const QString id = "network-server-plugin";
    return id;
}

QString NetworkServerPlugin::version() const
{
    static const QString version = "0.1.0";
    return version;
}

QString NetworkServerPlugin::friendlyName() const
{
    static const QString friendlyName = tr("Network");
    return friendlyName;
}

QWidget* NetworkServerPlugin::control()
{
    if(d->control == NULL)
    {
        qWarning() << "NetworkServerPlugin: Request for control before plugin initialization!";
    }

    return d->control;
}

bool NetworkServerPlugin::initialize(PluginManager *manager)
{
    qDebug() << "NetworkServerPlugin: Initializing plugin.";
    d->manager = manager;

    qDebug() << "NetworkServerPlugin: Loading configuration settings.";
    d->config->load();
    d->dialogConfig->load();

    d->control = new QMaemo5ValueButton(this->friendlyName());
    d->control->setValueText(tr("Disabled"));

    d->dialog = new NetworkOptionsDialog(d->dialogConfig, d->manager->controller()->logger(), d->control);

    QObject::connect(d->control, SIGNAL(clicked()), d->dialog, SLOT(show()));
    QObject::connect(d->dialog, SIGNAL(finished(int)), this, SLOT(onDialogFinished()));

    if(d->config->isEnabled() && d->config->isRunOnStartup())
    {
        qDebug() << "NetworkServerPlugin: Automatically starting up service.";
        this->startup();
    }

    return true;
}

bool NetworkServerPlugin::startup()
{
    qDebug() << "NetworkServerPlugin: Starting up services.";
    AbstractGeoLoggingFormatter *formatter = d->manager->controller()->logger()->formatters().value(d->config->formatId());

    if(formatter == NULL)
    {
        qWarning() << "NetworkServerPlugin: Unknown logging format:" << d->config->formatId();
        return false;
    }

    qDebug() << "NetworkServerPlugin: Constructing service factory for" << formatter->id() << "data streaming.";
    d->factory = new LoggingServicePerformerFactory(formatter, d->manager->controller()->logger(), this);

    if(d->config->transport() == NetworkServerConfig::TransportUdp)
    {
        if(d->config->udpConnectionType() == NetworkServerConfig::UdpUnicast)
        {
            qDebug() << "NetworkServerPlugin: Creating UDP unicast connector.";
            d->provider = new UDPNetworkConnector(QHostAddress(d->config->udpUnicastAddress()), d->config->udpUnicastPort(), d->factory, this);
        }
        else if(d->config->udpConnectionType() == NetworkServerConfig::UdpMulticast)
        {
            qDebug() << "NetworkServerPlugin: Creating UDP multicast connector. (NOT IMPLEMENTED)";
            this->shutdown();
            d->control->setValueText(tr("Not Implemented"));
            return false;
        }
    }
    else if(d->config->transport() == NetworkServerConfig::TransportTcp)
    {
        if(d->config->tcpConnectionType() == NetworkServerConfig::TcpClient)
        {
            qDebug() << "NetworkServerPlugin: Creating TCP client provider.";
            d->provider = new TCPNetworkConnector(d->config->tcpClientAddress(), d->config->tcpClientPort(), d->factory, this);
        }
        else if(d->config->tcpConnectionType() == NetworkServerConfig::TcpServer)
        {
            qDebug() << "NetworkServerPlugin: Creating TCP server provider.";
            d->provider = new TCPNetworkAcceptor(d->config->tcpServerAddress(), d->config->tcpServerPort(), d->factory, this);
        }
    }

    QObject::connect(d->provider, SIGNAL(statusChanged(QString)), d->control, SLOT(setValueText(QString)));
    return d->provider->initialize();
}

void NetworkServerPlugin::shutdown()
{
    qDebug() << "NetworkServerPlugin: Shutting down services.";

    if(d->provider != NULL)
    {
        qDebug() << "NetworkServerPlugin: Destroying redundant service provider.";
        delete d->provider;
        d->provider = NULL;
    }

    if(d->factory != NULL)
    {
        qDebug() << "NetworkServerPlugin: Destroying redundant performer factory.";
        delete d->factory;
        d->factory = NULL;
    }
}

void NetworkServerPlugin::onDialogFinished()
{
    qDebug() << "NetworkServerPlugin: Processing configuration.";
    if(*d->config == *d->dialogConfig)
    {
        qDebug() << "NetworkServerPlugin: Configuration not changed.";
        d->dialogConfig->save(); //  We need to do this just incase the
                                 // special isRunOnStartup value has changed.
        return;
    }

    d->dialogConfig->save();
    d->config->load();

    this->shutdown();

    if(!d->config->isEnabled())
    {
        qDebug() << "NetworkServerPlugin: Doing nothing, service is not enabled.";
        d->control->setValueText(tr("Disabled"));
        return;
    }

    this->startup();
}
