/*
 * garchivereader.c - routines to abstract archive reading logic
 *
 * Copyright (C) 2008 Benoit Goudreault-Emond (bgoudreaultemond@gmail.com)
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <string.h>
#include <ctype.h>
#include <strings.h> /* for strcasecmp */

#include "garchivereader.h"

static void
g_archive_reader_base_init(gpointer g_class)
{
    static gboolean initialized = FALSE;
    if(!initialized)
	initialized = TRUE;
}

GType
g_archive_reader_get_type(void)
{
    static GType type = 0;
    if(type)
	return type;
    static const GTypeInfo info = {
	sizeof(GArchiveReaderClass),
	g_archive_reader_base_init,
	NULL,
	NULL,
	NULL,
	NULL,
	0,
	0,
	NULL
    };
    type = g_type_register_static(G_TYPE_INTERFACE,
				  "GArchiveReader",
				  &info,
				  0);
    return type;
}

/* methods */
int    
g_archive_reader_register_path(GArchiveReader* reader,
			       const gchar* path)
{
    return G_ARCHIVE_READER_GET_CLASS(reader)->register_path(reader, path);
}

void
g_archive_reader_get_file_list(GArchiveReader* reader,
			       FileCallback callback,
			       gpointer user_data)
{
    G_ARCHIVE_READER_GET_CLASS(reader)->get_file_list(reader, callback, user_data);
}

GdkPixbuf*  
g_archive_reader_get_page     (GArchiveReader* reader,
			       const gchar* page,
			       gdouble* w,
			       gdouble* h)
{
    return G_ARCHIVE_READER_GET_CLASS(reader)->get_page(reader,
							page,
							w,
							h);
}

gchar*
g_archive_reader_get_last_error(GArchiveReader* reader)
{
    return G_ARCHIVE_READER_GET_CLASS(reader)->get_last_error(reader);
}

gboolean
g_archive_reader_is_reentrant(GArchiveReader* reader)
{
    return G_ARCHIVE_READER_GET_CLASS(reader)->is_reentrant(reader);
}

int
g_archive_reader_check_magic  (const gchar* path,
			       const magic_sign* magic)
{
    FILE* f;
    char sign[20];
    int retval = 0;

    if(!(f = fopen(path, "rb")))
    {
	return -1;
    }

    if(!fseek(f, magic->offs, SEEK_SET) &&
       fread(sign, 1, magic->len, f) == magic->len)
    {
	/* If the read went well, we need to compare the characters */
	/* strstr works here, but only if no \0's are in the string */
	/* and if we first terminate the string read too */
	sign[magic->len] = '\0';
	if(!strcmp(sign, magic->sign))
	{
	    fclose(f);
	    return 1;
	}
    }
    fclose(f);
    return 0;
}

static gint 
compare_ascii(gconstpointer first, gconstpointer second)
{
    return strcmp(*((const char**)first), *((const char**)second));
}

static gint 
compare_insensitive(gconstpointer first, gconstpointer second)
{
    return strcasecmp(*((const char**)first), *((const char**)second));
}

static gint 
compare_natural(gconstpointer first, gconstpointer second)
{
    const gchar* lhs = *(const gchar**)first;
    const gchar* rhs = *(const gchar**)second;
    int current_lhs_number = 0;
    int current_rhs_number = 0;
    while(*lhs || *rhs) {
	char lhs_char = isalpha(*lhs) ? tolower(*lhs) : *lhs;
	char rhs_char = isalpha(*rhs) ? tolower(*rhs) : *rhs;
	int lhs_digit = isdigit(lhs_char);
	int rhs_digit = isdigit(rhs_char);
    
	if(!lhs_digit && !rhs_digit) {
	    if(lhs_char < rhs_char)
		return -1;
	    else if(lhs_char > rhs_char)
		return 1;
	    else {
		if(lhs_char)
		    ++lhs;
		if(rhs_char)
		    ++rhs;
	    }
	} else {
	    if(lhs_digit) {
		current_lhs_number = current_lhs_number * 10 + (lhs_char - '0');
		++lhs;
	    }
	    if(rhs_digit) {
		current_rhs_number = current_rhs_number * 10 + (rhs_char - '0');
		++rhs;
	    }
	}
    }
    /* if we are here, need to compare numbers */
    return current_lhs_number - current_rhs_number;
}

GCompareFunc
g_archive_reader_translate_strategy(SortStrategy strategy)
{
    static GCompareFunc functions[] = {
	compare_ascii,
	compare_insensitive,
	compare_natural
    };

    g_return_val_if_fail(strategy >= 0, NULL);
    g_return_val_if_fail(strategy < 3, NULL);
  
    return functions[strategy];
}
