#include "editlistwidget.h"

EditListWidget::EditListWidget(const QString &fileName, QWidget *parent) :
    QMainWindow(parent)
{
    setAttribute(Qt::WA_Maemo5StackedWindow);
    setAttribute(Qt::WA_Maemo5AutoOrientation, true);
    filePath = fileName;
    QFileInfo f(fileName);
    setWindowTitle(f.baseName());
    listWidget = new QListWidget;
    listWidget->setContextMenuPolicy(Qt::ActionsContextMenu);
    listWidget->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    QAction *editSelected = new QAction(tr("Edit"), listWidget);
    QAction *delSelected = new QAction(tr("Delete"), listWidget);
    listWidget->addAction(editSelected);
    listWidget->addAction(delSelected);
    connect(editSelected, SIGNAL(triggered()), this, SLOT(editNumber()));
    connect(delSelected, SIGNAL(triggered()), this, SLOT(delNumber()));

    menuBar()->addAction(tr("Import contacts"), this, SLOT(importContactNumbers()));
    menuBar()->addAction(tr("Import list"), this, SLOT(importList()));
    menuBar()->addAction(tr("Add number"), this, SLOT(addNewNumber()));
    menuBar()->addAction(tr("Delete numbers"), this, SLOT(delNumbers()));
    menuBar()->addAction(tr("Clear all"), this, SLOT(clearList()));


    statusLabel = new QLabel(this);
    statusLabel->setStyleSheet("color: rgb(120, 120, 120); font-size: 20px");

    QWidget *centralWidget = new QWidget();
    QVBoxLayout *layout = new QVBoxLayout(centralWidget);
    layout->addWidget(listWidget);
    layout->addWidget(statusLabel);
    setCentralWidget(centralWidget);
    loadList();
    statusLabel->setText(tr("Count:[ %1 ] - Path:\"%2\"").arg(listWidget->count()).arg(filePath));
}
EditListWidget::~EditListWidget()
{
}

QMap<QString, QString> EditListWidget::fetchContacts()
{
    QContactManager *cm = new QContactManager;
    QList<QContactLocalId> contactIds = cm->contactIds();
    QContact currContact;
    QMap<QString, QString> contacts;
    foreach (const QContactLocalId& id, contactIds) {
        currContact = cm->contact(id);
        QString number(currContact.detail(QContactPhoneNumber::DefinitionName).value(QContactPhoneNumber::FieldNumber));
        if (!number.remove(QRegExp("[a-zA-Z+.-#()@ \t\n\v\f\r]")).isEmpty()){
            contacts.insert(number.remove(QRegExp("[a-zA-Z+.-#()@ \t\n\v\f\r]")), currContact.displayLabel());
        }
    }
    return contacts;
}

QString EditListWidget::cleanString(const QString &str, bool prefix)
{
    QString cleanednumber = str;
    if (cleanednumber.contains("unknown")){
        return cleanednumber.trimmed();
    }
    else if (prefix){
        return cleanednumber.remove(QRegExp("[a-zA-Z.-#()@ \t\n\v\f\r]"));
    }
    else {
        return cleanednumber.remove(QRegExp("[a-zA-Z+.-#()@ \t\n\v\f\r]"));
    }
}

void EditListWidget::delNumber()
{
    delete(listWidget->takeItem(listWidget->currentRow()));
    updateStatus();
    writeToFile();
}
void EditListWidget::addNewNumber()
{
    bool ok;
    QString text = QInputDialog::getText(this, tr("Enter new number"), "", QLineEdit::Normal, "", &ok);
    text = cleanString(text, false);
    if (ok && !text.isEmpty()){
        for (int i = 0; i < listWidget->count(); ++i){
            if (text.startsWith(listWidget->item(i)->text())){
                QMessageBox::information(this, tr("Duplicated entry!"), tr("%1 already existed in list at line %2 (%3)").arg(text).arg(i+1).arg(listWidget->item(i)->text()));
                return;
            }
        }
        listWidget->addItem(text);
        updateStatus();
        QFile list(filePath);
        if (list.exists()) {
            if (!list.open(QIODevice::WriteOnly | QIODevice::Text | QIODevice::Append)){
                return;
            }
            QTextStream out(&list);
            out << text << "\n";
            list.close();
        }
    }
}

void EditListWidget::editNumber()
{
    if (listWidget->count() < 1){
        return;
    }
    bool ok;
    QString text = QInputDialog::getText(this, tr("Edit number"), "", QLineEdit::Normal, listWidget->currentItem()->text(), &ok);
    text = cleanString(text, false);
    if (ok && !text.isEmpty()){
        listWidget->currentItem()->setText(text);
        writeToFile();
    }
}

void EditListWidget::loadList()
{
    QFile list(filePath);
    if (list.exists()) {
        if (!list.open(QIODevice::ReadOnly | QIODevice::Text)){
            return;
        }
        QTextStream in(&list);
        QString line = in.readLine();
        while (!line.isNull()) {
            listWidget->addItem(line);
            line = in.readLine();
        }
        list.close();
    }
}

void EditListWidget::clearList()
{
    listWidget->clear();
    updateStatus();
    writeToFile();
}

void EditListWidget::writeToFile()
{
    QFile list(filePath);
    if (list.exists()) {
        if (!list.open(QIODevice::WriteOnly | QIODevice::Text | QIODevice::Truncate)){
            return;
        }
        QTextStream out(&list);
        for (int i = 0; i < listWidget->count(); ++i){
            out << listWidget->item(i)->text() << "\n";
        }
        list.close();
    }
}

/* Edit and Import Window Related Functions */
void EditListWidget::delNumbers()
{
    QStringList clone;
    for (int i = 0; i < listWidget->count(); ++i){
        clone.append(listWidget->item(i)->text());
    }
    EditListWindow *editlist = new EditListWindow(clone, this);
    connect(editlist, SIGNAL(returnList(const QStringList&)), this, SLOT(updateList(const QStringList&)));
    editlist->show();
}

void EditListWidget::updateList(const QStringList &list)
{
    listWidget->clear();
    listWidget->addItems(list);
    updateStatus();
    writeToFile();
}

void EditListWidget::importContactNumbers()
{
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
    QMap<QString, QString> contacts = fetchContacts();
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
    ImportListWindow *importlist = new ImportListWindow(contacts, this);
    connect(importlist, SIGNAL(returnList(const QStringList&)), this, SLOT(addToList(const QStringList&)));
    importlist->show();
}

void EditListWidget::importList()
{
    QStringList items;
    QString fileName = QFileDialog::getOpenFileName(this, tr("Choose list/file"), "/home/user/MyDocs");
    if (!fileName.isEmpty()) {
        QFile list(fileName);
        if (!list.open(QIODevice::ReadOnly | QIODevice::Text)){
            return;
        }
        QTextStream in(&list);
        QString line = in.readLine();
        while (!line.isNull()) {
            items.append(line);
            line = in.readLine();
        }
        list.close();
        ImportFileWindow *importfile = new ImportFileWindow(items, this);
        connect(importfile, SIGNAL(returnList(const QStringList&)), this, SLOT(addToList(const QStringList&)));
        importfile->show();
    }
}

void EditListWidget::addToList(const QStringList &list)
{
    listWidget->addItems(list);
    updateStatus();
    writeToFile();
}

void EditListWidget::updateStatus()
{
    statusLabel->setText(tr("Count:[ %1 ] - Path:\"%2\"").arg(listWidget->count()).arg(filePath));
}

