/*
 * belltower
 * an app to find belltowers under Maemo 5
 *
 * Copyright (c) 2009 Thomas Thurman <tthurman@gnome.org>
 * Released under the GPL
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include <hildon/hildon.h>
#include <gtk/gtk.h>
#include <location/location-gps-device.h>
#include <location/location-distance-utils.h>
#include <dbus/dbus-glib.h>

#define MAX_FIELDS 50
#define MAX_RECENT 5
#define CONFIG_GENERAL_GROUP "General"
#define CONFIG_BOOKMARK_GROUP "Bookmarks"
#define CONFIG_RECENT_GROUP "Recent"
#define CONFIG_SEEN_CREDITS_KEY "seen_credits"
#define CONFIG_DIRECTORY "/home/user/.config/belltower"
#define CONFIG_FILENAME CONFIG_DIRECTORY "/belltower.ini"

/**
 * Somewhat arbitrary minimum number of belltowers in
 * one country for the country to be considered to have
 * "many" belltowers.
 */
#define MANY_BELLTOWERS 10

GtkWidget *window;
LocationGPSDevice *device;
GKeyFile *static_content;
GKeyFile *config;

typedef enum {
  /** stop scanning the database */
  FILTER_STOP,
  /** ignore this one */
  FILTER_IGNORE,
  /** add this one to the list */
  FILTER_ACCEPT
} FilterResult;

/*
  FIXME:
  We should really do this by looking at the header row of the table.
  They might decide to put in new columns some day.
*/
typedef enum {
  FieldPrimaryKey,
  FieldNationalGrid,
  FieldAccRef,
  FieldSNLat,
  FieldSNLong,
  FieldPostcode,
  FieldTowerBase,
  FieldCounty,
  FieldCountry,
  FieldDiocese,
  FieldPlace,
  FieldPlace2,
  FieldPlaceCL,
  FieldDedication,
  FieldBells,
  FieldWt,
  FieldApp,
  FieldNote,
  FieldHz,
  FieldDetails,
  FieldGF,
  FieldToilet,
  FieldUR,
  FieldPDNo,
  FieldPracticeNight,
  FieldPSt,
  FieldPrXF,
  FieldOvhaulYr,
  FieldContractor,
  FieldExtraInfo,
  FieldWebPage,
  FieldUpdated,
  FieldAffiliations,
  FieldAltName,
  FieldLat,
  FieldLong,
  FieldSimulator
} field;

typedef struct {
  int serial;

  /* the raw data */

  char* fields[MAX_FIELDS];
  int n_fields;
} tower;

static void show_towers_from_list (GSList *list, gchar *list_name);
static void free_tower_list (GSList *list);

static void
show_message (char *message)
{
  HildonNote* note = HILDON_NOTE
    (hildon_note_new_information (GTK_WINDOW (window),
				  message?message:
				  "Some message was supposed to be here."));
  gtk_dialog_run (GTK_DIALOG (note));
  gtk_widget_destroy (GTK_WIDGET (note));
}

/**
 * Loads the content of the static and dynamic data files.
 * Possibly puts up a warning if we can't load the static file.
 */
static void
load_config (void)
{
  static_content = g_key_file_new ();

  if (!g_key_file_load_from_file (static_content,
				  "/usr/share/belltower/static.ini",
				  G_KEY_FILE_NONE,
				  NULL))
    {
      show_message ("Could not load static content.  Attempting to continue.");
    }

  config = g_key_file_new ();
  /* it doesn't matter if this fails */
  g_key_file_load_from_file (config,
			     CONFIG_FILENAME,
			     G_KEY_FILE_KEEP_COMMENTS,
			     NULL);
}

/**
 * Saves the dynamic data file to disk.
 * Puts up a message if there was any error.
 */
static void
save_config (void)
{
  gchar *data;

  g_mkdir_with_parents (CONFIG_DIRECTORY, 0700);

  data = g_key_file_to_data (config, NULL, NULL);

  if (!g_file_set_contents (CONFIG_FILENAME,
			    data,
			    -1,
			    NULL))
    {
      show_message ("Could not write config file.");
    }

  g_free (data);
}

static gint
distance_to_tower (tower *details)
{
  char *endptr;
  double tower_lat;
  double tower_long;
  double km_distance;
  const double km_to_miles = 1.609344;

  tower_lat = strtod(details->fields[FieldLat], &endptr);
  if (*endptr) return -1;
  tower_long = strtod(details->fields[FieldLong], &endptr);
  if (*endptr) return -1;

  km_distance = location_distance_between (device->fix->latitude,
					   device->fix->longitude,
					   tower_lat,
					   tower_long);

  return (int) (km_distance / km_to_miles);
}

static gchar*
distance_to_tower_str (tower *details)
{
  int miles = distance_to_tower (details);

  if (miles==-1)
    {
      return g_strdup ("unknown");
    }
  else
    {
      return g_strdup_printf("%dmi", (int) miles);
    }
}

static void
call_dbus (DBusBusType type,
	   char *name,
	   char *path,
	   char *interface,
	   char *method,
	   char *parameter)
{
  DBusGConnection *connection;
  GError *error = NULL;

  DBusGProxy *proxy;

  connection = dbus_g_bus_get (type,
                               &error);
  if (connection == NULL)
    {
      show_message (error->message);
      g_error_free (error);
      return;
    }

  proxy = dbus_g_proxy_new_for_name (connection, name, path, interface);

  error = NULL;
  if (!dbus_g_proxy_call (proxy, method, &error,
			  G_TYPE_STRING, parameter,
			  G_TYPE_INVALID,
			  G_TYPE_INVALID))
    {
      show_message (error->message);
      g_error_free (error);
    }
}

static void
show_browser (gchar *url)
{
  call_dbus (DBUS_BUS_SESSION,
	     "com.nokia.osso_browser",
	     "/com/nokia/osso_browser/request",
	     "com.nokia.osso_browser",
	     "load_url",
	     url);
}

typedef FilterResult (*ParseDoveCallback)(tower *details, gpointer data);
typedef void (*ButtonCallback)(void);

GtkWidget *tower_window, *buttons, *tower_table;
HildonAppMenu *menu;

static void
add_table_field (char *name,
		 char *value)
{
  int row;
  GtkLabel *label;
  gchar *str;

  g_object_get(tower_table, "n-rows", &row);

  row++;

  gtk_table_resize (GTK_TABLE (tower_table), row, 2);

  label = GTK_LABEL (gtk_label_new (NULL));
  str = g_strdup_printf("<b>%s</b>", name);
  gtk_label_set_markup (label, str);
  g_free (str);
  gtk_label_set_justify (label, GTK_JUSTIFY_RIGHT);
  gtk_table_attach_defaults (GTK_TABLE (tower_table),
			     GTK_WIDGET (label),
			     0, 1, row, row+1);

  label = GTK_LABEL (gtk_label_new (value));
  gtk_label_set_justify (label, GTK_JUSTIFY_LEFT);
  gtk_table_attach_defaults (GTK_TABLE (tower_table),
			     GTK_WIDGET (label),
			     1, 2, row, row+1);
}

static void
add_button (char *label,
	    ButtonCallback callback)
{
  GtkWidget *button;

  button = gtk_button_new_with_label (label);
  g_signal_connect (button, "clicked", G_CALLBACK (callback), NULL);
  hildon_app_menu_append (menu, GTK_BUTTON (button));
  button = hildon_button_new_with_text (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT,
					HILDON_BUTTON_ARRANGEMENT_VERTICAL,
					label, NULL);
  g_signal_connect (button, "clicked", G_CALLBACK (callback), NULL);
  gtk_box_pack_end (GTK_BOX (buttons), button, FALSE, FALSE, 0);
}


char *tower_displayed = NULL;
char *tower_website = NULL;
char *tower_map = NULL;
char *tower_directions = NULL;
char *peals_list = NULL;

#define BUTTON_BOOKMARKED_YES "Remove from bookmarks"
#define BUTTON_BOOKMARKED_NO "Bookmark"

static void
bookmark_toggled (GtkButton *button,
		  gpointer dummy)
{
  if (g_key_file_get_boolean (config,
			      CONFIG_BOOKMARK_GROUP,
			      tower_displayed,
			      NULL))
    {

      /* it's bookmarked; remove the bookmark */

      if (!g_key_file_remove_key (config,
				  CONFIG_BOOKMARK_GROUP,
				  tower_displayed,
				  NULL))
	{
	  show_message ("Could not remove bookmark.");
	  return;
	}

      save_config ();
      gtk_button_set_label (button,
			    BUTTON_BOOKMARKED_NO);
    }
  else
    {
      /* it's not bookmarked; add a bookmark */

      g_key_file_set_boolean (config,
			      CONFIG_BOOKMARK_GROUP,
			      tower_displayed,
			      TRUE);

      save_config ();
      gtk_button_set_label (button,
			    BUTTON_BOOKMARKED_YES);
    }
}

static void
show_tower_website (void)
{
  show_browser (tower_website);
}

static void
show_tower_map (void)
{
  show_browser (tower_map);
}

static void
show_peals_list (void)
{
  show_browser (peals_list);
}

static FilterResult
get_countries_cb (tower *details,
		  gpointer data)
{
  GHashTable *hash = (GHashTable *)data;
  gpointer value;

  if (details->serial==0)
    return TRUE; /* header row */

  if (!g_hash_table_lookup_extended (hash,
				     details->fields[FieldCountry],
				     NULL, &value))
    {
      g_hash_table_insert (hash,
			   g_strdup(details->fields[FieldCountry]),
			   GINT_TO_POINTER (0));
    }
  else
    {
      g_hash_table_replace (hash,
			    g_strdup(details->fields[FieldCountry]),
			    GINT_TO_POINTER (GPOINTER_TO_INT (value)+1));
    }

  return FILTER_IGNORE;
}

typedef struct {
  GHashTable *hash;
  gchar *country_name;
} country_cb_data;

typedef struct {
  char *country;
  char *county;
} country_and_county;

static FilterResult
get_counties_cb (tower *details,
		 gpointer data)
{
  country_cb_data *d = (country_cb_data *)data;

  if (details->serial==0)
    return FILTER_IGNORE; /* header row */

  if (strcmp(details->fields[FieldCountry], d->country_name)!=0)
    return FILTER_IGNORE; /* wrong country */

  if (!g_hash_table_lookup_extended (d->hash,
				    details->fields[FieldCounty],
				     NULL, NULL))
    {
      g_hash_table_insert (d->hash,
			   g_strdup(details->fields[FieldCounty]),
			   g_strdup (details->fields[FieldCounty]));
    }

  return FILTER_IGNORE;
}

static FilterResult
get_nearby_towers_cb (tower *details,
		      gpointer data)
{
  if (details->serial==0)
    return FILTER_IGNORE; /* header row */

  if (distance_to_tower (details) < 50)
    {
      return FILTER_ACCEPT;
    }
  else
    {
      return FILTER_IGNORE;
    }
}

static FilterResult
get_towers_by_county_cb (tower *details,
			 gpointer data)
{
  country_and_county *cac = (country_and_county *) data;

  if ((!cac->county || strcmp (cac->county, details->fields[FieldCounty])==0) &&
      (!cac->country || strcmp (cac->country, details->fields[FieldCountry])==0))
    {
      return FILTER_ACCEPT;
    }
  else
    {
      return FILTER_IGNORE;
    }
}

static FilterResult
get_towers_by_search_cb (tower *details,
			 gpointer data)
{
  char *s = (char *) data;

  if (strcasestr(details->fields[FieldCountry], s) ||
      strcasestr(details->fields[FieldCounty], s) ||
      strcasestr(details->fields[FieldDedication], s) ||
      strcasestr(details->fields[FieldPlace], s))
    {
      return FILTER_ACCEPT;
    }
  else
    {
      return FILTER_IGNORE;
    }
}

/**
 * A filter which accepts towers based on whether they
 * appear in a particular group in the config file.
 *
 * \param details  the candidate tower
 * \param data     pointer to a char* which names the group
 */
static FilterResult
get_group_of_towers_cb (tower *details,
			  gpointer data)
{
  if (g_key_file_has_key (config,
			  (char*) data,
			  details->fields[FieldPrimaryKey],
			  NULL))
    {
      return FILTER_ACCEPT;
    }
  else
    {
      return FILTER_IGNORE;
    }
}

/**
 * Removes the oldest entry from the [Recent] group in the config
 * file until there are only five entries left.  Does not save
 * the file; you have to do that.
 */
static void
remove_old_recent_entries (void)
{
  gint count;

  do
    {
      gchar **towers;
      gint oldest_date = 0;
      gchar *oldest_tower = NULL;
      gint i;

      /* It is a bit inefficient to do this every
       * time we go around the loop.  However, it
       * makes the code far simpler, and we almost
       * never go around more than once.
       */
      towers = g_key_file_get_keys (config,
				    CONFIG_RECENT_GROUP,
				    &count,
				    NULL);

      if (count <= MAX_RECENT)
	/* everything's fine */
	return;

      for (i=0; i<count; i++)
	{
	  gint date = g_key_file_get_integer (config,
					      CONFIG_RECENT_GROUP,
					      towers[i],
					      NULL);

	  if (date==0)
	    continue;

	  if (oldest_date==0 ||
	      date < oldest_date)
	    {
	      oldest_tower = towers[i];
	      oldest_date = date;
	    }
	}

      if (oldest_tower)
	{
	  g_key_file_remove_key (config,
				 CONFIG_RECENT_GROUP,
				 oldest_tower,
				 NULL);
	  count --;
	}
      g_strfreev (towers);
    }
  while (count > MAX_RECENT);
}

static FilterResult
single_tower_cb (tower *details,
		 gpointer data)
{

  GtkWidget *hbox, *button;
  gchar *str;
  gint tenor_weight;
  gchar *primary_key = (gchar*) data;
  gchar *miles;

  if (strcmp(details->fields[FieldPrimaryKey], primary_key)!=0)
    {
      /* not this one; keep going */
      return FILTER_IGNORE;
    }

  tower_window = hildon_stackable_window_new ();

  str = g_strdup_printf("%s, %s",
			details->fields[FieldDedication],
			details->fields[FieldPlace]);

  hildon_window_set_markup (HILDON_WINDOW (tower_window),
			    str);
  g_free (str);

  hbox = gtk_hbox_new (FALSE, 0);
  tower_table = gtk_table_new (0, 2, FALSE);
  buttons = gtk_vbox_new (TRUE, 0);
  menu = HILDON_APP_MENU (hildon_app_menu_new ());

  miles = distance_to_tower_str(details);

  add_table_field ("Distance", miles);
  add_table_field ("Postcode", details->fields[FieldPostcode]);
  add_table_field ("County", details->fields[FieldCounty]);
  add_table_field ("Country", details->fields[FieldCountry]);
  add_table_field ("Diocese", details->fields[FieldDiocese]);
  add_table_field ("Practice night", details->fields[FieldPracticeNight]);
  add_table_field ("Bells", details->fields[FieldBells]);

  g_free (miles);

  tenor_weight = atoi (details->fields[FieldWt]);
  str = g_strdup_printf("%dcwt %dqr %dlb in %s",
			tenor_weight/112,
			(tenor_weight % 112)/28,
			tenor_weight % 28,
			details->fields[FieldNote]
			);
  add_table_field ("Tenor", str);
  g_free (str);

  if (strcmp(details->fields[FieldWebPage], "")!=0)
    {
      add_button ("Tower website", show_tower_website);
    }
  add_button ("Peals", show_peals_list);
  add_button ("Map", show_tower_map);
  add_button ("Directions", NULL);

  /* don't use a toggle button: it looks stupid */
  button = hildon_button_new_with_text (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT,
					HILDON_BUTTON_ARRANGEMENT_VERTICAL,
					g_key_file_get_boolean (config,
								CONFIG_BOOKMARK_GROUP,
								details->fields[FieldPrimaryKey],
								NULL)?
					BUTTON_BOOKMARKED_YES: BUTTON_BOOKMARKED_NO,
					NULL);
  g_signal_connect (button, "clicked", G_CALLBACK (bookmark_toggled), NULL);
  gtk_box_pack_start (GTK_BOX (buttons), button, FALSE, FALSE, 0);

  gtk_widget_show_all (GTK_WIDGET (menu));
  hildon_window_set_app_menu (HILDON_WINDOW (tower_window), menu);

  gtk_box_pack_end (GTK_BOX (hbox), buttons, TRUE, TRUE, 0);
  gtk_box_pack_end (GTK_BOX (hbox), tower_table, TRUE, TRUE, 0);

  gtk_container_add (GTK_CONTAINER (tower_window), hbox);

  g_free (tower_website);
  tower_website = g_strdup_printf ("http://%s", details->fields[FieldWebPage]);
  g_free (peals_list);
  peals_list = g_strdup_printf ("http://www.pealbase.ismysite.co.uk/felstead/tbid.php?tid=%s",
       details->fields[FieldTowerBase]);
  g_free (tower_map);
  tower_map = g_strdup_printf ("http://maps.google.com/maps?q=%s,%s",
        details->fields[FieldLat],
        details->fields[FieldLong]);
  g_free (tower_displayed);
  tower_displayed = g_strdup (details->fields[FieldPrimaryKey]);

  g_key_file_set_integer (config,
			  CONFIG_RECENT_GROUP,
			  tower_displayed,
			  time (NULL));
  remove_old_recent_entries ();
  save_config ();

  gtk_widget_show_all (GTK_WIDGET (tower_window));

  return FILTER_STOP;
}

/**
 * A tower that was accepted by a filter.
 */
typedef struct {
  char *sortkey;
  char *primarykey;
  char *displayname;
} FoundTower;

static FoundTower *
found_tower_new (tower *basis)
{
  FoundTower* result = g_new (FoundTower, 1);

  result->sortkey = g_strdup (basis->fields[FieldPrimaryKey]);
  result->primarykey = g_strdup (basis->fields[FieldPrimaryKey]);

  if (device->fix->fields & LOCATION_GPS_DEVICE_LATLONG_SET)
    {
      gchar *distance = distance_to_tower_str (basis);
      result->displayname = g_strdup_printf ("%s, %s (%s, %s) (%s)",
					     basis->fields[FieldDedication],
					     basis->fields[FieldPlace],
					     basis->fields[FieldBells],
					     basis->fields[FieldPracticeNight],
					     distance);
      g_free (distance);
    }
  else
    {
      result->displayname = g_strdup_printf ("%s, %s (%s, %s)",
					     basis->fields[FieldDedication],
					     basis->fields[FieldPlace],
					     basis->fields[FieldBells],
					     basis->fields[FieldPracticeNight]);
    }

  return result;
}

static void
found_tower_free (FoundTower *tower)
{
  g_free (tower->sortkey);
  g_free (tower->primarykey);
  g_free (tower->displayname);
  g_free (tower);
}

/**
 * Calls a given function once for each tower in the world.
 * (The first call, however, is a header row.)
 *
 * \param callback       The function to call.
 * \param data           Arbitrary data to pass to the callback.
 * \param dialogue_title If non-NULL, a list will be displayed
 *                       with the results.  This is the title
 *                       used for that dialogue.  (The dialogue
 *                       will automatically free filter_results.)
 */
static void
parse_dove (ParseDoveCallback callback,
	    gpointer data,
	    gchar *dialogue_title)
{
  FILE *dove = fopen("/usr/share/belltower/dove.txt", "r");
  char tower_rec[4096];
  tower result;
  char *i;
  gboolean seen_newline;
  GSList *filter_results = NULL;

  if (!dove)
    {
      show_message ("Cannot open Dove database!");
      exit (255);
    }

  result.serial = 0;

  while (fgets (tower_rec, sizeof (tower_rec), dove))
    {
      seen_newline = FALSE;
      result.fields[0] = tower_rec;
      result.n_fields = 0;
      for (i=tower_rec; *i; i++) {
	if (*i=='\n')
	  {
	    seen_newline = TRUE;
	  }
	if (*i=='\\' || *i=='\n')
	  {
	    *i = 0;
	    result.n_fields++;
	    result.fields[result.n_fields] = i+1;
	  }
      }

      if (!seen_newline)
	{
	  /* keep it simple, stupid */
	  show_message ("Line too long, cannot continue.");
	  exit (255);
	}

      if (strcmp (result.fields[FieldCountry], "")==0)
	{
	  result.fields[FieldCountry] = "England";
	}

      switch (callback (&result, data))
	{
	case FILTER_IGNORE:
	  /* nothing */
	  break;

	case FILTER_STOP:
	  fclose (dove);
	  return;

	case FILTER_ACCEPT:
	  filter_results = g_slist_append (filter_results,
					   found_tower_new (&result));
	}

      result.serial++;
    }

  fclose (dove);

  if (dialogue_title)
    {
      show_towers_from_list (filter_results,
			     dialogue_title);
    }
  else
    {
      free_tower_list (filter_results);
    }
}

static void
show_tower (char *primary_key)
{
  parse_dove (single_tower_cb, primary_key, NULL);
}

static void
free_tower_list (GSList *list)
{
  GSList *cursor = list;

  while (cursor)
    {
      found_tower_free ((FoundTower*) cursor->data);
      cursor = cursor->next;
    }

  g_slist_free (list);
}

/**
 * Displays a list of towers for the user to choose from.
 * When one is chosen, we go to the display page for that tower.
 * If there are none, this will tell the user there were none.
 * If there is only one, we go straight to its display page.
 *
 * \param list       a GSList of FoundTower objects.
 * \param list_name  the title for the dialogue.
 */
static void
show_towers_from_list (GSList *list,
		       gchar *list_name)
{
  GtkWidget *dialog;
  GtkWidget *selector;
  gint result = -1;
  GSList *cursor;

  if (!list)
    {
      hildon_banner_show_information(window,
				     NULL,
				     "No towers found.");
      return;
    }

  if (!list->next)
    {
      /* only one; don't bother showing the list */
      FoundTower* found = (FoundTower*) list->data;

      hildon_banner_show_information(window,
				     NULL,
				     "One tower found.");
      show_tower (found->primarykey);

      free_tower_list (list);
      return;
    }

  dialog = hildon_picker_dialog_new (GTK_WINDOW (window));
  selector = hildon_touch_selector_new_text ();
  gtk_window_set_title (GTK_WINDOW (dialog), list_name);

  for (cursor=list; cursor; cursor=cursor->next)
    {
      FoundTower* found = (FoundTower*) cursor->data;
      hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector),
					 found->displayname);
    }

  hildon_picker_dialog_set_selector (HILDON_PICKER_DIALOG (dialog),
				     HILDON_TOUCH_SELECTOR (selector));

  gtk_widget_show_all (GTK_WIDGET (dialog));

  if (gtk_dialog_run (GTK_DIALOG (dialog))==GTK_RESPONSE_OK)
    {
      GList *rows = hildon_touch_selector_get_selected_rows (HILDON_TOUCH_SELECTOR (selector),
							     0);
      GtkTreePath *path = (GtkTreePath*) rows->data;
      gint *indices = gtk_tree_path_get_indices (path);

      result = *indices;
    }

  gtk_widget_destroy (GTK_WIDGET (dialog));

  if (result!=-1)
    {
      FoundTower *found = (FoundTower *) g_slist_nth_data (list, result);
      show_tower (found->primarykey);
    }

  free_tower_list (list);
}

static gint strcmp_f (gconstpointer a,
		      gconstpointer b)
{
  return strcmp ((char*)a, (char*)b);
}

static void
put_areas_into_list (gpointer key,
		     gpointer value,
		     gpointer data)
{
  GSList **list = (GSList **)data;
  *list = g_slist_insert_sorted (*list,
				 value,
				 strcmp_f);
}

static void
nearby_towers (void)
{
  if (!(device->fix->fields & LOCATION_GPS_DEVICE_LATLONG_SET))
    {
      show_message ("I don't know where you are!");
      return;
    }

  parse_dove (get_nearby_towers_cb,
	      NULL,
	      "Towers within fifty miles of you");
}

static void
towers_by_subarea (gchar *area)
{
  GtkWidget *dialog = hildon_picker_dialog_new (GTK_WINDOW (window));
  GtkWidget *selector = hildon_touch_selector_new_text ();
  GHashTable *hash = g_hash_table_new_full (g_str_hash,
					    g_str_equal,
					    g_free,
					    g_free);
  GSList *list=NULL, *cursor;
  gchar *title = g_strdup_printf ("Areas of %s", area);
  country_cb_data d = { hash, area };
  country_and_county cac = { area, NULL };

  gtk_window_set_title (GTK_WINDOW (dialog), title);
  g_free (title);

  parse_dove (get_counties_cb, &d, NULL);

  g_hash_table_foreach (hash,
			put_areas_into_list,
			&list);

  for (cursor=list; cursor; cursor=cursor->next)
    {
      hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector),
					 cursor->data);
    }

  hildon_picker_dialog_set_selector (HILDON_PICKER_DIALOG (dialog),
				     HILDON_TOUCH_SELECTOR (selector));

  gtk_widget_show_all (GTK_WIDGET (dialog));

  if (gtk_dialog_run (GTK_DIALOG (dialog))==GTK_RESPONSE_OK)
    {
      gchar *title;
      cac.county = g_strdup (hildon_touch_selector_get_current_text (HILDON_TOUCH_SELECTOR (selector)));
      title = g_strdup_printf ("Towers in %s",
			       cac.county);

      parse_dove (get_towers_by_county_cb,
		  &cac,
		  title);
      g_free (cac.county);
      g_free (title);
    }
  g_hash_table_unref (hash);
  gtk_widget_destroy (GTK_WIDGET (dialog));
}

/**
 * Maps a hash table from country names to counts of belltowers to a
 * newly-created hash table mapping country names to display
 * names, containing only those countries which have many
 * (or few) belltowers.
 *
 * \param source    the source table
 * \param want_many true if you want countries with many belltowers;
 *                  false if you want countries with few.
 */
static GHashTable*
get_countries_with_many (GHashTable *source,
			 gboolean want_many)
{
  GHashTable *result = g_hash_table_new_full (g_str_hash,
					      g_str_equal,
					      g_free,
					      NULL);
  GList *countries = g_hash_table_get_keys (source);
  GList *cursor = countries;

  while (cursor)
    {
      gboolean has_many =
	GPOINTER_TO_INT (g_hash_table_lookup (source,
					      cursor->data)) >= MANY_BELLTOWERS;

      if (has_many == want_many)
	{
	  g_hash_table_insert (result,
			       g_strdup (cursor->data),
			       g_strdup (cursor->data));
	}

      cursor = cursor->next;
    }

  g_list_free (countries);
  return result;
}

#define COUNTRIES_WITH_MANY "Countries with many belltowers"
#define COUNTRIES_WITH_FEW "Countries with few belltowers"

/**
 * Displays a list of areas of the world with many (or few)
 * belltowers.  If you ask for the areas with many, it include
 * a link to the areas with few.
 *
 * \param countries_with_many  True to list countries with many;
 *                             false to list countries with few.
 */
static void
towers_by_area_with_many (gboolean countries_with_many)
{
  GtkWidget *dialog = hildon_picker_dialog_new (GTK_WINDOW (window));
  GtkWidget *selector = hildon_touch_selector_new_text ();
  GHashTable *countries_to_counts = g_hash_table_new_full (g_str_hash,
							   g_str_equal,
							   g_free,
							   NULL);
  GHashTable *country_names;
  GSList *list = NULL, *cursor;
  gchar *result = NULL;

  gtk_window_set_title (GTK_WINDOW (dialog),
			countries_with_many?
			COUNTRIES_WITH_MANY : COUNTRIES_WITH_FEW);

  parse_dove (get_countries_cb, countries_to_counts, NULL);

  country_names = get_countries_with_many (countries_to_counts,
					   countries_with_many);

  g_hash_table_foreach (country_names,
			put_areas_into_list,
			&list);

  for (cursor=list; cursor; cursor=cursor->next)
    {
      hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector),
					 cursor->data);
    }

  if (countries_with_many)
    {
      hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector),
					 COUNTRIES_WITH_FEW);
    }

  hildon_picker_dialog_set_selector (HILDON_PICKER_DIALOG (dialog),
				     HILDON_TOUCH_SELECTOR (selector));

  gtk_widget_show_all (GTK_WIDGET (dialog));

  if (gtk_dialog_run (GTK_DIALOG (dialog))==GTK_RESPONSE_OK)
    {
      result = g_strdup (hildon_touch_selector_get_current_text (HILDON_TOUCH_SELECTOR (selector)));
    }

  g_hash_table_unref (countries_to_counts);
  g_hash_table_unref (country_names);
  gtk_widget_destroy (GTK_WIDGET (dialog));

  if (result)
    {
      if (countries_with_many)
	{
	  /* these countries have many towers, so
	   * show the sub-areas
	   */
	  if (strcmp (result, COUNTRIES_WITH_FEW)==0)
	    towers_by_area_with_many (FALSE);
	  else
	    towers_by_subarea (result);
	}
      else
	{
	  country_and_county cac = { result, NULL };
	  gchar *title = g_strdup_printf ("Belltowers in %s",
					  result);

	  parse_dove (get_towers_by_county_cb,
		      &cac,
		      title);

	  g_free (title);
	}

      g_free (result);
    }
}

/**
 * Shows all the towers in areas with many towers.
 */
static void
towers_by_area (void)
{
  towers_by_area_with_many (TRUE);
}

static void
show_bookmarks (void)
{
  parse_dove (get_group_of_towers_cb,
	      CONFIG_BOOKMARK_GROUP,
	      "Bookmarks");
}

static void
tower_search (void)
{
  GtkWidget *terms = gtk_dialog_new_with_buttons ("What are you looking for?",
						  GTK_WINDOW (window),
						  GTK_DIALOG_MODAL|GTK_DIALOG_DESTROY_WITH_PARENT,
						  "Search",
						  GTK_RESPONSE_OK,
						  NULL);
  GtkWidget *entry = gtk_entry_new ();

  gtk_box_pack_end (GTK_BOX (GTK_DIALOG (terms)->vbox),
		    entry, TRUE, TRUE, 0);

  gtk_widget_show_all (GTK_WIDGET (terms));

  if (gtk_dialog_run (GTK_DIALOG (terms))==GTK_RESPONSE_OK)
    {
      parse_dove (get_towers_by_search_cb,
		  (char*) gtk_entry_get_text (GTK_ENTRY (entry)),
		  "Search results");
    }

  gtk_widget_destroy (GTK_WIDGET (terms));
}

static void
recent_towers (void)
{
  parse_dove (get_group_of_towers_cb,
	      CONFIG_RECENT_GROUP,
	      "Towers you have recently viewed");
}

/**
 * Displays a web page.
 * (Perhaps this should be merged with show_browser().)
 *
 * \param url  The URL.
 */
static void
show_web_page (GtkButton *dummy,
	       gpointer url)
{
  show_browser (url);
}

/**
 * Shows the credits.
 *
 * \param source If non-null, we were called from a button press,
 *               so always show the credits.  If null, we were called
 *               automatically on startup, so show the credits if
 *               they haven't already been seen.
 */
static void
show_credits (GtkButton *source,
	      gpointer dummy)
{
  gboolean from_button = (source!=NULL);
  GtkWidget *dialog, *label, *button;

  if (!from_button &&
      g_key_file_get_boolean (config,
			      CONFIG_GENERAL_GROUP,
			      CONFIG_SEEN_CREDITS_KEY,
			      NULL))
    {
      return;
    }
			      

  dialog = gtk_dialog_new_with_buttons ("Credits",
					GTK_WINDOW (window),
					GTK_DIALOG_MODAL|GTK_DIALOG_DESTROY_WITH_PARENT,
					NULL
					);

  button = hildon_button_new_with_text (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT,
					HILDON_BUTTON_ARRANGEMENT_VERTICAL,
					"Welcome to Belltower.  The program is \xc2\xa9 2009 Thomas Thurman.",
					"View the program's home page.");
  g_signal_connect (button, "clicked", G_CALLBACK (show_web_page),
		    "http://belltower.garage.maemo.org");
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox),
		    button,
		    TRUE, TRUE, 0);

  button = hildon_button_new_with_text (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT,
					HILDON_BUTTON_ARRANGEMENT_VERTICAL,
					"This program is provided under the GNU GPL, with no warranty.",
					"View the GNU General Public Licence.");
  g_signal_connect (button, "clicked", G_CALLBACK (show_web_page),
		    "http://www.gnu.org/copyleft/gpl.html");
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox),
		    button,
		    TRUE, TRUE, 0);

  button = hildon_button_new_with_text (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT,
					HILDON_BUTTON_ARRANGEMENT_VERTICAL,
					"The data comes from Dove's Guide for Church Bell Ringers.",
					"View Dove's Guide.");
  g_signal_connect (button, "clicked", G_CALLBACK (show_web_page),
		    "http://dove.cccbr.org.uk");
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox),
		    button,
		    TRUE, TRUE, 0);

  button = hildon_button_new_with_text (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT,
					HILDON_BUTTON_ARRANGEMENT_VERTICAL,
					"The belfry image is \xc2\xa9 Amanda Slater, cc-by-sa.",
					"View the original photograph.");
  g_signal_connect (button, "clicked", G_CALLBACK (show_web_page),
		    "http://www.flickr.com/photos/pikerslanefarm/3398769335/");
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox),
		    button,
		    TRUE, TRUE, 0);
  
  gtk_widget_show_all (GTK_WIDGET (dialog));

  g_key_file_set_boolean (config,
			  CONFIG_GENERAL_GROUP,
			  CONFIG_SEEN_CREDITS_KEY,
			  TRUE);
  save_config ();
}

int
main(int argc, char **argv)
{
  GtkWidget *bell, *button, *hbox;
  GdkPixbuf *bell_picture;

  gtk_init (&argc, &argv);
  g_set_application_name ("Belltower");

  device = g_object_new (LOCATION_TYPE_GPS_DEVICE, NULL);

  window = hildon_stackable_window_new ();
  gtk_window_set_title (GTK_WINDOW (window), "Belltower");
  g_signal_connect (G_OBJECT (window), "delete_event", G_CALLBACK (gtk_main_quit), NULL);

  bell_picture = gdk_pixbuf_new_from_file ("/usr/share/belltower/bells1.jpg", NULL);

  buttons = gtk_vbox_new (TRUE, 0);
  menu = HILDON_APP_MENU (hildon_app_menu_new ());

  add_button ("Nearby", nearby_towers);
  add_button ("Recent", recent_towers);
  add_button ("Bookmarks", show_bookmarks);
  add_button ("By area", towers_by_area);
  add_button ("Search", tower_search);

  /* extra buttons for the app menu */
  button = gtk_button_new_with_label ("Credits");
  g_signal_connect (button, "clicked", G_CALLBACK (show_credits), NULL);
  hildon_app_menu_append (menu, GTK_BUTTON (button));

  gtk_widget_show_all (GTK_WIDGET (menu));
  hildon_window_set_app_menu (HILDON_WINDOW (window), menu);

  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_end (GTK_BOX (hbox), buttons, TRUE, TRUE, 0);
  gtk_box_pack_end (GTK_BOX (hbox),
		    gtk_image_new_from_pixbuf (bell_picture),
		    TRUE, TRUE, 0);

  gtk_container_add (GTK_CONTAINER (window), hbox);
  gtk_widget_show_all (GTK_WIDGET (window));

  load_config ();
  show_credits (NULL, NULL);

  gtk_main ();

  return EXIT_SUCCESS;
}
