#    This file is part of battery-eye.
#
#    battery-eye is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    battery-eye is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with battery-eye.  If not, see <http://www.gnu.org/licenses/>.

#    Copyright 2010 Jussi Holm


import sqlite3

# profiling
import time

class DataStorage(object):
    def __init__(self, path, dataSources):
        self.path = path
        self.con = sqlite3.connect(self.path)
        self.createTables()
        self.dataSources = dataSources
        self.storedTypes = {}
        for ds in self.dataSources:
            for k,v in ds.getProvidedValues().items():
                self.storedTypes[k] = v
        self.checkAndAddTypes(self.storedTypes)

    def checkAndAddTypes(self, types):
        cursor = self.con.cursor()
        for type, unit in types.items():
            try:
                cursor.execute('insert into observation_type (name, unit) values (?, ?)',
                               (type, unit))
            except sqlite3.IntegrityError:
                # It already exists
                pass
        self.con.commit()

    def addObservation(self, type, value, unixtime):
        cursor = self.con.cursor()
        cursor.execute('insert into observation (type, value, timestamp) values ((select id from observation_type where name = ?), ?, ?)',
                       (type, value, int(unixtime)))
        self.con.commit()

    def __getObservations(self, cursor, type, start, end):
        #t1 = time.time()
        if isinstance(type, (tuple, list)):
            cursor.execute('''select timestamp, value, observation_type.name
                              from observation
                              left join observation_type on (observation_type.id = observation.type)
                              where type in
                                    (select id from observation_type where name in (%s))
                              and timestamp between ? and ?
                              order by timestamp asc''' % ','.join(['?']*len(type)),
                           tuple(type) + (start, end))
        else:
            cursor.execute('''select timestamp, value, observation_type.name
                              from observation
                              left join observation_type on (observation_type.id = observation.type)
                              where type =
                                    (select id from observation_type where name = ?)
                              and timestamp between ? and ?
                              order by timestamp asc''',
                              (type, start, end))
        #print "Query: %.3fs" % (time.time() - t1)

    def getObservations(self, type, start, end):
        cursor = self.con.cursor()
        self.__getObservations(cursor, type, start, end)
        def intOrNone(input):
            try:
                return int(input)
            except TypeError:
                return None
        return map(lambda row: (row[0], intOrNone(row[1]), row[2]), cursor.fetchall())

    def getObservationsWithBreaks(self, type, start, end):
        cursor = self.con.cursor()
        ret = []
        self.__getObservations(cursor, 'internal.graph.break', start, end)
        breaks = cursor.fetchall()
        breaks.append((end+1, None, 'internal.graph.break'))
        intervalStart = start
        for b in breaks:
            self.__getObservations(cursor, type, intervalStart+1, b[0]-1)
            ret.append(map(lambda row: (row[0], int(row[1]), row[2]), cursor.fetchall()))
            intervalStart = b[0]
        return ret

    def getPreviousObservationTime(self, timestamp):
        cursor = self.con.cursor()
        cursor.execute('select max(timestamp) from observation where timestamp < ?', (timestamp, ))
        try:
            return cursor.fetchone()[0]
        except:
            return None

    def countObservations(self):
        cursor = self.con.cursor()
        cursor.execute('select count(id) from observation')
        return cursor.fetchone()[0]

    def deleteObservations(self, olderThan):
        cursor = self.con.cursor()
        cursor.execute('delete from observation where timestamp < ?',
                       (olderThan, ))
        self.con.commit()
        cursor.execute('vacuum')

    def createTables(self):
        tables = ['''
            config (
                id integer primary key autoincrement,
                name text not null,
                value text
            )''','''
            observation (
                id integer primary key autoincrement,
                type integer,
                value text,
                timestamp integer
            )''','''
            observation_type (
                id integer primary key autoincrement,
                name text,
                unit text,

                unique (name)
            )''']
        indexes = ['timestamp_index on observation (timestamp)']
        cursor = self.con.cursor()
        for table in tables:
            cursor.execute('create table if not exists %s' % table)
        for index in indexes:
            cursor.execute('create index if not exists %s' % index)

