/*
babyphone - A baby monitor application on the Nokia N900.
    Copyright (C) 2010  Roman Morawek <maemo@morawek.at>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "usernotifier.h"


// the DBUS phone call handling as implemented only works in Maemo 5 systems
#ifndef Q_WS_MAEMO_5
  #error "DBUS user notification is only supported for Maemo 5"
#endif

#include <QtDBus>
#include <QDebug>

#include <QMessage>
#include <QMessageService>
using namespace QtMobility;


// taken from telephony-maemo.c
#define CSD_CALL_STATUS_COMING          2
#define CSD_CALL_STATUS_MT_ALERTING     5

#define PHONE_PROFILE_SILENT            "silent"


/*!
  The constructor registers the phone DBus messages to the methods of this 
  class. Also, it switches the phone profile to silent and initialises the
  call timer (but does not start it).
*/
UserNotifier::UserNotifier(Settings *settings, QObject *parent) :
    QObject(parent), itsSettings(settings), itsCallPending(false)
{
    // register to receive incoming calls
    bool result = QDBusConnection::systemBus().connect("com.nokia.csd.Call",
                          "/com/nokia/csd/call", "com.nokia.csd.Call", "Coming",
                          this, SLOT(receiveCall(const QDBusMessage&)));
    if (result == false)
        qWarning() << tr("Cannot connect to incoming calls: ") << QDBusConnection::systemBus().lastError();


    // perform profile switching
    if (itsSettings->itsSwitchProfile) {
        // determine current phone profile
        QDBusMessage msg = QDBusMessage::createMethodCall(
                "com.nokia.profiled", // --dest
                "/com/nokia/profiled", // destination object path
                "com.nokia.profiled", // message name (w/o method)
                "get_profile" // method
            );
        QDBusMessage reply = QDBusConnection::sessionBus().call(msg);

        if (reply.type() != QDBusMessage::ErrorMessage) {
            itsInitialProfile = reply.arguments()[0].toString();
            qDebug() << tr("detected initial profile") << itsInitialProfile;

            // switch to silent profile
            if (itsInitialProfile != PHONE_PROFILE_SILENT) {
                QDBusMessage msg = QDBusMessage::createMethodCall(
                        "com.nokia.profiled", // --dest
                        "/com/nokia/profiled", // destination object path
                        "com.nokia.profiled", // message name (w/o method)
                        "set_profile" // method
                    );
                msg << PHONE_PROFILE_SILENT;
                QDBusMessage reply = QDBusConnection::sessionBus().call(msg);
                if (reply.type() == QDBusMessage::ErrorMessage) {
                    qWarning() << tr("Switching current phone profile failed:") << QDBusConnection::sessionBus().lastError();

                    // we did not switch profile, therefore we also do not need to switch it back
                    // this is achieved by faking the initial profile to silent
                    itsInitialProfile = PHONE_PROFILE_SILENT;
                }
            }
        }
        else
            qWarning() << tr("Determining current phone profile failed:") << QDBusConnection::sessionBus().lastError();
    }
    else {
        // to disable profile switch fake the initial profile to silent
        itsInitialProfile = PHONE_PROFILE_SILENT;
    }


    // setup call timer
    itsCallTimer = new QTimer(this);
    itsCallTimer->setSingleShot(true);
    connect(itsCallTimer, SIGNAL(timeout()), this, SLOT(callSetupTimer()));
}


/*!
  The destructor restores the initially set phone profile, if applicable.
*/
UserNotifier::~UserNotifier()
{
    // switch back to initial profile
    if (itsInitialProfile != PHONE_PROFILE_SILENT) {
        QDBusMessage msg = QDBusMessage::createMethodCall(
                "com.nokia.profiled", // --dest
                "/com/nokia/profiled", // destination object path
                "com.nokia.profiled", // message name (w/o method)
                "set_profile" // method
            );
        msg << itsInitialProfile;
        QDBusMessage reply = QDBusConnection::sessionBus().call(msg);
        if (reply.type() == QDBusMessage::ErrorMessage) {
            qWarning() << tr("Switching current phone profile failed:") << QDBusConnection::sessionBus().lastError();
        }
    }
}


/*!
  Notify initiates the phone call to the parent's phone number using the proper
  DBus message. Afterwards it starts the call timeout.
*/
bool UserNotifier::Notify()
{
    // if a call is already pending, do not start a second one
    if (itsCallPending) {
        qWarning() << tr("Call already pending. Denying call request.");
        return false;
    }

    // get DBUS system bus
    if (!QDBusConnection::systemBus().isConnected()) {
        qCritical() << tr("Cannot connect to DBUS system bus.");
        return false;
    }

    // initiate call
    QDBusMessage msg = QDBusMessage::createMethodCall(
            "com.nokia.csd.Call", // --dest
            "/com/nokia/csd/call", // destination object path
            "com.nokia.csd.Call", // message name (w/o method)
            "CreateWith" // method
        );
    msg << itsSettings->itsPhonenumber;
    msg << 0;
    QDBusMessage reply = QDBusConnection::systemBus().call(msg);
    if (reply.type() == QDBusMessage::ErrorMessage) {
        qCritical() << tr("Call initiation failed: ") << QDBusConnection::systemBus().lastError();
        return false;
    }
    itsCallPending = true;
    qDebug() << tr("Call successfully initiated to") << itsSettings->itsPhonenumber;

    // register to receive call establishment notification
    bool result = QDBusConnection::systemBus().connect("com.nokia.csd.Call",
                     "/com/nokia/csd/call/1", "com.nokia.csd.Call.Instance",
                     "AudioConnect", this, SLOT(callEstablished(const QDBusMessage&)));
    if (result == false)
        qCritical() << tr("Cannot connect to call establishment notifications: ") << QDBusConnection::systemBus().lastError();

    // start timer to abort call if not answered
    itsCallTimer->start(itsSettings->itsCallSetupTimer*1000);

    return true;
}


/*!
  callEstablished get called as a phone call start or stop event is received.
  If a call is started it extends the call timeout. If it is finished it signals
  the end of notification.
*/
void UserNotifier::callEstablished(const QDBusMessage &msg)
{
    bool flag0 = msg.arguments()[0].toBool();
    bool flag1 = msg.arguments()[1].toBool();

    // is this the start or end of the call?
    if (flag0 && flag1) {
        // start of call
        // restart the call timer with a longer timeout, once we got an active call
        qDebug() << tr("Call established, extending call timeout.");
        itsCallTimer->start(itsSettings->CALL_HOLD_TIMER);
    }
    else if (!flag0 && !flag1) {
        // end of call
        // clear potential running timer
        itsCallTimer->stop();

        // deregister call establishment notification
        QDBusConnection::systemBus().disconnect("com.nokia.csd.Call",
                         "/com/nokia/csd/call/1", "com.nokia.csd.Call.Instance",
                         "AudioConnect", this, SLOT(callEstablished(const QDBusMessage&)));

        // signal the end of the notification process
        qDebug() << tr("Call terminated, signal end of notification.");
        itsCallPending = false;
        emit notifyFinished();
    }
}


/*!
  callSetupTimer gets called on the call timeout and drops the current phone 
  call. Afterwards it signals the end of the notification.
*/
void UserNotifier::callSetupTimer()
{
    // terminate call after this timeout
    qDebug() << tr("Call setup timeout triggered. Releasing call.");
    dropIncomingCall();

    // deregister call establishment notification
    QDBusConnection::systemBus().disconnect("com.nokia.csd.Call",
                     "/com/nokia/csd/call/1", "com.nokia.csd.Call.Instance",
                     "AudioConnect", this, SLOT(callEstablished(const QDBusMessage&)));

    // signal the end of the notification process
    itsCallPending = false;
    emit notifyFinished();
}


/*!
  callSetupTimer gets called on an incoming phone call and handles this.
  The call handling equals a drop of the call if configured in the settings.
  Otherwise the event is ignored.
*/
void UserNotifier::receiveCall(const QDBusMessage &msg)
{
    QList<QVariant> lst = msg.arguments();
    QString caller = lst[1].toString();
    qDebug() << tr("Receive call from ") << caller;

    if (itsSettings->itsHandleIncomingCalls) {
        // handle incoming calls
        // take the call if it is from the parent's phone and no call is pending
        if ( (caller == itsSettings->itsPhonenumber) && !itsCallPending) {
            // to answer a call successfully we need the phone to be in a proper call state
            // register to receive call status notification
            bool result = QDBusConnection::systemBus().connect("com.nokia.csd.Call",
                             "/com/nokia/csd/call/1", "com.nokia.csd.Call.Instance",
                             "CallStatus", this, SLOT(callStatusUpdate(const QDBusMessage&)));
            if (result == false)
                qWarning() << tr("Cannot connect to call establishment notifications: ") << QDBusConnection::systemBus().lastError();
        }
        else {
            // drop the call
            if (dropIncomingCall()) {
                // notify parents on this event
                if (itsSettings->itsSendSMS)
                    notifySMS(caller);
            }
        }
    }
    else {
        // call handling is inactive
        // nevertheless, we may send an SMS
        if (itsSettings->itsSendSMS)
            notifySMS(caller);
    }
}


/*!
  callStatusUpdate gets called on an incoming phone call from the parents.
  It checks the call status event and takes the call as it is ready.
*/
void UserNotifier::callStatusUpdate(const QDBusMessage &msg)
{
    int callStatus = msg.arguments()[0].toInt();

    if (callStatus >= CSD_CALL_STATUS_COMING) {
        // now we are ready to take the call
        // disconnect DBUS notifications first
        QDBusConnection::systemBus().disconnect("com.nokia.csd.Call",
                  "/com/nokia/csd/call/1", "com.nokia.csd.Call.Instance",
                  "CallStatus", this, SLOT(callStatusUpdate(const QDBusMessage&)));

        // take the call
        takeIncomingCall();
    }
}


/*!
  dropIncomingCall drops an incoming call using the DBus.
*/
bool UserNotifier::dropIncomingCall() const
{
    // setup proper DBUS message
    QDBusMessage msg = QDBusMessage::createMethodCall(
            "com.nokia.csd.Call",       // --dest
            "/com/nokia/csd/call",      // destination object path
            "com.nokia.csd.Call",       // message name (w/o method)
            "Release"                   // method
        );
    QDBusMessage reply = QDBusConnection::systemBus().call(msg);

    if (reply.type() == QDBusMessage::ErrorMessage) {
        qWarning() << tr("Call handling failed: ") << QDBusConnection::systemBus().lastError();
        return false;
    }
    qDebug() << tr("Call dropped");

    return true;
}


/*!
  takeIncomingCall answers an incoming call using the DBus.
*/
bool UserNotifier::takeIncomingCall()
{
    QDBusMessage msg = QDBusMessage::createMethodCall(
            "com.nokia.csd.Call",       // --dest
            "/com/nokia/csd/call/1",    // destination object path
            "com.nokia.csd.Call.Instance",  // message name (w/o method)
            "Answer"                    // method
        );
    QDBusMessage reply = QDBusConnection::systemBus().call(msg);

    if (reply.type() == QDBusMessage::ErrorMessage) {
        qWarning() << tr("Call handling failed: ") << QDBusConnection::systemBus().lastError();
        return false;
    }
    itsCallPending = true;
    qDebug() << tr("Call taken");

    // register to receive call establishment notification
    bool result = QDBusConnection::systemBus().connect("com.nokia.csd.Call",
                     "/com/nokia/csd/call/1", "com.nokia.csd.Call.Instance",
                     "AudioConnect", this, SLOT(callEstablished(const QDBusMessage&)));
    if (result == false)
        qCritical() << tr("Cannot connect to call establishment notifications: ") << QDBusConnection::systemBus().lastError();

    // start safety timer for long phone calls
    itsCallTimer->start(itsSettings->CALL_HOLD_TIMER*1000);

    return true;
}


/*!
  notifySMS sends an SMS message to the parent's phone to inform it on missed
  calls.
*/
bool UserNotifier::notifySMS(const QString droppedPhoneNumber) const
{
    // use Qt mobility messaging interface
    QMap<QString, QMessageAccountId> accountDetails;
    QMessageService theService;
    QMessageManager manager;

    // scan all accounts
    foreach(const QMessageAccountId &id, manager.queryAccounts()){
      QMessageAccount account(id);
      accountDetails.insert(account.name(), account.id());
    }

    // setup SMS account
    QMessage theMessage;
    theMessage.setType(QMessage::Sms);
    theMessage.setParentAccountId(accountDetails["SMS"]);
    theMessage.setTo(QMessageAddress(QMessageAddress::Phone, itsSettings->itsPhonenumber));

    // set SMS text
    QString smsText = tr("N900 babyphone: An incoming phone call was rejected:");
    smsText.append(droppedPhoneNumber);
    theMessage.setBody(smsText);

    // send it
    bool success = theService.send(theMessage);
    qDebug() << "send SMS" << (success ? "successfull" : "failed");

    return success;
}
