#!/usr/bin/env python2.5
try:
  from xml.etree import cElementTree as ElementTree
except ImportError:
  try:
    import cElementTree as ElementTree
  except ImportError:
    try:
      from xml.etree import ElementTree
    except ImportError:
      from elementtree import ElementTree
import os
import atom
import gdata
import gdata.service
import gdata.youtube

YOUTUBE_SERVER = 'gdata.youtube.com'
YOUTUBE_SERVICE = 'youtube'
YOUTUBE_CLIENTLOGIN_AUTHENTICATION_URL = 'https://www.google.com/youtube/accounts/ClientLogin'
YOUTUBE_SUPPORTED_UPLOAD_TYPES = ('mov', 'avi', 'wmv', 'mpg', 'quicktime',
                                  'flv', 'mp4', 'x-flv')
YOUTUBE_QUERY_VALID_TIME_PARAMETERS = ('today', 'this_week', 'this_month',
                                       'all_time')
YOUTUBE_QUERY_VALID_ORDERBY_PARAMETERS = ('published', 'viewCount', 'rating',
                                          'relevance')
YOUTUBE_QUERY_VALID_RACY_PARAMETERS = ('include', 'exclude')
YOUTUBE_QUERY_VALID_FORMAT_PARAMETERS = ('1', '5', '6')
YOUTUBE_STANDARDFEEDS = ('most_recent', 'recently_featured',
                         'top_rated', 'most_viewed','watch_on_mobile')
YOUTUBE_UPLOAD_URI = 'http://uploads.gdata.youtube.com/feeds/api/users'
YOUTUBE_UPLOAD_TOKEN_URI = 'http://gdata.youtube.com/action/GetUploadToken'
YOUTUBE_VIDEO_URI = 'http://gdata.youtube.com/feeds/api/videos'
YOUTUBE_USER_FEED_URI = 'http://gdata.youtube.com/feeds/api/users'
YOUTUBE_PLAYLIST_FEED_URI = 'http://gdata.youtube.com/feeds/api/playlists'

YOUTUBE_SCHEMA = 'http://gdata.youtube.com/schemas/2007'

YOUTUBE_RATING_LINK_REL = '%s#video.ratings' % YOUTUBE_SCHEMA

YOUTUBE_COMPLAINT_CATEGORY_SCHEME = '%s/%s' % (YOUTUBE_SCHEMA,
                                               'complaint-reasons.cat')
YOUTUBE_SUBSCRIPTION_CATEGORY_SCHEME = '%s/%s' % (YOUTUBE_SCHEMA,
                                                  'subscriptiontypes.cat')

YOUTUBE_COMPLAINT_CATEGORY_TERMS = ('PORN', 'VIOLENCE', 'HATE', 'DANGEROUS',
                                    'RIGHTS', 'SPAM')
YOUTUBE_CONTACT_STATUS = ('accepted', 'rejected')
YOUTUBE_CONTACT_CATEGORY = ('Friends', 'Family')

UNKOWN_ERROR = 1000
YOUTUBE_BAD_REQUEST = 400
YOUTUBE_CONFLICT = 409
YOUTUBE_INTERNAL_SERVER_ERROR = 500
YOUTUBE_INVALID_ARGUMENT = 601
YOUTUBE_INVALID_CONTENT_TYPE = 602
YOUTUBE_NOT_A_VIDEO = 603
YOUTUBE_INVALID_KIND = 604

class Error(Exception):
  pass

class RequestError(Error):
  pass

class YouTubeError(Error):
  pass

class YouTubeService(gdata.service.GDataService):
  def __init__(self, email=None, password=None, source=None,
               server=YOUTUBE_SERVER, additional_headers=None, client_id=None,
               developer_key=None, **kwargs):
    
    gdata.service.GDataService.__init__(
        self, email=email, password=password, service=YOUTUBE_SERVICE,
        source=source, server=server, additional_headers=additional_headers,
        **kwargs)

    if client_id is not None:
      self.additional_headers['X-Gdata-Client'] = client_id

    if developer_key is not None:
      self.additional_headers['X-GData-Key'] = 'key=%s' % developer_key

    self.auth_service_url = YOUTUBE_CLIENTLOGIN_AUTHENTICATION_URL

  def GetYouTubeVideoFeed(self, uri):
    return self.Get(uri, converter=gdata.youtube.YouTubeVideoFeedFromString)

  def GetYouTubeVideoEntry(self, uri=None, video_id=None, api_version=2):
    if uri is None and video_id is None:
      raise YouTubeError('You must provide at least a uri or a video_id '
                         'to the GetYouTubeVideoEntry() method')
    elif video_id and not uri:
      uri = '%s/%s?v=%d' % (YOUTUBE_VIDEO_URI, video_id, api_version)
    return self.Get(uri, converter=gdata.youtube.YouTubeVideoEntryFromString)

  def GetYouTubeContactFeed(self, uri=None, username='default'):
    if uri is None:
      uri = '%s/%s/%s' % (YOUTUBE_USER_FEED_URI, username, 'contacts')
    return self.Get(uri, converter=gdata.youtube.YouTubeContactFeedFromString)

  def GetYouTubeContactEntry(self, uri):
    return self.Get(uri, converter=gdata.youtube.YouTubeContactEntryFromString)

  def GetYouTubeVideoCommentFeed(self, uri=None, video_id=None):
    if uri is None and video_id is None:
      raise YouTubeError('You must provide at least a uri or a video_id '
                         'to the GetYouTubeVideoCommentFeed() method')
    elif video_id and not uri:
      uri = '%s/%s/%s' % (YOUTUBE_VIDEO_URI, video_id, 'comments')
    return self.Get(
        uri, converter=gdata.youtube.YouTubeVideoCommentFeedFromString)

  def GetYouTubeVideoCommentEntry(self, uri):
    return self.Get(
        uri, converter=gdata.youtube.YouTubeVideoCommentEntryFromString)

  def GetYouTubeUserFeed(self, uri=None, username=None):
    if uri is None and username is None:
      raise YouTubeError('You must provide at least a uri or a username '
                         'to the GetYouTubeUserFeed() method')
    elif username and not uri:
      uri = '%s/%s/%s' % (YOUTUBE_USER_FEED_URI, username, 'uploads')
    return self.Get(uri, converter=gdata.youtube.YouTubeUserFeedFromString)

  def GetYouTubeUserEntry(self, uri=None, username=None):
    if uri is None and username is None:
      raise YouTubeError('You must provide at least a uri or a username '
                         'to the GetYouTubeUserEntry() method')
    elif username and not uri:
      uri = '%s/%s' % (YOUTUBE_USER_FEED_URI, username)
    return self.Get(uri, converter=gdata.youtube.YouTubeUserEntryFromString)

  def GetYouTubePlaylistFeed(self, uri=None, username='default'):
    if uri is None:
      uri = '%s/%s/%s' % (YOUTUBE_USER_FEED_URI, username, 'playlists')
    return self.Get(uri, converter=gdata.youtube.YouTubePlaylistFeedFromString)

  def GetYouTubePlaylistEntry(self, uri):
    return self.Get(uri, converter=gdata.youtube.YouTubePlaylistEntryFromString)

  def GetYouTubePlaylistVideoFeed(self, uri=None, playlist_id=None):
    if uri is None and playlist_id is None:
      raise YouTubeError('You must provide at least a uri or a playlist_id '
                         'to the GetYouTubePlaylistVideoFeed() method')
    elif playlist_id and not uri:
      uri = '%s/%s' % (YOUTUBE_PLAYLIST_FEED_URI, playlist_id)
    return self.Get(
        uri, converter=gdata.youtube.YouTubePlaylistVideoFeedFromString)

  def GetYouTubeVideoResponseFeed(self, uri=None, video_id=None):
    if uri is None and video_id is None:
      raise YouTubeError('You must provide at least a uri or a video_id '
                         'to the GetYouTubeVideoResponseFeed() method')
    elif video_id and not uri:
      uri = '%s/%s/%s' % (YOUTUBE_VIDEO_URI, video_id, 'responses')
    return self.Get(
        uri, converter=gdata.youtube.YouTubeVideoResponseFeedFromString)

  def GetYouTubeVideoResponseEntry(self, uri):
    return self.Get(
        uri, converter=gdata.youtube.YouTubeVideoResponseEntryFromString)

  def GetYouTubeSubscriptionFeed(self, uri=None, username='default'):
    if uri is None:
      uri = '%s/%s/%s' % (YOUTUBE_USER_FEED_URI, username, 'subscriptions')
    return self.Get(
        uri, converter=gdata.youtube.YouTubeSubscriptionFeedFromString)

  def GetYouTubeSubscriptionEntry(self, uri):
    return self.Get(
        uri, converter=gdata.youtube.YouTubeSubscriptionEntryFromString)

  def GetYouTubeRelatedVideoFeed(self, uri=None, video_id=None):
    if uri is None and video_id is None:
      raise YouTubeError('You must provide at least a uri or a video_id '
                         'to the GetYouTubeRelatedVideoFeed() method')
    elif video_id and not uri:
      uri = '%s/%s/%s' % (YOUTUBE_VIDEO_URI, video_id, 'related')
    return self.Get(
        uri, converter=gdata.youtube.YouTubeVideoFeedFromString)

  def GetUserFavoritesFeed(self, username='default'):
    favorites_feed_uri = '%s/%s/%s' % (YOUTUBE_USER_FEED_URI, username,
                                       'favorites')
    return self.GetYouTubeVideoFeed(favorites_feed_uri)

  def InsertVideoEntry(self, video_entry, filename_or_handle,
                       youtube_username='default',
                       content_type='video/quicktime'):

    try:
      assert(isinstance(video_entry, gdata.youtube.YouTubeVideoEntry))
    except AssertionError:
      raise YouTubeError({'status':YOUTUBE_INVALID_ARGUMENT,
          'body':'`video_entry` must be a gdata.youtube.VideoEntry instance',
          'reason':'Found %s, not VideoEntry' % type(video_entry)
          })
    if (isinstance(filename_or_handle, (str, unicode)) 
        and os.path.exists(filename_or_handle)):
      mediasource = gdata.MediaSource()
      mediasource.setFile(filename_or_handle, content_type)
    elif hasattr(filename_or_handle, 'read'):
      import StringIO
      if hasattr(filename_or_handle, 'seek'):
        filename_or_handle.seek(0)
      file_handle = filename_or_handle
      name = 'video'
      if hasattr(filename_or_handle, 'name'):
        name = filename_or_handle.name
      mediasource = gdata.MediaSource(file_handle, content_type,
          content_length=file_handle.len, file_name=name)
    else:
      raise YouTubeError({'status':YOUTUBE_INVALID_ARGUMENT, 'body':
          '`filename_or_handle` must be a path name or a file-like object',
          'reason': ('Found %s, not path name or object '
                     'with a .read() method' % type(filename_or_handle))})
    upload_uri = '%s/%s/%s' % (YOUTUBE_UPLOAD_URI, youtube_username,
                              'uploads')
    self.additional_headers['Slug'] = mediasource.file_name
    try:
      try:
        return self.Post(video_entry, uri=upload_uri, media_source=mediasource,
                         converter=gdata.youtube.YouTubeVideoEntryFromString)
      except gdata.service.RequestError, e:
        raise YouTubeError(e.args[0])
    finally:
      del(self.additional_headers['Slug'])

  def CheckUploadStatus(self, video_entry=None, video_id=None):
    if video_entry is None and video_id is None:
      raise YouTubeError('You must provide at least a uri or a video_id '
                         'to the CheckUploadStatus() method')
    elif video_id and not video_entry:
       video_entry = self.GetYouTubeVideoEntry(video_id=video_id)

    control = video_entry.control
    if control is not None:
      draft = control.draft
      if draft is not None:
        if draft.text == 'yes':
          yt_state = control.extension_elements[0]
          if yt_state is not None:
            state_value = yt_state.attributes['name']
            message = ''
            if yt_state.text is not None:
              message = yt_state.text

            return (state_value, message)

  def GetFormUploadToken(self, video_entry, uri=YOUTUBE_UPLOAD_TOKEN_URI):
    try:
      response = self.Post(video_entry, uri)
    except gdata.service.RequestError, e:
      raise YouTubeError(e.args[0])

    tree = ElementTree.fromstring(response)

    for child in tree:
      if child.tag == 'url':
        post_url = child.text
      elif child.tag == 'token':
        youtube_token = child.text
    return (post_url, youtube_token)

  def UpdateVideoEntry(self, video_entry):
    for link in video_entry.link:
      if link.rel == 'edit':
        edit_uri = link.href
    return self.Put(video_entry, uri=edit_uri,
                    converter=gdata.youtube.YouTubeVideoEntryFromString)

  def DeleteVideoEntry(self, video_entry):
    for link in video_entry.link:
      if link.rel == 'edit':
        edit_uri = link.href
    return self.Delete(edit_uri)
    
  def AddLikeOrDislike(self, like_or_dislike, video_entry):
    if like_or_dislike not in ('like', 'dislike'):
      raise YouTubeError('value must be "like" or "dislike" in AddLikeOrDislike()')

    entry = gdata.GDataEntry()
    rating = gdata.youtube.LikesAndDislikes()
    rating.extension_attributes['name'] = 'value'
    rating.extension_attributes['value'] = like_or_dislike
    entry.extension_elements.append(rating)

    for link in video_entry.link:
      if link.rel == YOUTUBE_RATING_LINK_REL:
        rating_uri = link.href

    return self.Post(entry, uri=rating_uri)

  def AddRating(self, rating_value, video_entry):
    if rating_value < 1 or rating_value > 5:
      raise YouTubeError('rating_value must be between 1 and 5 in AddRating()')

    entry = gdata.GDataEntry()
    rating = gdata.youtube.Rating(min='1', max='5')
    rating.extension_attributes['name'] = 'value'
    rating.extension_attributes['value'] = str(rating_value)
    entry.extension_elements.append(rating)

    for link in video_entry.link:
      if link.rel == YOUTUBE_RATING_LINK_REL:
        rating_uri = link.href

    return self.Post(entry, uri=rating_uri)

  def AddComment(self, comment_text, video_entry):
    content = atom.Content(text=comment_text)
    comment_entry = gdata.youtube.YouTubeVideoCommentEntry(content=content)
    comment_post_uri = video_entry.comments.feed_link[0].href

    return self.Post(comment_entry, uri=comment_post_uri)

  def AddVideoResponse(self, video_id_to_respond_to, video_response):
    post_uri = '%s/%s/%s' % (YOUTUBE_VIDEO_URI, video_id_to_respond_to,
                             'responses')
    return self.Post(video_response, uri=post_uri)

  def DeleteVideoResponse(self, video_id, response_video_id):
    delete_uri = '%s/%s/%s/%s' % (YOUTUBE_VIDEO_URI, video_id, 'responses',
                                  response_video_id)
    return self.Delete(delete_uri)

  def AddComplaint(self, complaint_text, complaint_term, video_id):
    if complaint_term not in YOUTUBE_COMPLAINT_CATEGORY_TERMS:
      raise YouTubeError('Your complaint_term is not valid')

    content = atom.Content(text=complaint_text)
    category = atom.Category(term=complaint_term,
                             scheme=YOUTUBE_COMPLAINT_CATEGORY_SCHEME)

    complaint_entry = gdata.GDataEntry(content=content, category=[category])
    post_uri = '%s/%s/%s' % (YOUTUBE_VIDEO_URI, video_id, 'complaints')

    return self.Post(complaint_entry, post_uri)

  def AddVideoEntryToFavorites(self, video_entry, username='default'):
    post_uri = '%s/%s/%s' % (YOUTUBE_USER_FEED_URI, username, 'favorites')

    return self.Post(video_entry, post_uri,
                     converter=gdata.youtube.YouTubeVideoEntryFromString)

  def DeleteVideoEntryFromFavorites(self, video_id, username='default'):
    edit_link = '%s/%s/%s/%s' % (YOUTUBE_USER_FEED_URI, username, 'favorites',
                                 video_id)
    return self.Delete(edit_link)

  def AddPlaylist(self, playlist_title, playlist_description,
                  playlist_private=None):
    playlist_entry = gdata.youtube.YouTubePlaylistEntry(
        title=atom.Title(text=playlist_title),
        description=gdata.youtube.Description(text=playlist_description))
    if playlist_private:
      playlist_entry.private = gdata.youtube.Private()

    playlist_post_uri = '%s/%s/%s' % (YOUTUBE_USER_FEED_URI, 'default', 
                                      'playlists')
    return self.Post(playlist_entry, playlist_post_uri,
                     converter=gdata.youtube.YouTubePlaylistEntryFromString)

  def UpdatePlaylist(self, playlist_id, new_playlist_title,
                     new_playlist_description, playlist_private=None,
                     username='default'):
    updated_playlist = gdata.youtube.YouTubePlaylistEntry(
        title=atom.Title(text=new_playlist_title),
        description=gdata.youtube.Description(text=new_playlist_description))
    if playlist_private:
      updated_playlist.private = gdata.youtube.Private()

    playlist_put_uri = '%s/%s/playlists/%s' % (YOUTUBE_USER_FEED_URI, username,
                                               playlist_id)

    return self.Put(updated_playlist, playlist_put_uri,
                    converter=gdata.youtube.YouTubePlaylistEntryFromString)

  def DeletePlaylist(self, playlist_uri):
    return self.Delete(playlist_uri)

  def AddPlaylistVideoEntryToPlaylist(
      self, playlist_uri, video_id, custom_video_title=None,
      custom_video_description=None):
    playlist_video_entry = gdata.youtube.YouTubePlaylistVideoEntry(
        atom_id=atom.Id(text=video_id))
    if custom_video_title:
      playlist_video_entry.title = atom.Title(text=custom_video_title)
    if custom_video_description:
      playlist_video_entry.description = gdata.youtube.Description(
          text=custom_video_description)

    return self.Post(playlist_video_entry, playlist_uri,
                    converter=gdata.youtube.YouTubePlaylistVideoEntryFromString)

  def UpdatePlaylistVideoEntryMetaData(
      self, playlist_uri, playlist_entry_id, new_video_title, 
      new_video_description, new_video_position):
    playlist_video_entry = gdata.youtube.YouTubePlaylistVideoEntry(
        title=atom.Title(text=new_video_title),
        description=gdata.youtube.Description(text=new_video_description),
        position=gdata.youtube.Position(text=str(new_video_position)))

    playlist_put_uri = playlist_uri + '/' + playlist_entry_id

    return self.Put(playlist_video_entry, playlist_put_uri,
                    converter=gdata.youtube.YouTubePlaylistVideoEntryFromString)

  def DeletePlaylistVideoEntry(self, playlist_uri, playlist_video_entry_id):
    delete_uri = '%s/%s' % (playlist_uri, playlist_video_entry_id)
    return self.Delete(delete_uri)

  def AddSubscriptionToChannel(self, username_to_subscribe_to,
                               my_username = 'default'):
    subscription_category = atom.Category(
        scheme=YOUTUBE_SUBSCRIPTION_CATEGORY_SCHEME,
        term='channel')
    subscription_username = gdata.youtube.Username(
        text=username_to_subscribe_to)

    subscription_entry = gdata.youtube.YouTubeSubscriptionEntry(
        category=subscription_category,
        username=subscription_username)

    post_uri = '%s/%s/%s' % (YOUTUBE_USER_FEED_URI, my_username, 
                             'subscriptions')

    return self.Post(subscription_entry, post_uri,
                     converter=gdata.youtube.YouTubeSubscriptionEntryFromString)

  def AddSubscriptionToFavorites(self, username, my_username = 'default'):
    subscription_category = atom.Category(
        scheme=YOUTUBE_SUBSCRIPTION_CATEGORY_SCHEME,
        term='favorites')
    subscription_username = gdata.youtube.Username(text=username)

    subscription_entry = gdata.youtube.YouTubeSubscriptionEntry(
        category=subscription_category,
        username=subscription_username)

    post_uri = '%s/%s/%s' % (YOUTUBE_USER_FEED_URI, my_username,
                             'subscriptions')

    return self.Post(subscription_entry, post_uri,
                     converter=gdata.youtube.YouTubeSubscriptionEntryFromString)

  def AddSubscriptionToQuery(self, query, my_username = 'default'):
    subscription_category = atom.Category(
        scheme=YOUTUBE_SUBSCRIPTION_CATEGORY_SCHEME,
        term='query')
    subscription_query_string = gdata.youtube.QueryString(text=query)

    subscription_entry = gdata.youtube.YouTubeSubscriptionEntry(
        category=subscription_category,
        query_string=subscription_query_string)

    post_uri = '%s/%s/%s' % (YOUTUBE_USER_FEED_URI, my_username,
                             'subscriptions')

    return self.Post(subscription_entry, post_uri,
                     converter=gdata.youtube.YouTubeSubscriptionEntryFromString)



  def DeleteSubscription(self, subscription_uri):
    return self.Delete(subscription_uri)

  def AddContact(self, contact_username, my_username='default'):
    contact_category = atom.Category(
        scheme = 'http://gdata.youtube.com/schemas/2007/contact.cat',
        term = 'Friends')
    contact_username = gdata.youtube.Username(text=contact_username)
    contact_entry = gdata.youtube.YouTubeContactEntry(
        category=contact_category,
        username=contact_username)

    contact_post_uri = '%s/%s/%s' % (YOUTUBE_USER_FEED_URI, my_username,
                                     'contacts')

    return self.Post(contact_entry, contact_post_uri,
                     converter=gdata.youtube.YouTubeContactEntryFromString)

  def UpdateContact(self, contact_username, new_contact_status, 
                    new_contact_category, my_username='default'):
    if new_contact_status not in YOUTUBE_CONTACT_STATUS:
      raise YouTubeError('New contact status must be one of %s' %
                          (' '.join(YOUTUBE_CONTACT_STATUS)))
    if new_contact_category not in YOUTUBE_CONTACT_CATEGORY:
      raise YouTubeError('New contact category must be one of %s' %
                         (' '.join(YOUTUBE_CONTACT_CATEGORY)))

    contact_category = atom.Category(
        scheme='http://gdata.youtube.com/schemas/2007/contact.cat',
        term=new_contact_category)

    contact_status = gdata.youtube.Status(text=new_contact_status)
    contact_entry = gdata.youtube.YouTubeContactEntry(
        category=contact_category,
        status=contact_status)

    contact_put_uri = '%s/%s/%s/%s' % (YOUTUBE_USER_FEED_URI, my_username,
                                       'contacts', contact_username)

    return self.Put(contact_entry, contact_put_uri,
                    converter=gdata.youtube.YouTubeContactEntryFromString)

  def DeleteContact(self, contact_username, my_username='default'):
    contact_edit_uri = '%s/%s/%s/%s' % (YOUTUBE_USER_FEED_URI, my_username,
                                        'contacts', contact_username)
    return self.Delete(contact_edit_uri)

  def _GetDeveloperKey(self):
    if 'X-GData-Key' in self.additional_headers:
      return self.additional_headers['X-GData-Key'][4:]
    else:
      return None

  def _SetDeveloperKey(self, developer_key):
    self.additional_headers['X-GData-Key'] = 'key=' + developer_key

  developer_key = property(_GetDeveloperKey, _SetDeveloperKey,
                           doc="""The Developer Key property""")

  def _GetClientId(self):
    if 'X-Gdata-Client' in self.additional_headers:
      return self.additional_headers['X-Gdata-Client']
    else:
      return None

  def _SetClientId(self, client_id):
    self.additional_headers['X-Gdata-Client'] = client_id

  client_id = property(_GetClientId, _SetClientId,
                       doc="""The ClientId property""")

  def Query(self, uri):
    result = self.Get(uri)
    return result

  def YouTubeQuery(self, query):
    result = self.Query(query.ToUri())
    if isinstance(query, YouTubeUserQuery):
      return gdata.youtube.YouTubeUserFeedFromString(result.ToString())
    elif isinstance(query, YouTubePlaylistQuery):
      return gdata.youtube.YouTubePlaylistFeedFromString(result.ToString())
    elif isinstance(query, YouTubeVideoQuery):
      return gdata.youtube.YouTubeVideoFeedFromString(result.ToString())
    else:
      return result

class YouTubeVideoQuery(gdata.service.Query):

  def __init__(self, video_id=None, feed_type=None, text_query=None,
               params=None, categories=None, feed=None):

    if feed_type in YOUTUBE_STANDARDFEEDS and feed is None:
      feed = 'http://%s/feeds/standardfeeds/%s' % (YOUTUBE_SERVER, feed_type)
    elif (feed_type is 'responses' or feed_type is 'comments' and video_id
          and feed is None):
      feed = 'http://%s/feeds/videos/%s/%s' % (YOUTUBE_SERVER, video_id,
                                               feed_type)
    elif feed is None:
      feed = 'http://%s/feeds/videos' % (YOUTUBE_SERVER)

    gdata.service.Query.__init__(self, feed, text_query=text_query,
                                 params=params, categories=categories)
 
  def _GetVideoQuery(self):
    if 'q' in self:
      return self['q']
    else:
      return None

  def _SetVideoQuery(self, val):
    self['q'] = val

  q = property(_GetVideoQuery, _SetVideoQuery,
                doc="""The video query (q) query parameter""")

  def _GetOrderBy(self):
    if 'orderby' in self:
      return self['orderby']
    else:
      return None

  def _SetOrderBy(self, val):
    if val not in YOUTUBE_QUERY_VALID_ORDERBY_PARAMETERS:
      if val.startswith('relevance_lang_') is False:
        raise YouTubeError('OrderBy must be one of: %s ' %
                           ' '.join(YOUTUBE_QUERY_VALID_ORDERBY_PARAMETERS))
    self['orderby'] = val

  orderby = property(_GetOrderBy, _SetOrderBy,
                     doc="""The orderby query parameter""")

  def _GetTime(self):
    if 'time' in self:
      return self['time']
    else:
      return None

  def _SetTime(self, val):
    if val not in YOUTUBE_QUERY_VALID_TIME_PARAMETERS:
      raise YouTubeError('Time must be one of: %s ' %
                         ' '.join(YOUTUBE_QUERY_VALID_TIME_PARAMETERS))
    self['time'] = val

  time = property(_GetTime, _SetTime,
                  doc="""The time query parameter""")

  def _GetFormat(self):
    if 'format' in self:
      return self['format']
    else:
      return None

  def _SetFormat(self, val):
    if val not in YOUTUBE_QUERY_VALID_FORMAT_PARAMETERS:
      raise YouTubeError('Format must be one of: %s ' %
                         ' '.join(YOUTUBE_QUERY_VALID_FORMAT_PARAMETERS))
    self['format'] = val

  format = property(_GetFormat, _SetFormat,
                    doc="""The format query parameter""")

  def _GetRacy(self):
    if 'racy' in self:
      return self['racy']
    else:
      return None

  def _SetRacy(self, val):
    if val not in YOUTUBE_QUERY_VALID_RACY_PARAMETERS:
      raise YouTubeError('Racy must be one of: %s ' %
                         ' '.join(YOUTUBE_QUERY_VALID_RACY_PARAMETERS))
    self['racy'] = val

  racy = property(_GetRacy, _SetRacy, 
                  doc="""The racy query parameter""")

  def _GetLanguageRestriction(self):
    if 'lr' in self:
      return self['lr']
    else:
      return None

  def _SetLanguageRestriction(self, val):
    self['lr'] = val

  lr = property(_GetLanguageRestriction, _SetLanguageRestriction,
                doc="""The lr (language restriction) query parameter""")

  def _GetIPRestriction(self):
    if 'restriction' in self:
      return self['restriction']
    else:
      return None

  def _SetIPRestriction(self, val):
    self['restriction'] = val

  restriction = property(_GetIPRestriction, _SetIPRestriction,
                         doc="""The restriction query parameter""")

  def _GetLocation(self):
    if 'location' in self:
      return self['location']
    else:
      return None

  def _SetLocation(self, val):
    self['location'] = val

  location = property(_GetLocation, _SetLocation,
                      doc="""The location query parameter""")



class YouTubeUserQuery(YouTubeVideoQuery):
  def __init__(self, username=None, feed_type=None, subscription_id=None,
               text_query=None, params=None, categories=None):

    uploads_favorites_playlists = ('uploads', 'favorites', 'playlists')

    if feed_type is 'subscriptions' and subscription_id and username:
      feed = "http://%s/feeds/users/%s/%s/%s" % (YOUTUBE_SERVER, username,
                                                 feed_type, subscription_id)
    elif feed_type is 'subscriptions' and not subscription_id and username:
      feed = "http://%s/feeds/users/%s/%s" % (YOUTUBE_SERVER, username,
                                              feed_type)
    elif feed_type in uploads_favorites_playlists:
      feed = "http://%s/feeds/users/%s/%s" % (YOUTUBE_SERVER, username, 
                                              feed_type)
    else:
      feed = "http://%s/feeds/users" % (YOUTUBE_SERVER)

    YouTubeVideoQuery.__init__(self, feed=feed, text_query=text_query,
                               params=params, categories=categories)


class YouTubePlaylistQuery(YouTubeVideoQuery):
  def __init__(self, playlist_id, text_query=None, params=None,
               categories=None):
    if playlist_id:
      feed = "http://%s/feeds/playlists/%s" % (YOUTUBE_SERVER, playlist_id)
    else:
      feed = "http://%s/feeds/playlists" % (YOUTUBE_SERVER)

    YouTubeVideoQuery.__init__(self, feed=feed, text_query=text_query,
                               params=params, categories=categories)
