/*
 *
 *  Copyright (c) 2010 Christoph Keller <gri@nospam@not-censored.com>
 *
 *  This file is part of Web2SMS.
 *
 *  Web2SMS is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Web2SMS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Web2SMS. If not, see <http://www.gnu.org/licenses/>
 *
 */

// Local includes
#include "mainwindow.hpp"
#include "messagewidget.hpp"
#include "startwidget.hpp"
#include "providers/providerinterface.hpp"
#include "providers/accountsdialog.hpp"
#include "providers/accounts.hpp"

// Global includes
#include <QtCore/QSettings>
#include <QtCore/QDir>
#include <QtGui/QProgressDialog>
#include <QtGui/QMessageBox>
#include <QtGui/QMenuBar>
#include <QtGui/QCloseEvent>
#include <QtGui/QDesktopServices>

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class MainWindowPrivate : public QObject
{
  Q_OBJECT

public:
  MainWindowPrivate()
  : accounts(new Accounts)
  , messageWidget(NULL)
  , settings(NULL)
  {}

  MainWindow* self;

  // Properties
  AccountsPtr accounts;
  StartWidget* startWidget;
  MessageWidget* messageWidget;

  QSettings* settings;

  // Functions
  void init();

  void saveMessageTypeSettings();

  void login();
  void logout();

public slots:
  // Ui callbacks
  void onEditAccounts();
  void onAliasClicked(const QString& alias);
  void onEnableRotation();
};

//////////////////////////////////////////////////////////////////////////

void MainWindowPrivate::init()
{
  self->setWindowTitle( tr("Web2SMS") );

  accounts->loadPlugins();
  accounts->loadSettings();

  // Hackish solution for remembering the ui settings
  QDir settingsDir(QDesktopServices::storageLocation(QDesktopServices::HomeLocation));
  settingsDir.cd(".web2sms");
  settings = new QSettings(settingsDir.absoluteFilePath("web2sms.conf"), QSettings::IniFormat, this);

  startWidget = new StartWidget(self);
  startWidget->setAccounts(accounts);

  messageWidget = new MessageWidget(self, Qt::Window);

#ifdef Q_WS_MAEMO_5
  self->setAttribute(Qt::WA_Maemo5StackedWindow);
  messageWidget->setAttribute(Qt::WA_Maemo5StackedWindow);
#endif // Q_WS_MAEMO_5

  // Set the message widget as central widget
  self->setCentralWidget(startWidget);

  // Setup the menu
  QMenu* menu = self->menuBar()->addMenu(QString());
  menu->addAction(tr("Accounts ..."), this, SLOT(onEditAccounts()));
  menu->addAction(tr("Enable rotation"), this, SLOT(onEnableRotation()));

#if !defined(Q_WS_S60) && !defined(Q_WS_MAEMO_5)
  // We have to set a window title, otherwise Windows won't show the menu
  menu->setTitle("menu");
#endif

  // Make the connections
  connect(startWidget, SIGNAL(accountClicked(QString)), this, SLOT(onAliasClicked(QString)));
}

//////////////////////////////////////////////////////////////////////////

void MainWindowPrivate::onEnableRotation()
{
#ifdef Q_WS_MAEMO_5
  self->setAttribute(Qt::WA_Maemo5AutoOrientation);
  messageWidget->setAttribute(Qt::WA_Maemo5AutoOrientation);
#endif // Q_WS_MAEMO_5
}

//////////////////////////////////////////////////////////////////////////

void MainWindowPrivate::onEditAccounts()
{
  // Show the accounts dialog
  AccountsDialog dialog(self);
  dialog.setAccounts(accounts);

  // Run the dialog (we don't care about the exit code)
  dialog.exec();

  // Always save the account settings when the dialog is finished
  accounts->saveSettings();
}

//////////////////////////////////////////////////////////////////////////

void MainWindowPrivate::onAliasClicked(const QString& alias)
{
  // Save the message type settings
  saveMessageTypeSettings();

  // Find the provider interface
  ProviderInterface* provider = accounts->providerForAlias(alias);

  // Make the message view prettier
  ProviderInfo info = accounts->providerInfoForAlias(alias);
  messageWidget->setWindowTitle( tr("Send SMS with %1").arg(info.visibleName()) );

  // Show the message widget
  messageWidget->show();

  // Set the provider to the message widget
  messageWidget->setProvider(provider);

  // Restore the previously selected message type
  {
    // Create the key and lookup the value
    QString keyName = QString("messageWidget/%1/currentMessageType").arg(info.providerId().toString());
    if ( settings->contains(keyName) )
    {
      // Get the message type name
      QString messageTypeName = settings->value(keyName).toString();

      // Create a dummy message type for lookup
      MessageType dummyMessageType(QString(), messageTypeName, QIcon(), QString(), 0, 0, 0);
      messageWidget->setCurrentMessageType(dummyMessageType);
    }
  }
}

//////////////////////////////////////////////////////////////////////////

void MainWindowPrivate::saveMessageTypeSettings()
{
  // Save the settings of the old provider if possible
  ProviderInterface* oldProvider = messageWidget->provider();
  if ( oldProvider )
  {
    // Get information about the provider
    ProviderInfo providerInfo = accounts->providerInfoForProvider(oldProvider);

    // Get the currently selected message type
    MessageType messageType = messageWidget->currentMessageType();

    // Create the key and set the value
    QString keyName = QString("messageWidget/%1/currentMessageType").arg(providerInfo.providerId().toString());
    settings->setValue(keyName, messageType.name());
  }
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

MainWindow::MainWindow(QWidget *parent /* = 0 */)
: QMainWindow(parent)
, d(new MainWindowPrivate)
{
  d->self = this;
  d->init();
}

//////////////////////////////////////////////////////////////////////////

MainWindow::~MainWindow()
{
  d->saveMessageTypeSettings();
  delete d;
}

//////////////////////////////////////////////////////////////////////////

void MainWindow::init()
{
}

//////////////////////////////////////////////////////////////////////////

void MainWindow::closeEvent(QCloseEvent* closeEvent)
{
  // Logout first?
  ProviderInterface* provider = d->messageWidget ? d->messageWidget->provider() : NULL;

  if ( provider && provider->isLoggedIn() )
  {
    CallbackType cb = provider->logout();
    if ( cb == Callback_Sync )
    {
      // Save the settings when closing the program
      d->accounts->saveSettings();

      QWidget::closeEvent(closeEvent);
    }
    else
    {
      // Ignore the close event
      closeEvent->ignore();

      // Show a dialog that we're logging out first
      QProgressDialog* progressDialog = new QProgressDialog(this);
      progressDialog->setAttribute(Qt::WA_DeleteOnClose);
      progressDialog->setWindowTitle( tr("Logging out ...") );
      progressDialog->setRange(0, 0);

      // Close the information dialog and this window
      connect(provider, SIGNAL(logoutReply(bool)), progressDialog, SLOT(accept()));
      connect(provider, SIGNAL(logoutReply(bool)), this, SLOT(close()));

      progressDialog->show();
    }
  }
  else
  {
    // Save the settings when closing the program
    d->accounts->saveSettings();

    QWidget::closeEvent(closeEvent);
  }
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

#include "mainwindow.moc"
