unit conicconnection;

{$mode objfpc}{$H+}
{$packrecords C}

interface

uses
  Forms,Classes, SysUtils, glib2, conicconnectionevent, dialogs, dbus;

type
{**
 * ConIcConnection object. The contents of the object is private, use only
 * the functions provided by conicconnection.h.
 *}
  TConIcConnection = record
  end;
  PConIcConnection = ^TConIcConnection;

  TConIcConnectionClass = record
  end;
  PConIcConnectionClass = ^TConIcConnectionClass;

  {**
   * Connection request flags. With these flags special connection requests
   * can be made. The flags can be ORred.
   *
   * Normally just use CON_IC_CONNECT_FLAG_NONE.
   *}
  TConIcConnectFlags = (CON_IC_CONNECT_FLAG_NONE = 0,CON_IC_CONNECT_FLAG_AUTOMATICALLY_TRIGGERED = 1 shl 0, CON_IC_CONNECT_FLAG_UNMANAGED = 1 shl 1 );
{  	/** No flags set. */
  	CON_IC_CONNECT_FLAG_NONE = 0,
  	/** Connection establishment wasn't started by a user action.
  	 * Instead it was triggered by a timer, or something similar, event
  	 * from the application. Using this flags means that if a
  	 * connection isn't already established, the connection request
  	 * will fail. */
  	CON_IC_CONNECT_FLAG_AUTOMATICALLY_TRIGGERED = 1 << 0,
  	/** Process requesting the connection won't be monitored for
  	 * killing. Normally if the process died, it will be automatically
  	 * detached from the connection. If this flag is set and process
  	 * dies, the connection won't closed automatically.*/
  	CON_IC_CONNECT_FLAG_UNMANAGED = 1 << 1}







function con_ic_connection_get_type:GType; cdecl; external 'libconic.so';

{**
 * Creates new ConIcConnection. When not needed anymore, release with
 * g_object_unref().
 *
 * Creating the object does not yet open the connection. Use, for example,
 * con_ic_connection_connect() to do that.
 *
 * @return New ConIcConnection object. NULL, if the object creation
 * failed.
 *}
 function con_ic_connection_new:PConIcConnection; cdecl; external 'libconic.so';



 {**
  * Request for a connection. The Internet Connectivity subsystem will
  * choose the best connection based on user settings and input provided by
  * the user.
  *
  * Answer is sent using the connection-event signal from a ConIcConnection
  * object. If the connection establishment succeeded, or there was already
  * a connection established, a CON_IC_STATUS_CONNECTED inside
  * ConIcConnectionEvent is sent. If connection establishment failed, a
  * CON_IC_STATUS_DISCONNECTED inside ConIcConnectionEvent is sent and error
  * is set accordingly.
  *
  * Normally this one should be used.
  *
  * @param connection ConIcConnection object.
  * @param flags Flags for the request.
  * @retval TRUE If no DBUS errors.
  * @retval FALSE If a DBUS message couldn't be sent.
  *}
 function con_ic_connection_connect(connection : PConIcConnection; flags : TConIcConnectFlags):gboolean; cdecl; external 'libconic.so';



 {**
  * Request for a connection using the IAP id. The Internet Connectivity
  * subsystem will choose the best connection based on user settings and
  * input provided by the user.
  *
  * Answer is sent using the connection-event signal from a ConIcConnection
  * object. If the connection establishment succeeded, or there was already
  * a connection established, a CON_IC_STATUS_CONNECTED inside
  * ConIcConnectionEvent is sent. If connection establishment failed, a
  * CON_IC_STATUS_DISCONNECTED inside ConIcConnectionEvent is sent and error
  * is set accordingly.
  *
  * Normally con_ic_connection_connect() should be used. Use this one if you
  * want to use a specific connection.
  *
  * @param connection ConIcConnection object.
  * @param flags Flags for the request.
  * @param id Id of the requested IAP.
  * @retval TRUE If no DBUS errors.
  * @retval FALSE If a DBUS message couldn't be sent.
  *}
 function con_ic_connection_connect_by_id(connection : PConIcConnection; const id : pgchar; flags : TConIcConnectFlags ):gboolean;  cdecl; external 'libconic.so';



 {**
  * Disconnects all IAPs associated with the connection.
  *
  * Normally use this one.
  *
  * @param connection ConIcConnection object.
  * @retval TRUE If no DBUS errors.
  * @retval FALSE If a DBUS message couldn't be sent.
  *}
 function con_ic_connection_disconnect(connection : PConIcConnection): gboolean; cdecl; external 'libconic.so';

 {**
  * Disconnects specific IAP associated with the application.
  *
  * Normally use con_ic_connection_disconnect().
  *
  * @param connection ConIcConnection object.
  * @param id Id of the IAP to disconnected.
  * @retval TRUE If no DBUS errors.
  * @retval FALSE If a DBUS message couldn't be sent.
  *}
 function con_ic_connection_disconnect_by_id(connection : PConIcConnection; const id : pgchar):gboolean; cdecl; external 'libconic.so';



 {**
  * Requests statistics for a IAP. The answer is sent as ConIcStatistics in
  * statistics signal.
  *
  * @param connection ConIcConnection object.
  * @param id Id of the IAP
  * @retval TRUE If no DBUS errors.
  * @retval FALSE If a DBUS message couldn't be sent. No statistics events will
  *         be emitted.
  *}
 function con_ic_connection_statistics(connection :  PConIcConnection; const id : pgchar):gboolean; cdecl; external 'libconic.so';

 { My macro to open the connection}
 function ConIcOpenConnection(flags : TConIcConnectFlags; var connected : boolean):PConIcConnection;
 { My macro to close the connection }
 procedure ConIcCloseConnection(var connection : PConIcConnection);


 procedure dbus_connection_setup_with_g_main(connection : PDBusConnection; context : PGMainContext); cdecl; external 'libdbus-glib-1.so';

 { Callbacks... }
 procedure on_conn_event(connection : PConIcConnection; event : PConIcConnectionEvent; user_data : gpointer); cdecl;

implementation

var
  user_data : GPointer;
  got_connected : boolean;
  got_error : boolean;

procedure on_conn_event(connection : PConIcConnection; event : PConIcConnectionEvent; user_data : gpointer); cdecl;
var
  status : TConIcConnectionStatus;
  error : TConIcConnectionError;
begin
  status := con_ic_connection_event_get_status(event);
  if status = CON_IC_STATUS_CONNECTED then
    begin
      got_connected := True;
    end;
  error := con_ic_connection_event_get_error(event);
  if error <> CON_IC_CONNECTION_ERROR_NONE then
    begin
      got_error := True;
    end;
end;

function ConIcOpenConnection(flags : TConIcConnectFlags; var connected : boolean):PConIcConnection;
var
  dbusEr : DBusError;
  dbusConn : PDBusConnection;
begin
  g_type_init();
  dbus_error_init(@dbusEr);
  dbusConn := dbus_bus_get(DBUS_BUS_SYSTEM, @dbusEr);
  if dbusConn <> nil then
    begin
      dbus_connection_setup_with_g_main(dbusConn, nil);
    end
  else
    begin
      Exit(nil);
    end;
  Result := con_ic_connection_new;
  if Result = nil then
    begin
      Exit(nil);
    end;
  // Create callbacks...
  got_connected := False;
  got_error := False;
  g_signal_connect(g_object(Result), 'connection-event', G_CALLBACK(@on_conn_event), nil);
  g_object_set(G_OBJECT(Result), 'automatic-connection-events', TRUE, nil);
  if not con_ic_connection_connect(Result,flags) then
    begin
      Exit(nil);
    end;
  while true do
    begin
      Application.ProcessMessages;
      if got_connected then
        Break;
      if got_error then
        Break;
      Sleep(1000);
    end;
  if got_error then
    connected := False
  else
    connected := True;
end;

procedure ConIcCloseConnection(var connection : PConIcConnection);
begin
  g_object_unref(connection);
end;

end.

