#include <QtGui>
#include <QtDebug>
#include <QInputDialog>
#ifdef Q_WS_MAEMO_5
#include <QtMaemo5/QMaemo5InformationBox>
#endif
#include "mainwindow.h"
#include "optiondialog.h"
#include "equalizerdialog.h"
#include "time.h"

//#define AVOID_INPUT_DIALOG

MainWindow::MainWindow()
    : eqEffect (0),
    plman (this), 
    settings (tr ("TomAmp"), "TomAmp"),
    isPlaying (false),
    autostopped (false)
{
    mwa = new MainWindowActions (this, settings.value("repeat", false).toBool (), settings.value("shuffle", false).toBool ());
    QList<Phonon::EffectDescription> effectDescriptions =
                 Phonon::BackendCapabilities::availableAudioEffects();
    audioOutput = new Phonon::AudioOutput(Phonon::MusicCategory, this);
    mediaObject = new Phonon::MediaObject(this);

    mediaObject->setTickInterval(1000);
    connect(mediaObject, SIGNAL(tick(qint64)), this, SLOT(tick(qint64)));
    connect(mediaObject, SIGNAL(stateChanged(Phonon::State,Phonon::State)),
        this, SLOT(stateChanged(Phonon::State,Phonon::State)));
    connect(mediaObject, SIGNAL(currentSourceChanged(Phonon::MediaSource)),
        this, SLOT(sourceChanged(Phonon::MediaSource)));
    connect(mediaObject, SIGNAL(aboutToFinish()), this, SLOT(aboutToFinish()));
    connect (&plman, SIGNAL (playlistChanged (int)), this, SLOT (playlistChanged(int)));
    connect (&plman, SIGNAL (itemUpdated(int)), this, SLOT (itemUpdated (int)));
    connect (&plman, SIGNAL (itemRemoved(int)), this, SLOT (itemRemoved (int)));

    connect (QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));
    timer.setSingleShot(true);
    connect (&timer, SIGNAL(timeout()), this, SLOT(hideVolumeBar()));
#ifdef Q_WS_MAEMO_5
    QDBusConnection::systemBus().connect(QString(), "/org/freedesktop/Hal/devices/platform_headphone", "org.freedesktop.Hal.Device", "Condition", this, SLOT(headphoneEvent(QString,QString)));
#endif    
    foreach (Phonon::EffectDescription effectDescription, effectDescriptions) 
    {
        qDebug () << effectDescription.name();
        if ((effectDescription.name() == "KEqualizer") || (effectDescription.name() == "equalizer-10bands"))
        {
            eqEffect = new Phonon::Effect(effectDescription);
//            qDebug() << "Effect: " << eqEffect;
            int i = 0;
            foreach (Phonon::EffectParameter par, eqEffect->parameters())
            {
                QString valName = QString ("eq_%1").arg(i++);
                double dval = settings.value(valName, 0.0).toDouble();
                eqEffect->setParameterValue(par, dval);
                qDebug() << par.name() << " => " << par.minimumValue() << " to " << par.maximumValue();
            }
        }
        else
        {
            Phonon::Effect *tmpEffect = new Phonon::Effect(effectDescription);
            foreach (Phonon::EffectParameter par, tmpEffect->parameters())
            {
                qDebug() << par.name() << " => " << par.minimumValue() << " to " << par.maximumValue();
            }
            delete tmpEffect;
        }
    }

    audioPath = Phonon::createPath(mediaObject, audioOutput);
    if (eqEffect && settings.value("equalizer", false).toBool())
    {
        audioPath.insertEffect(eqEffect);
    }

    qsrand (time (NULL));
    repeat = settings.value("repeat", false).toBool();
    shuffle = settings.value("shuffle", false).toBool();
    QStringList allowedHeaders;
    allowedHeaders << "Artist" << "Title" << "Album" << "Controls";
    foreach (QString h, settings.value ("headers", QStringList()).toStringList())
    {
        if (allowedHeaders.indexOf(h) >= 0)
            headers << h;
    }
    if (!headers.size())
        headers << "Artist" << "Title" << "Album";
    
    setupShuffleList();
    setupActions();
    setupMenus();
    
    QRect screenGeometry = QApplication::desktop()->screenGeometry();
    portrait = screenGeometry.width() < screenGeometry.height();
    setupUi ();
    setWindowTitle("TomAmp");
    show ();
    timeLcd->display("00:00:00");
    plman.addStringList(settings.value("lastPlaylist").toStringList(),
                        settings.value("playlistTitles").toStringList(),
                        settings.value("playlistPositions").toList()
                        );
    setupShuffleList();
    int curind = settings.value("currentIndex", -1).toInt ();
    if (curind >= 0)
        setItem (curind, false);
    audioOutput->setVolume(settings.value("volume", .5).toDouble());
    QApplication::setWindowIcon(QIcon (QPixmap (":images/Tomamp.png")));
    setOrientation();
}

MainWindow::~MainWindow()
{
    settings.setValue("shuffle", shuffle);
    settings.setValue("repeat", repeat);
    settings.setValue("lastPlaylist", plman.playlistStrings());
    settings.setValue("playlistTitles", plman.playlistTitles());
    settings.setValue("playlistPositions", plman.playlistPositions());
    settings.setValue("volume", audioOutput->volume());
    settings.setValue("currentIndex", lastPlayed);
    settings.setValue("headers", headers);
    for (int i = 0; i < musicTable->columnCount(); ++i)
    {
        QString lab = QString ("colWidth_%1").arg (i);
        settings.setValue(lab, musicTable->columnWidth(i));
    }
}

void MainWindow::setOrientation ()
{
#ifdef Q_WS_MAEMO_5
    QString orient = settings.value("orientation", "Automatic").toString();
    if (orient == "Portrait")
    {
        setAttribute(Qt::WA_Maemo5PortraitOrientation, true);
    }
    else if (orient == "Landscape")
    {
        setAttribute(Qt::WA_Maemo5LandscapeOrientation, true);
    }
    else
    {
        setAttribute(Qt::WA_Maemo5AutoOrientation, true);
    }
#endif
}

void MainWindow::orientationChanged()
{
    QRect screenGeometry = QApplication::desktop()->screenGeometry();
    portrait = screenGeometry.width() < screenGeometry.height();
    delete centralWidget();
    setupUi();
    int current = musicTable->currentRow();
    playlistChanged(0);
    if (current >= 0)
    {
        musicTable->selectRow(current);
        highlightRow(current);
    }
}


void MainWindow::addFiles()
{
    QString folder = settings.value("LastFolder").toString();
    if (folder.isEmpty())
        folder = QDesktopServices::storageLocation(QDesktopServices::MusicLocation);
    QString ext = "*." + plman.allowedExt().join(" *.");
    ext = "Music files (" + ext + ");;Playlists (*.m3u *.pls *.cue)";
    QStringList files = QFileDialog::getOpenFileNames(this, tr("Select Files To Add"),
                                                      folder, ext);

    if (files.isEmpty())
        return;

    QString dir = QFileInfo (files[0]).absoluteDir().absolutePath();
    settings.setValue("LastFolder", dir);
    QStringList toadd;
    foreach (QString string, files)
    {
        if (string.toLower().endsWith(".pls") || string.toLower().endsWith(".m3u") || string.toLower().endsWith(".cue"))
            plman.addPlaylist(string);
        else
            toadd.append (string);
    }
    plman.addStringList(toadd);
}

void MainWindow::addFolder()
{
    QString folder = settings.value("LastFolder").toString();
    if (folder.isEmpty())
        folder = QDesktopServices::storageLocation(QDesktopServices::MusicLocation);
    QString dir = QFileDialog::getExistingDirectory(this,
            tr("Select Directory To Add"),
            folder);

    if (dir.isEmpty())
        return;

    settings.setValue("LastFolder", dir);

    QStringList filters;
    QStringList files = QDir (dir).entryList(filters, QDir::AllDirs);
    files.removeAll(".");
    files.removeAll("..");
    bool recursive = false;
    if (files.size())
        recursive = QMessageBox::question(this, "Add all folders", "Subfolders have been detected, add everything?", QMessageBox::Yes | QMessageBox::No) == QMessageBox::Yes;
    plman.parseAndAddFolder(dir, recursive);
}


void MainWindow::addUrl()
{
#ifdef AVOID_INPUT_DIALOG
    QString url = "http://streams.bigvibez.com:7000/listen.pls";
#else
    QString url = QInputDialog::getText(this, "Get URL", "Please type in the stream URL");
#endif
    if (url.isEmpty() || !url.toLower().startsWith("http"))
        return;
    QStringList toadd;
    toadd << url;
    plman.addStringList(toadd);
}


void MainWindow::about()
{
    QMessageBox msg;
    msg.setIconPixmap(QPixmap (":images/Tomamp.png"));
    msg.setModal(true);
    msg.setWindowTitle(tr ("About TomAmp v0.4.4.1"));
    msg.setText(tr("TomAmp is a simple playlist-based music player.\n\n"
                "(c) 2010 Tamas Marki <tmarki@gmail.com>\n"
                "Please send comments and bug reports to the above e-mail address.\n\n"
                "Icons by Lako of #eperland @ quakenet\n"
                "Special thanks to Attila Csipa, and the people at talk.maemo.org"));
    msg.exec ();
}

void MainWindow::stateChanged(Phonon::State newState, Phonon::State /* oldState */)
{
    int ind = plman.indexOf(mediaObject->currentSource());
    switch (newState)
    {
        case Phonon::ErrorState:
            if (mediaObject->errorType() == Phonon::FatalError)
            {
//                QMessageBox::warning(this, tr("Fatal Error"),
//                mediaObject->errorString() + mediaObject->currentSource().fileName() + ", " + mediaObject->currentSource().url().toString());
            }
            else
            {
//                QMessageBox::warning(this, tr("Error"),
//                mediaObject->errorString());
            }
            next ();
            break;
        case Phonon::PlayingState:
            mwa->playingState();
            setWindowTitle(mediaObject->metaData().value("TITLE") + "(" + mediaObject->metaData().value("ARTIST") + ") - TomAmp");
            unhighlightRow(lastPlayed);
            lastPlayed = plman.indexOf(mediaObject->currentSource());
            highlightRow(plman.indexOf(mediaObject->currentSource()));
            musicTable->selectRow(plman.indexOf(mediaObject->currentSource()));
            autostopped = false;
            if (plman.getItem(ind).subPos)
            {
                qDebug () << "subpos " << plman.getItem(ind).subPos;
                mediaObject->seek(plman.getItem(ind).subPos);
            }
            break;
        case Phonon::StoppedState:
            mwa->stoppedState();
            setWindowTitle("TomAmp");
            timeLcd->display("00:00:00");
            unhighlightRow(plman.indexOf(mediaObject->currentSource()));
            autostopped = false;
            break;
        case Phonon::PausedState:
            mwa->pausedState();
            break;
        case Phonon::BufferingState:
            break;
        default:
        ;
    }
}

void MainWindow::next()
{
    bool wasPlaying = isPlaying;
    if (mediaObject->state () == Phonon::ErrorState)
        wasPlaying = true;
    if (mediaObject->queue().size())
    {
        setItem (plman.indexOf(mediaObject->queue()[0]), wasPlaying);
        mediaObject->queue().clear();
        return;
    }
    int index = plman.indexOf(mediaObject->currentSource());
    if (shuffle)
    {
        index = shuffleList.indexOf(plman.indexOf(mediaObject->currentSource())) + 1;
        while (index < shuffleList.size () && !plman.getItem(shuffleList[index]).playable)
        {
            index += 1;
        }
        if (index < shuffleList.size ())
        {
            setItem (index, wasPlaying);
        }
        else if (repeat)
        {
            index = 0;
            while ((index) < shuffleList.size () && !plman.getItem(shuffleList[index]).playable)
            {
                index += 1;
            }
            setItem (index, wasPlaying);
        }
    }
    else
    {
        index++;
        while ((index) < plman.size () && !plman.getItem(index).playable)
        {
            index += 1;
        }
        if (index < plman.size())
        {
            setItem (index, wasPlaying);
        }
        else if (repeat)
        {
            index = 0;
            while ((index) < plman.size () && !plman.getItem(index).playable)
            {
                index += 1;
            }
            setItem (index, wasPlaying);
        }
    }
}

void MainWindow::setItem(int i, bool doplay)
{
    if (i < plman.size() && i >= 0)
    {
/*        if (lastPlayed >= 0)
            unhighlightRow(lastPlayed);*/
        if (shuffle)
        {
            mediaObject->setCurrentSource(plman.at (shuffleList[i]));
        }
        else
        {
            mediaObject->setCurrentSource(plman.at(i));
        }
    }
    if (doplay && mediaObject->currentSource().type() != Phonon::MediaSource::Invalid)
    {
        play();
    }
    else
        stop ();
}

void MainWindow::previous()
{
    bool wasPlaying = isPlaying;//(mediaObject->state () == Phonon::PlayingState);
    if (mediaObject->state () == Phonon::ErrorState)
        wasPlaying = true;
    int index = plman.indexOf(mediaObject->currentSource());
    if (shuffle)
    {
        index = shuffleList.indexOf(plman.indexOf(mediaObject->currentSource())) - 1;
        while (index >= 0 && !plman.getItem(shuffleList[index]).playable)
        {
            index--;
        }
        if (index >= 0)
        {
            setItem (index, wasPlaying);
        }
        else if (repeat)
        {
            index = plman.size () - 1;
            while (index >= 0 && !plman.getItem(shuffleList[index]).playable)
            {
                index--;
            }
            setItem (index, wasPlaying);
        }
/*        if (index < 0)
            wasPlaying = false;*/

    }
    else
    {
        index--;
        while ((index) >= 0 && !plman.getItem(index).playable)
        {
            index--;
        }
        if (index >= 0)
        {
            setItem (index, wasPlaying);
        }
        else if (repeat)
        {
            index = plman.size() - 1;
            while ((index) >= 0 && !plman.getItem(index).playable)
            {
                index--;
            }
            setItem (index, wasPlaying);
        }
    }
}

void MainWindow::highlightRow (int i)
{
    for (int j = 0; j < 3; ++j)
    {
        QTableWidgetItem* item = musicTable->item(i, j);
        if (item)
        {
            QFont font = item->font();
            font.setBold(true);
            font.setItalic(true);
            item->setFont(font);
        }
    }
}

void MainWindow::unhighlightRow (int i)
{
    for (int j = 0; j < 3; ++j)
    {
        QTableWidgetItem* item = musicTable->item(i, j);
        if (item)
        {
            QFont font = item->font();
            font.setBold(false);
            font.setItalic(false);
            item->setFont(font);
        }
    }
}


void MainWindow::tick(qint64 time)
{
    QTime displayTime((time / 3600000), (time / 60000) % 60, (time / 1000) % 60);

    timeLcd->display(displayTime.toString("HH:mm:ss"));
}

void MainWindow::tableClicked(int row, int /* column */)
{
//    bool wasPlaying = mediaObject->state() == Phonon::PlayingState;

/*    mediaObject->stop();
    mediaObject->clearQueue();*/

    if (row >= plman.size())
        return;

    int index = row;
    while (index < shuffleList.size () && !plman.getItem(index).playable)
    {
        index += 1;
    }
    if (plman.size() > index)
    {
        if (shuffle)
            index = shuffleList.indexOf(index);
        setItem (index, true);
//        mediaObject->play();
    }
    else
    {
        index = 0;
        while (index < plman.size () && !plman.getItem(index).playable)
        {
            index += 1;
        }
        if (plman.size() > index)
        {
            if (shuffle)
                index = shuffleList.indexOf(index);
            setItem (index, true);
//            mediaObject->play();
        }
    }

}

void MainWindow::sourceChanged(const Phonon::MediaSource &source)
{
    int ind = plman.indexOf(source);
    highlightRow(ind);
    unhighlightRow(lastPlayed);
    lastPlayed = ind;
    musicTable->selectRow(ind);
    timeLcd->display("00:00:00");
}


void MainWindow::aboutToFinish()
{
    if (mediaObject->queue().size())
        return;
    int index = plman.indexOf(mediaObject->currentSource()) + 1;
    if (shuffle)
    {
        index = shuffleList.indexOf(plman.indexOf(mediaObject->currentSource())) + 1;
        if (index < shuffleList.size ())
        {
            mediaObject->enqueue(plman.at (shuffleList[index]));
        }
        else if (repeat)
        {
            mediaObject->enqueue(plman.at (shuffleList[0]));
        }

    }
    else
    {
        if (plman.size() > index)
        {
            mediaObject->enqueue(plman.at(index));
        }
        else if (repeat)
        {
            mediaObject->enqueue(plman.at(0));
        }
    }
}

void MainWindow::finished()
{
}

void MainWindow::setupActions()
{
}

void MainWindow::removeSelectedItem()
{
    if (QMessageBox::question(this, "Confirm remove", "Are you sure you want to remove this item?", QMessageBox::Yes, QMessageBox::No) == QMessageBox::No)
        return;
    int row = musicTable->currentRow();
    if (row >= 0)
        plman.removeItem(row);
}

void MainWindow::removeAllButSelectedItem()
{
    if (QMessageBox::question(this, "Confirm remove", "Are you sure you want to remove all other items?", QMessageBox::Yes, QMessageBox::No) == QMessageBox::No)
        return;
    int row = musicTable->currentRow();
    if (row >= 0)
    {
        QString uri = plman.getItem(row).uri;
        QStringList lst;
        lst << uri;
        plman.clearPlaylist();
        plman.addStringList(lst);
    }
}

void MainWindow::repeatToggle ()
{
    repeat = !repeat;
    settings.setValue("repeat", QVariant (repeat));
    mwa->repeatToggle(repeat);
}

void MainWindow::shuffleToggle ()
{
    shuffle = !shuffle;
    settings.setValue("shuffle", QVariant (shuffle));
    mwa->shuffleToggle(shuffle);
}

void MainWindow::volumeToggle ()
{
    volumeSlider->setVisible(mwa->isVolumeChecked());
    timer.start(5000);
}

void MainWindow::hideVolumeBar()
{
    volumeSlider->hide();
    musicTable->show();
    mwa->setVolumeCheck(false);
}

void MainWindow::play()
{
    mediaObject->play();
//    lastPlayed = plman.indexOf(mediaObject->currentSource());
    highlightRow(lastPlayed);
    isPlaying = true;
}

void MainWindow::stop()
{
    mediaObject->stop();
    isPlaying = false;
}


void MainWindow::setupMenus()
{
    QMenu* fm = menuBar()->addMenu(tr("&File"));
    QMenu* hm = menuBar()->addMenu(tr("&Help"));
    mwa->setupMenus(fm, hm);
}

void MainWindow::setupUi()
{
    QToolBar *bar = new QToolBar;
    bool flip = !portrait;

    if(!flip) bar->setOrientation(Qt::Vertical);
    bar->setStyleSheet("padding:0px");

    seekSlider = new Phonon::SeekSlider(this);
    seekSlider->setMediaObject(mediaObject);
    if (flip) seekSlider->setOrientation(Qt::Vertical);

    volumeSlider = new Phonon::VolumeSlider(this);
    volumeSlider->setAudioOutput(audioOutput);
    if (flip)
    {
        volumeSlider->setOrientation(Qt::Vertical);
        volumeSlider->setMinimumHeight(150);
    }
    else
    {
        volumeSlider->setMinimumWidth(150);
    }
    volumeSlider->setMuteVisible(false);
    int iconsize = settings.value ("iconsize", 48).toInt();
    bar->setIconSize(QSize (iconsize, iconsize));
    mwa->setupToolbar(bar);
    contextMenu = new QMenu (this);

    addMenu = new QMenu (this);
    
    mwa->setupContextAndAddMenu(contextMenu, addMenu, this);    

    timeLcd = new QLCDNumber;

    if (!headers.size ())
    {
        headers << "Artist" << "Title" << "Album";
        settings.setValue("headers", headers);
    }

    musicTable = new QTableWidget(0, headers.size ());
    musicTable->setHorizontalHeaderLabels(headers);
    musicTable->setSelectionMode(QAbstractItemView::SingleSelection);
    musicTable->setSelectionBehavior(QAbstractItemView::SelectRows);
    musicTable->verticalHeader ()->hide();
    connect(musicTable, SIGNAL(cellDoubleClicked(int,int)),
        this, SLOT(tableClicked(int,int)));
    connect(musicTable, SIGNAL(cellClicked(int,int)),
        this, SLOT(cellClicked(int,int)));
    for (int i = 0; i < musicTable->columnCount(); ++i)
    {
        QString lab = QString ("colWidth_%1").arg (i);
        int val = settings.value(lab, 0).toInt();
        if (val)
            musicTable->setColumnWidth(i, val);
    }

    QLayout *seekerLayout;
    QLayout *playbackLayout;
    if (flip)
    {
        seekerLayout = new QVBoxLayout;
        playbackLayout = new QHBoxLayout;
        bar->addWidget(timeLcd);
    }
    else
    {
        seekerLayout = new QHBoxLayout;
        playbackLayout = new QVBoxLayout;
    }
    seekerLayout->addWidget(seekSlider);
    if (!flip)
        seekerLayout->addWidget(timeLcd);
    volumeSlider->hide ();
    playbackLayout->addWidget(bar);

    QVBoxLayout *seekAndTableLayout = new QVBoxLayout;

    QHBoxLayout *mainLayout = new QHBoxLayout;
    if (flip)
        mainLayout->addWidget(volumeSlider);
    else
        seekAndTableLayout->addWidget(volumeSlider);
    seekAndTableLayout->addWidget(musicTable);
    mainLayout->addLayout(seekAndTableLayout);
    if (flip)
    {
        seekAndTableLayout->addLayout(playbackLayout);
        mainLayout->addLayout(seekerLayout);
    }
    else
    {
        seekAndTableLayout->addLayout(seekerLayout);
        mainLayout->addLayout(playbackLayout);
    }


    QWidget *widget = new QWidget;
    widget->setLayout(mainLayout);

    setCentralWidget(widget);
}

void MainWindow::cellClicked(int /*row*/, int)
{
}

void MainWindow::enqueueSelected()
{
    int sel = musicTable->currentRow();
    if (sel >= 0)
    {
        mediaObject->queue().clear();
        mediaObject->enqueue(plman.at(sel));
#ifdef Q_WS_MAEMO_5
        QMaemo5InformationBox::information(this, plman.getItem(sel).title + tr (" enqueued as next song"),
        QMaemo5InformationBox::DefaultTimeout);
#endif

    }
}

void MainWindow::contextMenuEvent (QContextMenuEvent*e)
{
    qDebug () << "0: " << childAt (e->pos());
    if (!childAt (e->pos()))
        return;
    qDebug () << "1: " << childAt (e->pos())->parentWidget();
    if (childAt (e->pos())->parentWidget() != musicTable)
        return;
    qDebug () << "2";
    contextMenu->popup(e->globalPos());
}


void MainWindow::setupShuffleList()
{
    QList<int> tmp;
    int index = plman.indexOf(mediaObject->currentSource());
    if (index < 0)
        index = 0;
    for (int i = 0; i < plman.size(); ++i)
    {
        if ((i != index))
            tmp.append(i);
    }
    shuffleList.clear();
    shuffleList.append (index);
    while (tmp.size ())
    {
        int ind = qrand () % tmp.size();
        shuffleList.append(tmp[ind]);
        tmp.removeAt(ind);
    }
}

void MainWindow::savePlaylist ()
{
    QString filename = QFileDialog::getSaveFileName(this, tr("Please select file name"), settings.value ("LastFolder", "").toString(), "Playlist Files (*.m3u *.pls)");
    if (filename.isEmpty())
        return;
    QString dir = QFileInfo (filename).absoluteDir().absolutePath();
    settings.setValue("LastFolder", dir);
    if (!plman.savePlaylist(filename))
    {
#ifdef Q_WS_MAEMO_5
        QMaemo5InformationBox::information(this, tr ("Error writing playlist file"),
        QMaemo5InformationBox::DefaultTimeout);
#else
        QMessageBox::critical(this, "Write error", "Error writing playlist file", QMessageBox::Ok);
#endif
    }

}

void MainWindow::loadPlaylist ()
{
    QString filename = QFileDialog::getOpenFileName(this, tr("Select playlist file to load"), settings.value ("LastFolder", "").toString(), "*.m3u *.pls *.cue");
    if (filename.isEmpty())
        return;
    QString dir = QFileInfo (filename).absoluteDir().absolutePath();
    settings.setValue("LastFolder", dir);
    plman.loadPlaylist (filename);
}

void MainWindow::playlistChanged(int from)
{
    while (musicTable->rowCount() > from)
    {
        musicTable->removeRow(musicTable->rowCount () - 1);
    }
    int firstGood = -1;
    for (int i = from; i < plman.size (); ++i)
    {
        if (firstGood < 0 && plman.getItem (i).playable)
            firstGood = i;
        int currentRow = musicTable->rowCount();
        musicTable->insertRow(currentRow);
        musicTable->setRowHeight(currentRow, (int)(settings.value("rowHeight", DEFAULT_ROW_HEIGHT).toDouble() * 1.3));
        setRowFromItem (currentRow, plman.getItem(i));
    }
/*    if (plman.indexOf(mediaObject->currentSource()) < 0)
    {
        setItem (firstGood, false);
    }*/
    setupShuffleList();
}

void MainWindow::setRowFromItem (int row, const PlaylistItem& item)
{
    if (row >= musicTable->rowCount())
        return;
    if (item.artist.isEmpty() && item.title.isEmpty())
    {
        int col = headers.indexOf("Title");
        if (col >= 0)
        {
            QString title = item.uri;
            QFileInfo fi (title);
            title = fi.fileName();
            QTableWidgetItem *item1 = new QTableWidgetItem(title);
            item1->setFlags(item1->flags() ^ Qt::ItemIsEditable);
            musicTable->setItem(row, col, item1);
        }
    }
    else
    {
        int col = headers.indexOf("Artist");
        if (col >= 0)
        {
            QTableWidgetItem *item1 = new QTableWidgetItem(item.artist);
            item1->setFlags(item1->flags() ^ Qt::ItemIsEditable);
            musicTable->setItem(row, col, item1);
        }
        col = headers.indexOf("Title");
        if (col >= 0)
        {
            if (!musicTable->item (row, col))
            {
                QTableWidgetItem *item2 = new QTableWidgetItem(item.title);
                item2->setFlags(item2->flags() ^ Qt::ItemIsEditable);
                musicTable->setItem(row, col, item2);
            }
            else
            {
                musicTable->item (row, col)->setText(item.title);
            }
        }
        col = headers.indexOf("Album");
        if (col >= 0)
        {
            QTableWidgetItem *item3 = new QTableWidgetItem(item.album);
            item3->setFlags(item3->flags() ^ Qt::ItemIsEditable);

            musicTable->setItem(row, col, item3);
        }
    }

    int controlCol = headers.indexOf("Controls");
    if (controlCol >= 0 && !musicTable->cellWidget(row, controlCol))
    {
        QLabel* label = new QLabel;
        label->setText(QString::fromUtf8("<b><a style='text-decoration:none' href='up'>▲</a> <a style='text-decoration:none' href='down'>▼</a> <a style='text-decoration:none' href='del'>╳</a> <a style='text-decoration:none' href='enq'>E</a></b>"));
        label->setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Minimum);
        label->setProperty("row", row);
        musicTable->setCellWidget(row, controlCol, label);
        connect (label, SIGNAL (linkActivated (const QString&)),  this, SLOT (playlistControl (const QString&)));
    }
    QFont fnt;
    fnt.setPixelSize(settings.value("rowHeight", DEFAULT_ROW_HEIGHT).toInt());
    musicTable->setFont (fnt);
}

void MainWindow::playlistControl (const QString& con)
{
    int i = sender ()->property("row").toInt();
    if (con == "up")
        buttonUp(i);
    else if (con == "down")
        buttonDown(i);
    else if (con == "del")
        buttonDel (i);
    else if (con == "enq")
    {
        musicTable->selectRow(i);
        enqueueSelected();
    }
    else
        QMessageBox::information(this, tr ("Coming up..."), tr ("This feature is not implemented yet."));
}


void MainWindow::buttonUp(int i)
{
    if (i)
    {
        int currentPlaying = plman.indexOf(mediaObject->currentSource());
        plman.moveItemUp(i);
        setRowFromItem (i, plman.getItem(i));
        setRowFromItem (i - 1, plman.getItem(i - 1));
        int controlCol = headers.indexOf("Controls");
        if (controlCol >= 0)
        {
            musicTable->cellWidget(i, controlCol)->setProperty("row", i);
            musicTable->cellWidget(i - 1, controlCol)->setProperty("row", i - 1);
        }
        musicTable->selectRow(i - 1);
        if (currentPlaying == i)
        {
            unhighlightRow(i);
            highlightRow(i - 1);
        }
        else if (currentPlaying == i - 1)
        {
            unhighlightRow(i - 1);
            highlightRow(i);
        }
    }
}

void MainWindow::buttonDown(int i)
{
    if (i < plman.size() - 1)
    {
        int currentPlaying = plman.indexOf(mediaObject->currentSource());
        plman.moveItemDown(i);
        setRowFromItem (i, plman.getItem(i));
        setRowFromItem (i + 1, plman.getItem(i + 1));
        int controlCol = headers.indexOf("Controls");
        if (controlCol >= 0)
        {
            musicTable->cellWidget(i, controlCol)->setProperty("row", i);
            musicTable->cellWidget(i + 1, controlCol)->setProperty("row", i + 1);
        }
        musicTable->selectRow(i + 1);
        if (currentPlaying == i)
        {
            unhighlightRow(i);
            highlightRow(i + 1);
        }
        else if (currentPlaying == i + 1)
        {
            unhighlightRow(i + 1);
            highlightRow(i);
        }
    }
}

void MainWindow::buttonDel(int i)
{
    if (QMessageBox::question(this, "Confirm remove", "Are you sure you want to remove this item?", QMessageBox::Yes, QMessageBox::No) == QMessageBox::No)
        return;
    if (i < plman.size())
    {
        plman.removeItem(i);
    }
}

void MainWindow::itemUpdated(int index)
{
    if (plman.indexOf(mediaObject->currentSource()) < 0 && plman.getItem (index).playable)
    {
        setItem (index, false);
    }
    setRowFromItem (index, plman.getItem(index));
    if (plman.indexOf(mediaObject->currentSource()) == index)
    {
        if (shuffle) index = shuffleList.indexOf(index);
        setItem (index, false);
    }
}

void MainWindow::itemRemoved (int i)
{
    musicTable->removeRow(i);
    int controlCol = headers.indexOf("Controls");
    if (controlCol < 0)
        return;
    for (int j = i ? (i - 1) : 0; j < musicTable->rowCount(); ++j)
    {
        if (musicTable->cellWidget(j, controlCol))
            musicTable->cellWidget(j, controlCol)->setProperty("row", j);
    }
}

void MainWindow::upSelected()
{
    int sel = musicTable->currentRow();
    if (sel > 0)
        buttonUp(sel);
}

void MainWindow::downSelected()
{
    int sel = musicTable->currentRow();
    if (sel >= 0)
        buttonDown(sel);
}



void MainWindow::showOptions ()
{
    int oldsize = settings.value("rowHeight", DEFAULT_ROW_HEIGHT).toInt();
    int oldiconsize = settings.value("iconsize", 48).toInt();
    OptionDialog* dlg = new OptionDialog (this, settings);
    dlg->exec();
    delete dlg;
    qDebug () << "Options executed";
    if (oldiconsize != settings.value("iconsize", 48).toInt())
    {
        qDebug () << "Iconsize change, gui redraw";
        delete centralWidget();
        setupUi();
    }
    setOrientation ();
    if (headers != settings.value("headers", QStringList ()).toStringList()
        || oldsize != settings.value("rowHeight", DEFAULT_ROW_HEIGHT).toInt()
        || oldiconsize != settings.value("iconsize", 48).toInt())
    {
        qDebug () << "Headers, row height or iconsize change: " << headers << " != " << settings.value("headers", QStringList ()).toStringList() << " or " << oldsize << " != " << settings.value("rowHeight", DEFAULT_ROW_HEIGHT).toInt() << " or " << oldiconsize << " != " << settings.value("iconsize", 48).toInt();
        headers = settings.value("headers", QStringList ()).toStringList();
        musicTable->setColumnCount(headers.size ());
        musicTable->setHorizontalHeaderLabels(headers);
        int current = musicTable->currentRow();
        playlistChanged(0);
        musicTable->selectRow(current);
    }
    int curitem = plman.indexOf(mediaObject->currentSource());
    if (curitem >= 0 && isPlaying)
        highlightRow(curitem);
}

void MainWindow::showEqualizer ()
{
    if (eqEffect)
    {
        QList<int> vals;
        QStringList labels;
        foreach(Phonon::EffectParameter ep, eqEffect->parameters())
        {
            vals.push_back(int (eqEffect->parameterValue(ep).toDouble()));
            labels.push_back(ep.name());
        }
        EqualizerDialog* dlg = new EqualizerDialog (this, vals, labels, settings.value("equalizer", false).toBool ());
        connect (dlg, SIGNAL(equalizerChanged(int,int)), this, SLOT(equalizerChanged(int,int)));
        connect (dlg, SIGNAL(toggleEqualizer(bool)), this, SLOT(toggleEqualizer(bool)));
        dlg->exec();
        delete dlg;
    }
    else
    {
        QMessageBox::warning(this, tr ("Equalizer"), tr ("The required equalizer component was not found on your system, sorry!"));
    }
}

void MainWindow::toggleEqualizer(bool val)
{
    settings.setValue("equalizer", val);
    if (val)
        audioPath.insertEffect(eqEffect);
    else
        audioPath.removeEffect (eqEffect);
}


void MainWindow::showAddMenu()
{
    addMenu->popup(cursor().pos());
}

void MainWindow::headphoneEvent(QString what,QString how)
{
//    qDebug () << "HP event: " <<  what << ", " << how;
    if (what != "ButtonPressed" || how != "connection")
        return;
    if (!settings.value("headphoneSense", false).toBool())
        return;
    bool present = QDBusInterface ("org.freedesktop.Hal",
                    "/org/freedesktop/Hal/devices/platform_headphone",
                    "org.freedesktop.Hal.Device", 
                    QDBusConnection::systemBus()).call ("GetProperty", "button.state.value").arguments().at(0).toBool();

    if (mediaObject->state() == Phonon::PlayingState && !present)
    {
        mwa->triggerPause();
        autostopped = true;
    }
    else if (mediaObject->state() == Phonon::PausedState && autostopped && present)
    {
        mwa->triggerPlay();
        autostopped = false;
    }
}

void MainWindow::equalizerChanged (int ind, int val)
{
    if (eqEffect)
    {
        double dval = (double)val;
        QList<Phonon::EffectParameter> pars = eqEffect->parameters();
        if (pars.size() > ind)
        {
            eqEffect->setParameterValue(pars[ind], dval);
            QString valName = QString ("eq_%1").arg(ind);
            settings.setValue(valName, dval);
        }
    }
}

void MainWindow::controlsToggled ()
{
    mwa->controlsToggled(mediaObject->state() == Phonon::PlayingState);
}

