/*
 * Qt C++/QML WebThumbnail example
 * (c) 2010 Dennis Noordsij <dennis.noordsij@movial.com>
 * http://wiki.forum.nokia.com/index.php/Custom_QML_Component:_Website_Thumbnails
 *
 * You may re-use parts or all of this sourcecode in your own projects,
 * including proprietary and/or commercial, without attribution or restriction.
 */

#include "webthumbnailcontext.h"
#include "webthumbnailjob.h"
#include "webthumbnail.h"

WebThumbnailContext::WebThumbnailContext(QDeclarativeItem *parent)
	: QDeclarativeItem(parent),
	  m_thumbnails(),
	  m_queue(),
	  m_activecount(0),
	  m_totalcount(0),
	  m_concurrency(0)
{
}

WebThumbnailContext::~WebThumbnailContext()
{
}

void WebThumbnailContext::bindThumbnailJob(WebThumbnail *wt, WebThumbnailJob *job)
{
	switch(job->getState()) {
		case WebThumbnailJob::Loaded: {
			wt->finished(job->getPixmap());
			break;
		}
		case WebThumbnailJob::Error: {
			wt->error();
			break;
		}
		default: {
			connect(job, SIGNAL(started()), wt, SLOT(started()));
			connect(job, SIGNAL(progress(int)), wt, SLOT(progress(int)));
			connect(job, SIGNAL(finished(const QPixmap&)), wt, SLOT(finished(const QPixmap &)));
			connect(job, SIGNAL(error()), wt, SLOT(error()));
		}
	}
}

void WebThumbnailContext::startRequest(WebThumbnail *wt)
{
	// Do we already have an active/completed job for this URL ?
	foreach(WebThumbnailJob * job, m_thumbnails) {
		if (job->getUrl() == wt->getUrl()) {
			bindThumbnailJob(wt, job);
			return;
		}
	}
	// Do we already have a queued job for this URL ?
	foreach(WebThumbnailJob * job, m_queue) {
		if (job->getUrl() == wt->getUrl()) {
			bindThumbnailJob(wt, job);
			return;
		}
	}
	// Create new job for this URL
	WebThumbnailJob * job = new WebThumbnailJob(wt->getUrl());
	bindThumbnailJob(wt, job);
	m_queue.append(job);
	process();
}

void WebThumbnailContext::registerThumbnail()
{
	m_totalcount++;
	emit thumbnailCountChanged();
}

void WebThumbnailContext::unregisterThumbnail()
{
	m_totalcount--;
	emit thumbnailCountChanged();
}

void WebThumbnailContext::process()
{
	while (m_queue.count() > 0 && (m_activecount < m_concurrency || m_concurrency == 0)) {
		WebThumbnailJob * job = m_queue.takeFirst();
		connect(job, SIGNAL(jobDone(WebThumbnailJob *)), this, SLOT(jobDone(WebThumbnailJob *)));
		m_thumbnails.append(job);
		m_activecount++;
		emit activeChanged();
		// No plugins, no javascript
		job->start(false, false, 256, 256);
	}
}

void WebThumbnailContext::jobDone(WebThumbnailJob * job)
{
	disconnect(job, SIGNAL(jobDone(WebThumbnailJob *)), this, SLOT(jobDone(WebThumbnailJob *)));
	m_activecount--;
	emit activeChanged();
	process();
}

void WebThumbnailContext::setConcurrency(int c)
{
	if (m_concurrency == c)
		return;
	m_concurrency = c;
	emit concurrencyChanged();
	process();
}

int WebThumbnailContext::getQueued() const
{
	return m_queue.count();
}

int WebThumbnailContext::getActive() const
{
	return m_activecount;
}

int WebThumbnailContext::getThumbnailCount() const
{
	return m_totalcount;
}

int WebThumbnailContext::getConcurrency() const
{
	return m_concurrency;
}


