/*
 * This file is part of TabletBridge
 *
 * Copyright (C) 2007-2008 Frank Banul.
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <hildon/hildon-program.h>
#include <hildon/hildon-note.h>
#include <hildon/hildon-banner.h>
#include <hildon/hildon-entry.h>
#include <gtk/gtk.h>
#include <libosso.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <gdk/gdkkeysyms.h>
#include <gconf/gconf-client.h>
#include <dbus/dbus-glib.h>
#include <device_symbols.h>
#include <glib.h>
#include <glib-object.h>
#include <conic.h>

void dbus_connection_setup_with_g_main (DBusConnection *connection, GMainContext *context);

//#define DEBUG_PRINTF

#ifdef DEBUG_PRINTF
#  define my_printf(...)	printf(__VA_ARGS__)
#else
#  define my_printf(...)
#endif

#define VERSION "1.0"
#define GETTEXT_PACKAGE "tabletbridge"
#define _(String) String
#define DEST_PORT 5555
#define SCAN_PORT 3483
#define UPDATE_RATE 400 /* in ms */
#define SAMPLES_AFTER_EVENT 5

#define GCONF_KEY_PREFIX "/apps/maemo/tabletbridge"
#define GCONF_KEY_NAME_ONE   GCONF_KEY_PREFIX"/name_one"
#define GCONF_KEY_IP_ONE     GCONF_KEY_PREFIX"/ip_one"
#define GCONF_KEY_NAME_TWO   GCONF_KEY_PREFIX"/name_two"
#define GCONF_KEY_IP_TWO     GCONF_KEY_PREFIX"/ip_two"
#define GCONF_KEY_NAME_THREE GCONF_KEY_PREFIX"/name_three"
#define GCONF_KEY_IP_THREE   GCONF_KEY_PREFIX"/ip_three"
#define GCONF_KEY_NAME_FOUR  GCONF_KEY_PREFIX"/name_four"
#define GCONF_KEY_IP_FOUR    GCONF_KEY_PREFIX"/ip_four"
#define MRU_NUM 4
#define UNUSED_TEXT "-unused-"
const gchar *name_list[MRU_NUM] = {GCONF_KEY_NAME_ONE, GCONF_KEY_NAME_TWO,
                                   GCONF_KEY_NAME_THREE, GCONF_KEY_NAME_FOUR};
const gchar *ip_list[MRU_NUM] = {GCONF_KEY_IP_ONE, GCONF_KEY_IP_TWO,
                                 GCONF_KEY_IP_THREE, GCONF_KEY_IP_FOUR};


/** The main OSSO context of the application. */
static osso_context_t *_osso = NULL;
static HildonProgram *program;
static GtkWidget *window;
static GtkWidget *screen;
static GtkWidget *frame;
static GdkPixbuf *pixbuf;

struct DisplayProperties {
	int bytes_length;
	int width;
	int height;
	int offset;
	int horiz_scale;
	int vertical_scale;
};

#define SMALL_VFD_BYTE_SIZE 280*16/8
#define SMALL_TALL_VFD_BYTE_SIZE 280*32/8
#define LARGE_VFD_BYTE_SIZE 512*32/8

#define SMALL_VFD_IDX 0
#define SMALL_TALL_VFD_IDX 1
#define LARGE_VFD_IDX 2
static struct DisplayProperties displays[] = {
	{SMALL_VFD_BYTE_SIZE, 280, 16, 33, 2, 3},
	{SMALL_TALL_VFD_BYTE_SIZE, 280, 32, 34, 2, 3},
	{LARGE_VFD_BYTE_SIZE, 512, 32, 34, 1, 2}
};
static int display_idx = SMALL_VFD_IDX;

/* Menu items for the "Connect" submenu. */
static GtkWidget *_menu_connect_new_item = NULL;
static GtkWidget *_menu_connect_scan_item = NULL;
static GtkWidget *_menu_connect_sb_one_item = NULL;
static GtkWidget *_menu_connect_sb_two_item = NULL;
static GtkWidget *_menu_connect_sb_three_item = NULL;
static GtkWidget *_menu_connect_sb_four_item = NULL;
static GtkWidget **menu_list[MRU_NUM] = {&_menu_connect_sb_one_item,
                                        &_menu_connect_sb_two_item,
                                        &_menu_connect_sb_three_item,
                                        &_menu_connect_sb_four_item};
/* connection variables */
static gboolean connected = FALSE, has_display=FALSE;
static int sockfd;
/* packet size needs to be increased for the Largest display packet response
 * 512 * 32 / 8 = 2048 bytes * 2 = 4096 characters
 * 34 character offset + 3 for additional /r/n characters
 * total 4133, this could be calculated depending on the screen size to allow for
 * easy additions of new screens */
static char packet[4352]; // rounded to a 256 byte boundary
static unsigned int activity_counter;

/** VARIABLES PERTAINING TO THE INTERNET CONNECTION. */
enum {
  NOT_WAITING,
  WAITING_FOR_SB,
  WAITING_TO_SCAN
};
static gboolean _iap_connected = FALSE;
static gboolean _iap_connecting = FALSE;
static gint waiting_reason = NOT_WAITING;
static gint sb_entry = 0;
static DBusConnection *system_dbus;
static GMainLoop *main_loop;
static ConIcConnection *connection;

/* scanning variables */
#define SCAN_RATE 1000
typedef struct _ScanInfo ScanInfo;
struct _ScanInfo {
    GtkWidget *scan_dialog;
    GtkListStore *store;
    unsigned int progress;
    gboolean currently_scanning;
    int sockfd;
};
static ScanInfo scan_info = {0};

/**
 * Pop up a modal dialog box with simple error information in it.
 */
static void
popup_error(GtkWidget *window, const gchar *error)
{
    GtkWidget *dialog;
    my_printf("%s(\"%s\")\n", __PRETTY_FUNCTION__, error);

    dialog = GTK_WIDGET(hildon_note_new_information(GTK_WINDOW(window), error));

    gtk_dialog_run(GTK_DIALOG(dialog));
    gtk_widget_destroy(dialog);

    my_printf("%s(): return\n", __PRETTY_FUNCTION__);
}

static void update_pixbuf(char *display)
{
    int rowstride, n_channels, i, j, k, l, m, bit, display_ptr;
    unsigned int nibble;
    guchar *pixels, *p, red, green, blue;
    int horiz_scale, vertical_scale;
    
    my_printf("%s()\n", __PRETTY_FUNCTION__);

    rowstride = gdk_pixbuf_get_rowstride (pixbuf);
    pixels = gdk_pixbuf_get_pixels (pixbuf);
    n_channels = gdk_pixbuf_get_n_channels (pixbuf);
    horiz_scale = displays[display_idx].horiz_scale;
    vertical_scale = displays[display_idx].vertical_scale;
    
    display_ptr = 0;
    for (i=0; i<displays[display_idx].width; i++) { /* columns */
      for (j=0; j<displays[display_idx].height/4; j++) {
        if (display[display_ptr] <= 0x39)
          nibble = display[display_ptr] - 0x30;
        else
          nibble = display[display_ptr] - 87;

	display_ptr++;
	for (k=0; k<4; k++) {
          bit = nibble & (1<<(3-k));
          if (bit) {
            red = 0x40;
            green = 0xff;
            blue = 0xff;
          } else {
            red = 0x0;
            green = 0x0;
            blue = 0x0;
          }
          /* scale */
          for (l=i*horiz_scale; l<(i*horiz_scale)+horiz_scale; l++) {
            for (m=((j*4)+k)*vertical_scale; m<(((j*4)+k)*vertical_scale)+vertical_scale; m++) {
              p = pixels + m * rowstride + l * n_channels;
              p[0] = red;
              p[1] = green;
              p[2] = blue;
            }
          }
 	} /* for k */
      } /* for j */
    } /* for i */

    gtk_image_set_from_pixbuf((GtkImage *)screen, pixbuf);

    my_printf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
}

static void clear_pixbuf(void)
{
    guchar *pixels;
    gint width, height;

    pixels = gdk_pixbuf_get_pixels(pixbuf);
    width = gdk_pixbuf_get_width(pixbuf);
    height = gdk_pixbuf_get_height(pixbuf);
    memset(pixels, 0, (width*height*3));
    gtk_image_set_from_pixbuf((GtkImage *)screen, pixbuf);
}

static char *get_response(char *search, int offset)
{
    gint ptr, ret;
    gchar *display_ptr;
    gboolean found_start=FALSE, found_end=FALSE;

    my_printf("%s()\n", __PRETTY_FUNCTION__);
    ptr=0;
    memset(packet, 0, sizeof(packet));

    if (connected == FALSE)
      return NULL;

    /* get response */
    do {
      my_printf("sizeof(packet)-ptr=%d\n", sizeof(packet)-ptr);

      ret = recv(sockfd, packet+ptr, sizeof(packet)-ptr, 0);
      if (ret == -1) {
        hildon_banner_show_information(GTK_WIDGET(window), NULL,
                             "get_response recv failed");
        if (sockfd != -1) {
		close(sockfd);
		sockfd = -1;
	}
        return NULL;
      }

      my_printf("ret = %d\n", ret);

      ptr += ret;
      display_ptr = strstr(packet, search);
      if (display_ptr != NULL)
        found_start = TRUE;
      if (found_start == TRUE) {
        if (strstr((display_ptr+offset), "\r\n") != NULL)
          found_end = TRUE;
      }
    } while (((found_start == FALSE) || (found_end == FALSE)) && (ptr < sizeof(packet)));
    if (ptr >= sizeof(packet)) {
      my_printf(packet);
      my_printf("%s(): return NULL\n", __PRETTY_FUNCTION__);
      return NULL;
    }
    my_printf("%s(): return\n", __PRETTY_FUNCTION__);
    return (display_ptr+offset);
}

static gint send_request(char *msg)
{
    gint len, bytes_sent=0;

    my_printf("%s()\n", __PRETTY_FUNCTION__);

    if (connected == TRUE) {
      len = strlen(msg);
      bytes_sent = send(sockfd, msg, len, 0);
      if (bytes_sent == -1) {
        hildon_banner_show_information(GTK_WIDGET(window), NULL,
                             "send_request: send failure");
        if (sockfd != -1) {
		close(sockfd);
		sockfd = -1;
	}
        return -1;
      }
      if (bytes_sent != len) {
        hildon_banner_show_information(GTK_WIDGET(window), NULL,
                             "send_request: send sent less");
        return bytes_sent;
      }
    }
    my_printf("%s(): return %d\n", __PRETTY_FUNCTION__, bytes_sent);
    return bytes_sent;
}

static gboolean query_type(void)
{
    gchar *msg = "ListVisualizers\r\n";
    gint len, bytes_sent = 0, ptr, ret, offset=0, lines_found=0, lines_expected=0;
    gchar *display_ptr, *search = "ListVisualizers: ListResultSize ", *end;
    gboolean found_start=FALSE;
    
    my_printf("%s()\n", __PRETTY_FUNCTION__);
    if (connected == FALSE) {
      return FALSE;
    } else {
      len = strlen(msg);
      bytes_sent = send(sockfd, msg, len, 0);
      if (bytes_sent == -1) {
        hildon_banner_show_information(GTK_WIDGET(window), NULL,
                             "query_type: send failure");
        return FALSE;
      }
      if (bytes_sent != len) {
        hildon_banner_show_information(GTK_WIDGET(window), NULL,
                             "query_type: send sent less");
        return FALSE;
      }
      
      ptr=0;
      memset(packet, 0, sizeof(packet));
    
      /* get response */
      do {
        my_printf("sizeof(packet)-ptr=%d\n", sizeof(packet)-ptr);
      
        ret = recv(sockfd, packet+ptr, sizeof(packet)-ptr, 0);
        if (ret == -1) {
          hildon_banner_show_information(GTK_WIDGET(window), NULL,
                             "query_type: recv failed");
          if (sockfd != -1) {
		  close(sockfd);
		  sockfd = -1;
	  }
          return FALSE;
        }
        my_printf("ret = %d\n", ret);
        
        my_printf(packet);
        
        ptr += ret;
        
        if (found_start == FALSE) {
          display_ptr = strstr(packet, search);
          if (display_ptr != NULL) {
            found_start = TRUE;
            offset = strlen(search) + (gint)(display_ptr - packet);
            my_printf("0 %c\n", packet[offset]);
            lines_expected = packet[offset] - 0x30;
            lines_expected++; /* ListResultEnd */
            offset += 3; /* past \r\n */
          }
        }
        if (found_start == TRUE) {
          while(offset < ptr) {
            my_printf("offset = %d\n", offset);
            end = strstr((packet+offset), "\r\n");
            if (end != NULL) {
              offset += (gint)(end - &packet[offset]) + 3;
              lines_found++;
            } else {
              break;
            }
          }
        }
      } while (((found_start == FALSE) || (lines_found < lines_expected)) &&
               (ptr < sizeof(packet)));
      if (ptr >= sizeof(packet)) {
        my_printf(packet);
        my_printf("%s(): return FALSE\n", __PRETTY_FUNCTION__);
        return FALSE;
      }
      if (lines_found == 4) {
        my_printf("%s(): return FALSE\n", __PRETTY_FUNCTION__);
        return FALSE;
      } else {
        my_printf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
        return TRUE;
      }
    }
    my_printf("%s(): return FALSE\n", __PRETTY_FUNCTION__);
    return FALSE;
}

static gboolean periodic_update(gpointer data)
{
    char *msg = "GetDisplayData\r\n";
    char *screen_data;
    
    my_printf("%s()\n", __PRETTY_FUNCTION__);
    
    if (connected == FALSE) {
      hildon_banner_show_information(GTK_WIDGET(window), NULL,
                             "Not connected");
      return FALSE;
    }
    
    if (has_display == FALSE) {
      return FALSE;
    }
    
    if (scan_info.currently_scanning == TRUE)
      return FALSE;
    
    if ((activity_counter < SAMPLES_AFTER_EVENT) ||
       ((activity_counter % SAMPLES_AFTER_EVENT) == 0)) {
	    /* request a screen update */
	    if (send_request(msg) != strlen(msg)) {
		    hildon_banner_show_information(GTK_WIDGET(window), NULL,
				    "periodic_update: send failed");
		    return FALSE;
	    }
	    
	    screen_data = get_response("GetDisplayData: data bytes:", 
    				displays[display_idx].offset);
	    if (screen_data != NULL) {
		    update_pixbuf(screen_data);
	    } else {
		    my_printf("%s(): return FALSE\n", __PRETTY_FUNCTION__);
		    return FALSE;
	    }
    }
    activity_counter++;
    
    my_printf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

void *get_in_addr(struct sockaddr *sa)
{
	if (sa->sa_family == AF_INET) {
		return &(((struct sockaddr_in*)sa)->sin_addr);
	}

	return &(((struct sockaddr_in6*)sa)->sin6_addr);
}

static gboolean periodic_scan(gpointer data)
{
    char msg[] = "Scanning |\0";
    unsigned char twirl;
    struct sockaddr_storage from;
    socklen_t fromlen;
    int recvlen;
    char buf[64];
    char s[INET6_ADDRSTRLEN];

    my_printf("%s()\n", __PRETTY_FUNCTION__);

    if (scan_info.currently_scanning == FALSE) {
	    if (scan_info.sockfd != -1) {
		    close(scan_info.sockfd);
		    scan_info.sockfd = -1;
	    }
	    return FALSE;
    }

    if (_iap_connected == FALSE) {
      hildon_banner_show_information(GTK_WIDGET(window), NULL,
                             "Not connected");
      return FALSE;
    }

    fromlen = sizeof(from);
    if ((recvlen = recvfrom(scan_info.sockfd, buf, sizeof(buf), 0,
        (struct sockaddr *)&from, &fromlen)) != -1) {
      /* got data from recvfrom */
      GtkTreeIter iter;
      gboolean valid, present = FALSE;

      inet_ntop(from.ss_family, get_in_addr((struct sockaddr *)&from), s, sizeof(s));
      my_printf("got packet from %s\n",s);
      my_printf("packet is %d bytes long\n", recvlen);
      /* Make sure duplicates don't get added */
      valid = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(scan_info.store), &iter);
      while (valid) {
        gchar *name_data, *ip_data;

        gtk_tree_model_get(GTK_TREE_MODEL(scan_info.store), &iter, 0, 
                           &name_data, 1, &ip_data, -1);
        my_printf("name = %s, ip = %s\n", name_data, ip_data);
        if (strcmp(s, ip_data) == 0) {
          present = TRUE;
          g_free(name_data);
          g_free(ip_data);
          break;
        }
        g_free(name_data);
        g_free(ip_data);
        valid = gtk_tree_model_iter_next(GTK_TREE_MODEL(scan_info.store), &iter);
      } /* while (valid) */

      /* It's not a duplicate */
      if (present == FALSE) {
        struct sockaddr_in dest_addr;
        struct timeval tv = {3, 0}; /* 3 sec, 0 usec */
        gchar *friendly_rqst = "GetFriendlyName\r\n";
        gchar *friendly_name;

        /* Get the friendly name */
        /* clear out from previous connections */
        if (sockfd != -1) {
		close(sockfd);
		sockfd = -1;
	}
        clear_pixbuf();
        gtk_frame_set_label(GTK_FRAME(frame),
                            _("Soundbridge Display - Not connected"));
        connected = FALSE;

        /* make a connection */
        sockfd = socket(PF_INET, SOCK_STREAM, 0);

        if (sockfd == -1) {
          hildon_banner_show_information(GTK_WIDGET(window), NULL,
                           "periodic_scan: Unable to get socket");
          goto try_next_sb;
        }

        if (setsockopt(sockfd, SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof(tv)) < 0) {
          hildon_banner_show_information(GTK_WIDGET(window), NULL,
                        "periodic_scan: O_RCVTIMEO setsockopt error");
        }
        if (setsockopt(sockfd, SOL_SOCKET, SO_SNDTIMEO, &tv, sizeof(tv)) < 0) {
          hildon_banner_show_information(GTK_WIDGET(window), NULL,
                        "periodic_scan: SO_SNDTIMEO setsockopt error");
        }

        dest_addr.sin_family = AF_INET;          // host byte order
        dest_addr.sin_port = htons(DEST_PORT);   // short, network byte order
        dest_addr.sin_addr.s_addr = inet_addr(s);
        memset(dest_addr.sin_zero, '\0', sizeof dest_addr.sin_zero);

        if (connect(sockfd, (struct sockaddr *)&dest_addr, sizeof dest_addr) == -1) {
          hildon_banner_show_information(GTK_WIDGET(window), NULL,
                           "periodic_scan: Unable to connect");
          goto try_next_sb;
        }
        connected = TRUE;
        if (send_request(friendly_rqst) != strlen(friendly_rqst)) {
          hildon_banner_show_information(GTK_WIDGET(window), NULL,
                             "periodic_scan: send failed");
          goto try_next_sb;
        }
        friendly_name = get_response("GetFriendlyName:", 17);
        if (friendly_name != NULL) {
          friendly_name[strlen(friendly_name)-2] = '\0';

          gtk_list_store_append(scan_info.store, &iter);
          gtk_list_store_set(scan_info.store, &iter,
                0, g_strdup(friendly_name),
                1, g_strdup(s),
                -1);
        }
try_next_sb:
        connected = FALSE;
        if (sockfd != -1) {
		close(sockfd);
		sockfd = -1;
	}
      } /* if (present == FALSE) */

    } /* got data from recvfrom */

    switch (scan_info.progress%4) {
      case 0:
        twirl = '|';
        break;
      case 1:
        twirl = '/';
        break;
      case 2:
        twirl = '-';
        break;
      case 3:
        twirl = '\\';
        break;
      default:
        twirl = '!';
        break;
    }
    scan_info.progress++;

    sprintf(msg, "Scanning %c", twirl);
    gtk_window_set_title(GTK_WINDOW(scan_info.scan_dialog), msg);

    my_printf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

/* keyboard callback */
static gboolean
window_cb_key_press(GtkWidget* widget, GdkEventKey *event)
{
    gchar *command_response;
    
    my_printf("%s()\n", __PRETTY_FUNCTION__);

    if (connected == TRUE) {
      switch(event->keyval)
      {
          case GDK_Up:
              send_request("IrDispatchCommand CK_NORTH\r\n");
              break;
          case GDK_Down:
              send_request("IrDispatchCommand CK_SOUTH\r\n");
              break;
          case GDK_Left:
              send_request("IrDispatchCommand CK_WEST\r\n");
              break;
          case GDK_Right:
              send_request("IrDispatchCommand CK_EAST\r\n");
              break;
          case GDK_Return:
	  case GDK_KP_Enter:
              send_request("IrDispatchCommand CK_SELECT\r\n");
              break;
          case GDK_F7:
              send_request("IrDispatchCommand CK_VOLUME_UP\r\n");
              break;
          case GDK_F8:
              send_request("IrDispatchCommand CK_VOLUME_DOWN\r\n");
              break;
          case GDK_F6:
	  case GDK_M:
	  case GDK_m:
              send_request("IrDispatchCommand CK_MENU\r\n");
              break;
          case GDK_Escape:
	  case GDK_BackSpace:
              send_request("IrDispatchCommand CK_EXIT\r\n");
              break;
          default:
              my_printf("%s(): return FALSE\n", __PRETTY_FUNCTION__);
              return FALSE;
      }
    
      /* get command response */
      command_response = get_response("IrDispatchCommand: OK", 0);
    
      activity_counter = 0;
    } /* if (connected == TRUE) */
    my_printf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

static int
get_display_byte_length()
{
    int bytes_length, ptr, ret;
    
    my_printf("%s()\n", __PRETTY_FUNCTION__);
    ptr=0;
    memset(packet, 0, sizeof(packet));
    
    if (connected == FALSE)
      return -1;
    
    do {
      my_printf("sizeof(packet)-ptr=%d\n", sizeof(packet)-ptr);
      
      ret = recv(sockfd, packet, sizeof(packet), MSG_PEEK);
      if (ret == -1) {
        hildon_banner_show_information(GTK_WIDGET(window), NULL,
                             "get_response recv failed");
        if (sockfd != -1) {
		close(sockfd);
		sockfd = -1;
	}
        return -1;
      }
      
      ptr = ret;

    } while ((strstr(packet, "GetDisplayData: data bytes:") == NULL)
    		&& (strstr(packet, "\n" ) == NULL)
    		&& (ptr < sizeof(packet)));
    /* this condition is not ideal, using the ptr check to ensure that the system doesn't
     * loop indefinately while waiting for response, however, it's possible that the
     * loop times out before enough of the message is recieved.
     * It is possible that this could be done in get_response() but that then ties the
     * function into GetDisplayData, also the general idea here was to allow packet to be
     * malloc'd for a suitable size ahead of actually reading the message */
    
    my_printf("%s\n", packet + strlen("GetDisplayData: data bytes: "));
    bytes_length = (int) strtol(packet + strlen("GetDisplayData: data bytes: "), NULL, 0);

    my_printf("%s(%i): return\n", __PRETTY_FUNCTION__, bytes_length);
    
    return bytes_length;
}

static void
setup_display_properties(int bytes_length) {

    int new_display_idx = -1, i;
    my_printf("%s(%i)\n", __PRETTY_FUNCTION__, bytes_length);

    for (i=0; i < sizeof displays / sizeof displays[0]; i++)
        if (displays[i].bytes_length == bytes_length)
            new_display_idx = i;

    if (new_display_idx != -1 && display_idx != new_display_idx) {
        my_printf("changing display\n");
        display_idx = new_display_idx;

        g_object_unref(G_OBJECT(pixbuf));

        pixbuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB, FALSE, 8,
                            displays[display_idx].width*displays[display_idx].horiz_scale,
                            displays[display_idx].height*displays[display_idx].vertical_scale);
    }
    my_printf("%s(%i): return\n", __PRETTY_FUNCTION__, bytes_length);
}

static gboolean
connect_to_soundbridge(const gchar *ip_addr)
{
    struct sockaddr_in dest_addr;   // will hold the destination addr
    gchar *msg = "GetDisplayData\r\n";
    gchar *screen_data;
    struct timeval tv = {3, 0}; /* 3 sec, 0 usec */

    my_printf("%s()\n", __PRETTY_FUNCTION__);

    if(!_iap_connected && !_iap_connecting) {
       _iap_connecting = TRUE;
       //osso_iap_connect(OSSO_IAP_ANY, OSSO_IAP_REQUESTED_CONNECT, NULL);
       con_ic_connection_connect(connection, CON_IC_CONNECT_FLAG_NONE);
       hildon_banner_show_information(GTK_WIDGET(window), NULL,
                             "Not connected to network");
      return FALSE;
    }

    /* clear out from previous connections */
    if (sockfd != -1) {
	    close(sockfd);
	    sockfd = -1;
    }
    clear_pixbuf();
    gtk_frame_set_label(GTK_FRAME(frame), _("Soundbridge Display - Not connected"));

    /* make a connection */
    sockfd = socket(PF_INET, SOCK_STREAM, 0);

    if (sockfd == -1) {
      hildon_banner_show_information(GTK_WIDGET(window), NULL,
                             "connect_to_soundbridge: Unable to get socket");
      return FALSE;
    }

    if (setsockopt(sockfd, SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof(tv)) < 0) {
      hildon_banner_show_information(GTK_WIDGET(window), NULL,
                                     "SO_RCVTIMEO setsockopt error");
    }
    if (setsockopt(sockfd, SOL_SOCKET, SO_SNDTIMEO, &tv, sizeof(tv)) < 0) {
      hildon_banner_show_information(GTK_WIDGET(window), NULL,
                                     "SO_SNDTIMEO setsockopt error");
    }

    dest_addr.sin_family = AF_INET;          // host byte order
    dest_addr.sin_port = htons(DEST_PORT);   // short, network byte order
    dest_addr.sin_addr.s_addr = inet_addr(ip_addr);
    memset(dest_addr.sin_zero, '\0', sizeof dest_addr.sin_zero);

    if (connect(sockfd, (struct sockaddr *)&dest_addr, sizeof dest_addr) == -1) {
      hildon_banner_show_information(GTK_WIDGET(window), NULL,
                             "connect_to_soundbridge: Unable to connect");
      if (sockfd != -1) {
        close(sockfd);
	sockfd = -1;
      }
      return FALSE;
    }

    connected = TRUE;
    activity_counter = 0;

    has_display = query_type();

    if (has_display) {
      /* request a screen update */
      if (send_request(msg) != strlen(msg)) {
        hildon_banner_show_information(GTK_WIDGET(window), NULL,
                             "connect_to_soundbridge: send failed");
        return FALSE;
      }

      setup_display_properties(get_display_byte_length());
      #if 0
      /* setup_display_properties eats the screen data, request it again */
      if (send_request(msg) != strlen(msg)) {
        hildon_banner_show_information(GTK_WIDGET(window), NULL,
                             "connect_to_soundbridge: send failed");
        return FALSE;
      }
#endif
      screen_data = get_response("GetDisplayData: data bytes:", 
      					displays[display_idx].offset);
      if (screen_data != NULL) {

        my_printf(packet);

        update_pixbuf(screen_data);
      }

      /* add a polling function */
      g_timeout_add(UPDATE_RATE, periodic_update, NULL);
    } else {
      hildon_banner_show_information(GTK_WIDGET(window), NULL,
                             "No display data available, remote control only");
    }

    my_printf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

/* move passed entry to top of MRU list, moving others down */
static void
move_in_menu(gchar *name, gchar *ip_addr, const gint entry)
{
    gchar *tmp_name[MRU_NUM] = {NULL, NULL, NULL, NULL},
          *tmp_ip[MRU_NUM] = {NULL, NULL, NULL, NULL},
          *test_name, *test_addr;
    gint tmp_entry=0, old_entry=0;
    GConfClient *gconf_client = gconf_client_get_default();

    my_printf("%s()\n", __PRETTY_FUNCTION__);
    /* remove it */
    gconf_client_unset(gconf_client, name_list[entry], NULL);
    gconf_client_unset(gconf_client, ip_list[entry], NULL);
    tmp_name[tmp_entry] = name;
    tmp_ip[tmp_entry] = ip_addr;
    tmp_entry++;
    while ((tmp_entry < MRU_NUM) && (old_entry < MRU_NUM)) {
      test_name = gconf_client_get_string(gconf_client, name_list[old_entry], NULL);
      test_addr = gconf_client_get_string(gconf_client, ip_list[old_entry], NULL);
      if (test_name) {
        tmp_name[tmp_entry] = test_name;
        tmp_ip[tmp_entry] = test_addr;
        tmp_entry++;
      }
      old_entry++;
    }
    /* now add to menu and saved config */
    tmp_entry = 0;
    while (tmp_entry < MRU_NUM) {
      if (tmp_name[tmp_entry]) {
        gconf_client_set_string(gconf_client, name_list[tmp_entry],
                              tmp_name[tmp_entry], NULL);
        gconf_client_set_string(gconf_client, ip_list[tmp_entry],
                              tmp_ip[tmp_entry], NULL);
        gtk_button_set_label(GTK_BUTTON(*menu_list[tmp_entry]), tmp_name[tmp_entry]);
        tmp_entry++;
      } else {
        break;
      }
    }
    my_printf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
}

static gboolean
add_to_menu(const gchar *name, const gchar *ip_addr)
{
    gboolean found_in_list=FALSE, search_more=TRUE;
    GConfClient *gconf_client = gconf_client_get_default();
    gchar *test_name, *test_addr;
    gint entry=0;
    
    my_printf("%s()\n", __PRETTY_FUNCTION__);

    /* If not in list, add to 1st entry and move any others down, dropping */
    /* the last entry. If found in the list, then move to 1st, move others */
    /* down. */
    
    /* search existing entries */
    do {
      my_printf("entry=%d\n", entry);
      test_name = gconf_client_get_string(gconf_client, name_list[entry], NULL);
      test_addr = gconf_client_get_string(gconf_client, ip_list[entry], NULL);
      if ((test_name) && (test_addr)) {
        if ((strcmp(name, test_name) == 0) && (strcmp(ip_addr, test_addr) == 0)) {
          found_in_list = TRUE;
          search_more = FALSE;
        }
      } else {
        found_in_list = FALSE;
        search_more = FALSE;
      }
    } while ((found_in_list == FALSE) && (search_more == TRUE) &&
             (++entry < MRU_NUM));
    
    if (found_in_list == TRUE)
      my_printf("found_in_list\n");
    if (search_more == TRUE)
      my_printf("search_more\n");
      
    if (found_in_list == FALSE) {
      gchar *tmp_name, *tmp_ip;
      
      for (entry=(MRU_NUM)-1; entry>0; --entry) {
        tmp_name = gconf_client_get_string(gconf_client, name_list[entry-1], NULL);
        tmp_ip = gconf_client_get_string(gconf_client, ip_list[entry-1], NULL);
        if ((tmp_name) && (tmp_ip)) {
          gconf_client_set_string(gconf_client, name_list[entry], tmp_name, NULL);
          gconf_client_set_string(gconf_client, ip_list[entry], tmp_ip, NULL);
          gtk_button_set_label(GTK_BUTTON(*menu_list[entry]), tmp_name);
        }
      }
      /* add to saved configuration */
      gconf_client_set_string(gconf_client, name_list[0], name, NULL);
      gconf_client_set_string(gconf_client, ip_list[0], ip_addr, NULL);
      gtk_button_set_label(GTK_BUTTON(*menu_list[0]), name);
    } else { /* found_in_list == TRUE */
      move_in_menu(test_name, test_addr, entry);
    }
    
    my_printf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

/* menu callbacks */
static gboolean
menu_cb_connect_new(GtkAction *action)
{
    GtkWidget *dialog, *device_label, *device_name, *ip_label,
              *ip_value;
    gboolean connect_status = FALSE;
    gint name_len, ip_len;
    const gchar *sb_name, *ip_addr;
    
    my_printf("%s()\n", __PRETTY_FUNCTION__);
   
    has_display = FALSE;

    /* Create the widgets */
    dialog = gtk_dialog_new_with_buttons(_("Add New"),
                                         GTK_WINDOW(window),
                                         GTK_DIALOG_MODAL,
                                         GTK_STOCK_OK,
                                         GTK_RESPONSE_OK,
                                         GTK_STOCK_CANCEL,
                                         GTK_RESPONSE_CANCEL,
                                         NULL);
    device_label = gtk_label_new (_("Soundbridge Name"));
    device_name = hildon_entry_new (HILDON_SIZE_AUTO);
    ip_label = gtk_label_new(_("Soundbridge IP Address"));
    ip_value = hildon_entry_new (HILDON_SIZE_AUTO);
    /* Add the label, and show everything we've added to the dialog. */
    gtk_container_add(GTK_CONTAINER (GTK_DIALOG(dialog)->vbox), device_label);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), device_name);
    gtk_container_add(GTK_CONTAINER (GTK_DIALOG(dialog)->vbox), ip_label);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), ip_value);   
    gtk_widget_show_all(dialog);
    gint result = gtk_dialog_run (GTK_DIALOG(dialog));
    switch (result) {
      case GTK_RESPONSE_OK:
	sb_name = hildon_entry_get_text(HILDON_ENTRY(device_name));
        ip_addr = hildon_entry_get_text(HILDON_ENTRY(ip_value));
        name_len = strlen(sb_name);
        ip_len = strlen(ip_addr);
    
        if ((sb_name > 0) && (ip_len > 0)) {
          connect_status = connect_to_soundbridge(ip_addr);
          if (connect_status == TRUE) {
            add_to_menu(sb_name, ip_addr);
            gtk_frame_set_label(GTK_FRAME(frame), sb_name);
          }
        } else {
          gtk_widget_destroy(dialog);
          if (name_len == 0)
            popup_error(window, "Soundbridge name invalid");
          if (ip_len == 0)
            popup_error(window, "IP Address invalid");
        }

        if (connect_status == FALSE) {
	  char *err_msg = (char *)malloc(strlen(ip_addr) + 25);
          if (err_msg != NULL) {
	    if (ip_len > 0) {
              sprintf(err_msg, "Unable to connect to %s", ip_addr);
              gtk_widget_destroy(dialog);
              popup_error(window, err_msg);
	    }
            free(err_msg);
          } else {
            gtk_widget_destroy(dialog);
            popup_error(window, "Unable to connect to Soundbridge");
          }
        } else {
          gtk_widget_destroy(dialog);
        }
        break;
      default:
        gtk_widget_destroy(dialog);
        break;
    }

    my_printf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

static gboolean
menu_cb_connect_scan()
{

    struct timeval tv = {0, 1000}; /* 0 sec, 1000 usec */
    struct sockaddr_in my_addr;    // my address information
    GtkWidget *lst_devices = NULL;
    GtkCellRenderer *renderer = NULL;
    GtkTreeViewColumn *column = NULL;
    GtkWidget *dialog = NULL;
    gchar *name, *ip;
    gboolean device_selected = FALSE;

    my_printf("%s()\n", __PRETTY_FUNCTION__);

    if (!_iap_connected) {
      _iap_connecting = TRUE;
      //osso_iap_connect(OSSO_IAP_ANY, OSSO_IAP_REQUESTED_CONNECT, NULL);
      con_ic_connection_connect(connection, CON_IC_CONNECT_FLAG_NONE);
      waiting_reason = WAITING_TO_SCAN;
      gtk_frame_set_label(GTK_FRAME(frame), _("Waiting for connection to scan"));
      return TRUE;
    }

    /* make a socket */
    scan_info.sockfd = socket(PF_INET, SOCK_DGRAM, 0);

    if (scan_info.sockfd == -1) {
      hildon_banner_show_information(GTK_WIDGET(window), NULL,
                             "menu_cb_connect_scan: Unable to get socket");
      return FALSE;
    }

    if (setsockopt(scan_info.sockfd, SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof(tv)) < 0) {
      hildon_banner_show_information(GTK_WIDGET(window), NULL,
                                     "SO_RCVTIMEO setsockopt error");
    }

    my_addr.sin_family = AF_INET;         // host byte order
    my_addr.sin_port = htons(SCAN_PORT);  // short, network byte order
    my_addr.sin_addr.s_addr = INADDR_ANY; // automatically fill with my IP
    memset(my_addr.sin_zero, '\0', sizeof my_addr.sin_zero);

    if (bind(scan_info.sockfd, (struct sockaddr *)&my_addr, sizeof(my_addr)) == -1) {
        hildon_banner_show_information(GTK_WIDGET(window), NULL,
          "menu_cb_connect_scan: unable to bind");
        return FALSE;
    }


    /* Create the scan dialog */
    dialog = gtk_dialog_new_with_buttons(_("Scan for Soundbridges"),
                                         GTK_WINDOW(window),
                                         GTK_DIALOG_MODAL,
                                         GTK_STOCK_OK,
                                         GTK_RESPONSE_OK,
                                         GTK_STOCK_CANCEL,
                                         GTK_RESPONSE_CANCEL,
                                         NULL);

    scan_info.scan_dialog = dialog;

    gtk_window_set_default_size(GTK_WINDOW(dialog), 500, 300);

    scan_info.store = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_STRING);

    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),
            lst_devices = gtk_tree_view_new_with_model(
                GTK_TREE_MODEL(scan_info.store)), TRUE, TRUE, 0);

    g_object_unref(G_OBJECT(scan_info.store));

    gtk_tree_selection_set_mode(
            gtk_tree_view_get_selection(GTK_TREE_VIEW(lst_devices)),
            GTK_SELECTION_SINGLE);
    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(lst_devices), TRUE);

    renderer = gtk_cell_renderer_text_new();
    column = gtk_tree_view_column_new_with_attributes(
            _("Name"), renderer, "text", 0, NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(lst_devices), column);

    renderer = gtk_cell_renderer_text_new();
    column = gtk_tree_view_column_new_with_attributes(
            _("IP Address"), renderer, "text", 1, NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(lst_devices), column);

    gtk_widget_show_all(dialog);

    /* add a polling function */
    g_timeout_add(SCAN_RATE, periodic_scan, NULL);

    scan_info.currently_scanning = TRUE;

    while (GTK_RESPONSE_OK == gtk_dialog_run(GTK_DIALOG(dialog)))
    {
        GtkTreeIter iter;
        if(gtk_tree_selection_get_selected(
                    gtk_tree_view_get_selection(GTK_TREE_VIEW(lst_devices)),
                    NULL, &iter))
        {
            gtk_tree_model_get(GTK_TREE_MODEL(scan_info.store),
                    &iter, 0, &name, 1, &ip, -1);
            device_selected = TRUE;
            break;
        }
        else
            popup_error(dialog,
                    _("Please select a device from the list."));
    }
    gtk_widget_destroy(dialog);

    scan_info.currently_scanning = FALSE;
    if (scan_info.sockfd != -1) {
	    close(scan_info.sockfd);
	    scan_info.sockfd = -1;
    }

    if (device_selected == TRUE) {
      if (connect_to_soundbridge(ip)) {
        add_to_menu(name, ip);
        gtk_frame_set_label(GTK_FRAME(frame), name);
      }
    }
    my_printf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

static gboolean
menu_cb_connect(gint entry)
{
    GtkWidget *dialog = NULL, *prompt;
    GConfClient *gconf_client = gconf_client_get_default();
    gchar *ip_addr=NULL, *name=NULL;

    my_printf("%s()\n", __PRETTY_FUNCTION__);

    if (entry < MRU_NUM)
      ip_addr = gconf_client_get_string(gconf_client, ip_list[entry], NULL);

    if (ip_addr) {
      if (connected == TRUE) {
        connected = FALSE;
        if (sockfd != -1) {
		close(sockfd);
		sockfd = -1;
	}
      }

      if (!_iap_connected) {
        gchar *prefix = "Waiting for connection to ", *frame_msg;

        _iap_connecting = TRUE;
        //osso_iap_connect(OSSO_IAP_ANY, OSSO_IAP_REQUESTED_CONNECT, NULL);
        con_ic_connection_connect(connection, CON_IC_CONNECT_FLAG_NONE);
        waiting_reason = WAITING_FOR_SB;
        sb_entry = entry;
        name = gconf_client_get_string(gconf_client, name_list[entry], NULL);
        frame_msg = (gchar *)malloc(strlen(name) + strlen(prefix) + 1);
        if (frame_msg) {
          sprintf(frame_msg, "%s%s", prefix, name);
          gtk_frame_set_label(GTK_FRAME(frame), frame_msg);
          free(frame_msg);
        } else {
          gtk_frame_set_label(GTK_FRAME(frame), _("Waiting for connection"));
        }
      }
      else if (connect_to_soundbridge(ip_addr) == FALSE) {
      /* ask if user wants to remove entry */
        dialog = gtk_dialog_new_with_buttons(_("Remove?"),
                                             GTK_WINDOW(window),
                                             GTK_DIALOG_MODAL,
                                             GTK_STOCK_YES,
                                             GTK_RESPONSE_YES,
                                             GTK_STOCK_NO,
                                             GTK_RESPONSE_NO,
                                             NULL);
        prompt = gtk_label_new (_("Connection failed - Remove from list?"));
        gtk_container_add(GTK_CONTAINER (GTK_DIALOG(dialog)->vbox), prompt);
        gtk_widget_show_all(dialog);
        gint result = gtk_dialog_run (GTK_DIALOG(dialog));
        switch (result) {

          case GTK_RESPONSE_YES:
            while ((entry+1) < MRU_NUM) {
              gchar *tmp_name, *tmp_addr;

              tmp_name = gconf_client_get_string(gconf_client, name_list[entry+1], NULL);
              if (tmp_name) {
                tmp_addr = gconf_client_get_string(gconf_client, ip_list[entry+1], NULL);
                gconf_client_set_string(gconf_client, name_list[entry], tmp_name, NULL);
                gconf_client_set_string(gconf_client, ip_list[entry], tmp_addr, NULL);
                gtk_button_set_label(GTK_BUTTON(*menu_list[entry]), tmp_name);
                gconf_client_unset(gconf_client, ip_list[entry+1], NULL);
                gconf_client_unset(gconf_client, name_list[entry+1], NULL);
                gtk_button_set_label(GTK_BUTTON(*menu_list[entry+1]), UNUSED_TEXT);
                entry++;
              }
              else {
                gconf_client_unset(gconf_client, name_list[entry], NULL);
                gconf_client_unset(gconf_client, ip_list[entry], NULL);
                gtk_button_set_label(GTK_BUTTON(*menu_list[entry]), UNUSED_TEXT);
                break;
              }
            }
            if (entry == MRU_NUM-1) {
              gconf_client_unset(gconf_client, name_list[entry], NULL);
              gconf_client_unset(gconf_client, ip_list[entry], NULL);
              gtk_button_set_label(GTK_BUTTON(*menu_list[entry]), UNUSED_TEXT);
            }
            break;
          default:
            /* move recently used menu item to top */
            name = gconf_client_get_string(gconf_client, name_list[entry], NULL);
            move_in_menu(name, ip_addr, entry);
            break;
        }
        gtk_widget_destroy (dialog);
      } else { /* connect_to_soundbridge(ip_addr) == TRUE */
        /* move recently used menu item to top */
        name = gconf_client_get_string(gconf_client, name_list[entry], NULL);
        gtk_frame_set_label(GTK_FRAME(frame), name);
        move_in_menu(name, ip_addr, entry);
      }
    }

    my_printf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

static gboolean
main_screen_button_cb(GtkWidget *widget, gchar *command)
{
    gchar *command_response;
    
    my_printf("%s()\n", __PRETTY_FUNCTION__);
    
    my_printf("%s", command);
    if (connected == TRUE) {
      send_request(command);
    
      /* get command response */
      command_response = get_response("IrDispatchCommand: OK", 0);
    
      activity_counter = 0;
    } /* if (connected == TRUE) */
    my_printf("%s(): return\n", __PRETTY_FUNCTION__);
    return TRUE;
}

static gboolean
menu_cb_connect_sb(GtkAction *action, gint i)
{
    my_printf("%s()\n", __PRETTY_FUNCTION__);
    has_display = FALSE;
    my_printf("%d\n", i);
    menu_cb_connect(i);
    my_printf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;    
}

/**
 * Create the menu items needed for the drop down menu.
 */
static void
menu_init()
{
    /* Create needed handles. */
    HildonAppMenu *menu;
    GConfClient *gconf_client = gconf_client_get_default();
    gchar *sb_name;
    gint i;

    my_printf("%s()\n", __PRETTY_FUNCTION__);

    /* Get the menu of our view. */
    menu = HILDON_APP_MENU (hildon_app_menu_new ());

    /* Create the menu items. */

    /* The "Connect" submenu. */
    _menu_connect_new_item = gtk_button_new_with_label("New...");
    hildon_app_menu_append(menu, GTK_BUTTON(_menu_connect_new_item));

    _menu_connect_scan_item = gtk_button_new_with_label("Scan...");
    hildon_app_menu_append(menu, GTK_BUTTON(_menu_connect_scan_item));

    /* clear menu for testing */
    //for (i=0; i<MRU_NUM; i++)
	//    gconf_client_unset(gconf_client, name_list[i], NULL);
    
    /* get saved soundbridge devices */
    for (i=0; i<MRU_NUM; i++) {
      sb_name = gconf_client_get_string(gconf_client, name_list[i], NULL);
      if (sb_name)
	*menu_list[i] = gtk_button_new_with_label(sb_name);
      else
	*menu_list[i] = gtk_button_new_with_label(_(UNUSED_TEXT));
      hildon_app_menu_append(menu, GTK_BUTTON(*menu_list[i]));
    }
    
    /* We need to show menu items. */
    gtk_widget_show_all(GTK_WIDGET(menu));

    hildon_window_set_app_menu(HILDON_WINDOW(window), menu);

    /* Connect the signals. */
    g_signal_connect_after(_menu_connect_new_item, "clicked", G_CALLBACK (menu_cb_connect_new), NULL);
    g_signal_connect_after(_menu_connect_scan_item, "clicked", G_CALLBACK (menu_cb_connect_scan), NULL);
    
    for (i=0; i<MRU_NUM; i++)
      g_signal_connect_after(*menu_list[i], "clicked", G_CALLBACK (menu_cb_connect_sb), (gint *)i);

    my_printf("%s(): return\n", __PRETTY_FUNCTION__);
}


static gint
dbus_cb_default(const gchar *interface, const gchar *method,
        GArray *arguments, gpointer data, osso_rpc_t *retval)
{
    my_printf("%s()\n", __PRETTY_FUNCTION__);

    retval->type = DBUS_TYPE_INVALID;

    my_printf("%s(): return\n", __PRETTY_FUNCTION__);
    return OSSO_OK;
}

static void my_connection_cb(ConIcConnection *connection,
                             ConIcConnectionEvent *event,
                             gpointer user_data)
{
    ConIcConnectionStatus status = con_ic_connection_event_get_status(event);

    my_printf("%s()\n", __PRETTY_FUNCTION__);

    _iap_connecting = FALSE;

    if(status == CON_IC_STATUS_CONNECTED && !_iap_connected) {
      _iap_connected = TRUE;
      if (waiting_reason == WAITING_FOR_SB) {
        gchar *ip_addr=NULL, *name=NULL;
        GConfClient *gconf_client = gconf_client_get_default();

        waiting_reason = NOT_WAITING;
        ip_addr = gconf_client_get_string(gconf_client, ip_list[sb_entry], NULL);
        if (ip_addr) {
          if (connect_to_soundbridge(ip_addr) == TRUE) {
            name = gconf_client_get_string(gconf_client, name_list[sb_entry], NULL);
            gtk_frame_set_label(GTK_FRAME(frame), name);
            move_in_menu(name, ip_addr, sb_entry);
          } else {
            gtk_frame_set_label(GTK_FRAME(frame), _("Soundbridge Display - Not connected"));
          }
        } else {
          gtk_frame_set_label(GTK_FRAME(frame), _("Soundbridge Display - Not connected"));
        }
      }
      else if (waiting_reason == WAITING_TO_SCAN) {
        waiting_reason = NOT_WAITING;
        menu_cb_connect_scan();
      }
      my_printf("iap_connected\n");
    }

    if (((status == CON_IC_STATUS_DISCONNECTING) || (status == CON_IC_STATUS_DISCONNECTED))
         && _iap_connected) {
      _iap_connected = FALSE;
      connected = FALSE;
      waiting_reason = NOT_WAITING;
      gtk_frame_set_label(GTK_FRAME(frame), _("Soundbridge Display - Not connected"));
      clear_pixbuf();
      my_printf("!iap_connected\n");
    }

    my_printf("%s(): return\n", __PRETTY_FUNCTION__);
}

gint main(gint argc, gchar *argv[])
{
    GtkWidget *vbox, *hbox1, *hbox2;
    //GdkColor color;
    /* buttons */
    GtkWidget *power_button, *alarm_button, *play_button, *pause_button,
              *stop_button, *next_button, *previous_button,
              *search_button, *add_button, *shuffle_button,
              *repeat_button, *brightness_button, *snooze_button, *vol_up_button, *vol_down_button;

    my_printf("%s()\n", __PRETTY_FUNCTION__);

    /* Initialize localization. */
    /*
    setlocale(LC_ALL, "");
    bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
    bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
    textdomain(GETTEXT_PACKAGE);
    */
    g_thread_init(NULL);

    /* Initialize _osso. */
    _osso = osso_initialize("com.kjainvestments.tabletbridge", VERSION, TRUE, NULL);
    if(!_osso)
    {
        g_printerr("osso_initialize failed.\n");
        return 1;
    }
    
    g_type_init();
        
    /* set when running on PC */
#ifndef ARM /* this doesn't seem to work on the tablet */
    //_iap_connected = TRUE;
#endif
    
    main_loop = g_main_loop_new(NULL, FALSE);

    system_dbus = dbus_bus_get(DBUS_BUS_SYSTEM, NULL);
    dbus_connection_setup_with_g_main(system_dbus, NULL);
    connection = con_ic_connection_new();
    g_signal_connect(G_OBJECT(connection), "connection-event",
                     G_CALLBACK(my_connection_cb), NULL);

    /* Connect to the internet pre-emptively to prevent lack thereof. */
    if(!_iap_connected && !_iap_connecting)
    {
        _iap_connecting = TRUE;
        con_ic_connection_connect(connection, CON_IC_CONNECT_FLAG_NONE);
    }

    gtk_init(&argc, &argv);

    /* Init gconf. */
    gconf_init(argc, argv, NULL);

    if(OSSO_OK != osso_rpc_set_default_cb_f(_osso, dbus_cb_default, NULL))
    {
        g_printerr("osso_rpc_set_default_cb_f failed.\n");
        return 1;
    }

    /* Create the hildon program and setup the title */
    program = HILDON_PROGRAM(hildon_program_get_instance());
    g_set_application_name("TabletBridge");

    /* Create HildonWindow and set it to HildonProgram */
    window = GTK_WIDGET(hildon_window_new());
    hildon_program_add_window(program, HILDON_WINDOW(window));

    /* Create pixel buf */
    pixbuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB, FALSE, 8,
                            displays[display_idx].width*displays[display_idx].horiz_scale,
                            displays[display_idx].height*displays[display_idx].vertical_scale);
    screen = gtk_image_new_from_pixbuf(pixbuf);
    clear_pixbuf();

    //gtk_container_add(GTK_CONTAINER(window), screen);

    hbox1 = gtk_hbox_new(FALSE, 2);
    hbox2 = gtk_hbox_new(FALSE, 2);
    vbox = gtk_vbox_new(FALSE, 2);
    //color.pixel = 0;
    //color.red = 0;
    //color.green = 0;
    //color.blue = 0;
    //gtk_widget_modify_bg(window, GTK_STATE_NORMAL, &color);
    frame = gtk_frame_new(_("Soundbridge Display - Not connected"));
    //gtk_widget_modify_bg(frame, GTK_STATE_NORMAL, &color);
    //gtk_widget_modify_bg(vbox, GTK_STATE_NORMAL, &color);
    //gtk_widget_modify_bg(screen, GTK_STATE_NORMAL, &color);
    //gtk_widget_modify_bg(hbox1, GTK_STATE_NORMAL, &color);
    //gtk_widget_modify_bg(hbox2, GTK_STATE_NORMAL, &color);
    gtk_container_add(GTK_CONTAINER(window), vbox);
    gtk_container_add(GTK_CONTAINER(vbox), frame);
    gtk_container_add(GTK_CONTAINER(vbox), hbox1);
    gtk_container_add(GTK_CONTAINER(vbox), hbox2);
    gtk_container_add(GTK_CONTAINER(frame), screen);
    //gtk_container_add(GTK_CONTAINER(vbox), screen);

    /* Create buttons and add them to main view */
    snooze_button = gtk_button_new_with_label(_("Sleep"));
    gtk_container_add(GTK_CONTAINER(hbox1), snooze_button);
    g_signal_connect(G_OBJECT(snooze_button), "clicked",
                          G_CALLBACK(main_screen_button_cb),
                          "IrDispatchCommand CK_SNOOZE\n");
    
    alarm_button = gtk_button_new_with_label(_("Alarm"));
    gtk_container_add(GTK_CONTAINER(hbox1), alarm_button);
    g_signal_connect(G_OBJECT(alarm_button), "clicked",
                          G_CALLBACK(main_screen_button_cb),
                          "IrDispatchCommand CK_ALARM\n");
    
    play_button = gtk_button_new_with_label(_("Play"));
    gtk_container_add(GTK_CONTAINER(hbox1), play_button);
    g_signal_connect(G_OBJECT(play_button), "clicked",
                          G_CALLBACK(main_screen_button_cb),
                          "IrDispatchCommand CK_PLAY\n");
    
    pause_button = gtk_button_new_with_label(_("Pause"));
    gtk_container_add(GTK_CONTAINER(hbox1), pause_button);
    g_signal_connect(G_OBJECT(pause_button), "clicked",
                          G_CALLBACK(main_screen_button_cb),
                          "IrDispatchCommand CK_PAUSE\n");
    
    stop_button = gtk_button_new_with_label(_("Stop"));
    gtk_container_add(GTK_CONTAINER(hbox1), stop_button);
    g_signal_connect(G_OBJECT(stop_button), "clicked",
                          G_CALLBACK(main_screen_button_cb),
                          "IrDispatchCommand CK_STOP\n");
    
    previous_button = gtk_button_new_with_label(_("Previous"));
    gtk_container_add(GTK_CONTAINER(hbox1), previous_button);
    g_signal_connect(G_OBJECT(previous_button), "clicked",
                          G_CALLBACK(main_screen_button_cb),
                          "IrDispatchCommand CK_PREVIOUS\n");
    
    next_button = gtk_button_new_with_label(_("Next"));
    gtk_container_add(GTK_CONTAINER(hbox1), next_button);
    g_signal_connect(G_OBJECT(next_button), "clicked",
                          G_CALLBACK(main_screen_button_cb),
                          "IrDispatchCommand CK_NEXT\n");
    /* Volume Up */
    vol_up_button = gtk_button_new_with_label(_("Vol +"));
    gtk_container_add(GTK_CONTAINER(hbox1), vol_up_button);
    g_signal_connect(G_OBJECT(vol_up_button), "clicked",
                          G_CALLBACK(main_screen_button_cb),
                          "IrDispatchCommand CK_VOLUME_UP\n");
    
    power_button = gtk_button_new_with_label(_("Power"));
    gtk_container_add(GTK_CONTAINER(hbox2), power_button);
    g_signal_connect(G_OBJECT(power_button), "clicked",
                          G_CALLBACK(main_screen_button_cb),
                          "IrDispatchCommand CK_POWER\n");
    
    search_button = gtk_button_new_with_label(_("Search"));
    gtk_container_add(GTK_CONTAINER(hbox2), search_button);
    g_signal_connect(G_OBJECT(search_button), "clicked",
                          G_CALLBACK(main_screen_button_cb),
                          "IrDispatchCommand CK_SEARCH\n");
    
    add_button = gtk_button_new_with_label(_("Add"));
    gtk_container_add(GTK_CONTAINER(hbox2), add_button);
    g_signal_connect(G_OBJECT(add_button), "clicked",
                          G_CALLBACK(main_screen_button_cb),
                          "IrDispatchCommand CK_ADD\n");
    
    shuffle_button = gtk_button_new_with_label(_("Shuffle"));
    gtk_container_add(GTK_CONTAINER(hbox2), shuffle_button);
    g_signal_connect(G_OBJECT(shuffle_button), "clicked",
                          G_CALLBACK(main_screen_button_cb),
                          "IrDispatchCommand CK_SHUFFLE\n");
    
    repeat_button = gtk_button_new_with_label(_("Repeat"));
    gtk_container_add(GTK_CONTAINER(hbox2), repeat_button);
    g_signal_connect(G_OBJECT(repeat_button), "clicked",
                          G_CALLBACK(main_screen_button_cb),
                          "IrDispatchCommand CK_REPEAT\n");
    
    brightness_button = gtk_button_new_with_label(_("Brightness"));
    gtk_container_add(GTK_CONTAINER(hbox2), brightness_button);
    g_signal_connect(G_OBJECT(brightness_button), "clicked",
                          G_CALLBACK(main_screen_button_cb),
                          "IrDispatchCommand CK_BRIGHTNESS\n");
    /* Volume Down */
    vol_down_button = gtk_button_new_with_label(_("Vol -"));
    gtk_container_add(GTK_CONTAINER(hbox2), vol_down_button);
    g_signal_connect(G_OBJECT(vol_down_button), "clicked",
                          G_CALLBACK(main_screen_button_cb),
                          "IrDispatchCommand CK_VOLUME_DOWN\n");


    menu_init();

    /* soundbridge scan */
    scan_info.progress = 0;
    scan_info.currently_scanning = FALSE;
    scan_info.sockfd = -1;

    sockfd = -1;

    /* Connect signal to X in the upper corner */
    g_signal_connect(G_OBJECT(window), "delete_event",
                     G_CALLBACK(gtk_main_quit), NULL);

    g_signal_connect(G_OBJECT(window), "key_press_event",
                     G_CALLBACK(window_cb_key_press), NULL);

    /* Begin the main application */
    gtk_widget_show_all(GTK_WIDGET(window));
    gtk_main();

    osso_deinitialize(_osso);

    my_printf("%s(): return\n", __PRETTY_FUNCTION__);
    return 0;
}
