import QtQuick 1.0

import "js/script.js" as Script
import "js/storage.js" as Storage

Item {
    id: mainWindow
    property bool showWaitIndicator: false
    width: 640
    height: 480

    property int startIndicator: 0
    property alias activePos: pfoMenu.activePos
    //property bool isLandscape: ( width > height )

    // Yahoo Finance model, no description!!
    XmlListModel {
        id: rssModel

        property string feedUrl: ""
        property bool loading: status == XmlListModel.Loading

        //source: "http://finance.yahoo.com/rss/headline?s=brcm" //feedUrl
        //source: "http://www.google.com/finance/company_news?q=NASDAQ:BRCM&output=rss"
        source: newsView.activeFeedUrl
        query: "/rss/channel/item"

        XmlRole { name: "title"; query: "title/string()" }
        XmlRole { name: "link"; query: "link/string()" }
        XmlRole { name: "desc"; query: "description/string()" }
        XmlRole { name: "date"; query: "pubDate/string()"; isKey: true }

        onStatusChanged: {
            if (status == XmlListModel.Ready) {
                console.log("FeedViewModel Status: ready")
            } else if (status == XmlListModel.Error) {
                console.log("FeedViewModel Status: error")
            } else if (status == XmlListModel.Loading) {
                console.log("FeedViewModel Status: loading")
            }
        }
    }

    ListModel {
         id: pfoModel
    }

    ListModel {
         id: posModel
    }

    Component.onCompleted: {
        Script.setComponents(pfoModel, posModel, waitState, errorState);
        Storage.getKeyValue("username", setLoginCredentials);
        Storage.getKeyValue("password", setLoginCredentials);
        Storage.getKeyValue("timer", setParams);
        Storage.getKeyValue("pfoStats", setParams);
        Storage.getKeyValue("activePos", setParams);
        Storage.getKeyValue("selectedIdx", setParams);
    }


    function setLoginCredentials(key, data) {
        if(key=="username") {
            settingMenu.username = data;
        } else {
            settingMenu.password = data;
        }
        // Auto login
        if(settingMenu.username.length>0 &&
                settingMenu.password.length>0) {
            mainWindow.state = "inLogin";
        } else {
            startIndicator++;
            if(startIndicator==2) {
                console.log("Try to show login dialog");
                mainWindow.state = "inSettingView";
            }
        }
    }

    function setParams (key, data) {
        if(key=="timer") {
            if (data == "")   { settingMenu.timer = 10; }
            else              { settingMenu.timer = data; }
        }
        else if (key=="pfoStats") {
            if (data != "")   { settingMenu.usePortfolioView = data; }
        }
        else if (key=="activePos") {
            if (data != "") { pfoMenu.activePos = data; }
        }
        else if (key=="pfoActiveIdx") {
            if (data != "") { pfoMenu.selectedIdx = data; }
        }
    }

    // Header: Minimize/Notification/Close
    TitleBar {
        id: header
        anchors { top: parent.top; left: parent.left; }
        width: parent.width
        showBusy: mainWindow.showWaitIndicator
        text: "stockona"
        z: 1
    }

    NewsView {
        id: newsView
        //anchors.fill: parent
        anchors { top: header.bottom; bottom: parent.bottom; left: parent.left; right: parent.right; }
        visible: false
        //clip: true

        onInMainView: {
            //mainWindow.state = 'inMainView'
            mainWindow.state = 'inStatsView'
        }
    }

    PosListView {
        id: mainView
        visible: false
        anchors { top: header.bottom; bottom: parent.bottom; left: parent.left; right: parent.right; }
        //anchors.fill: parent
        usePortfolioView: settingMenu.usePortfolioView
        //clip: true
        //contentHeight: parent.height - header.height

        onInNewsView: {
            mainWindow.state = 'inNewsView'
        }
        onInStatsView: {
            mainWindow.state = 'inStatsView'
        }
        onInMainView: {
            mainWindow.state = 'inMainView'
        }
    }

    PfoListView {
        id: pfoMenu
        visible: false
        //anchors.fill: parent
        anchors { top: header.bottom; bottom: parent.bottom; left: parent.left; right: parent.right; }
        //clip: true

        onUpdate: {
            mainWindow.state = "inLoadPortfolio";
            Storage.getKeyValue("activePos", loginUpdate);
            Storage.getKeyValue("pfoActiveIdx", loginUpdate);
        }
        onClose: {
            mainWindow.state = "inMainView";
        }
    }

    SettingMenu {
        id: settingMenu
        visible: false
        //anchors.fill: parent
        anchors { top: header.bottom; bottom: parent.bottom; left: parent.left; right: parent.right; }
        //clip: true

        onCallPfo: {
            mainWindow.state = "inPfoView";
        }
        onClose: {
            // Call loginUpdate function
            Storage.getKeyValue("username", loginUpdate);
            Storage.getKeyValue("password", loginUpdate);
            Storage.getKeyValue("timer", loginUpdate);
            Storage.getKeyValue("pfoStats", loginUpdate);
        }
    }

    function loginUpdate(key, data) {
        var loginChange = 0;

        if (key=="username" && (data != settingMenu.username)) {
            Storage.setKeyValue(key, settingMenu.username);
            loginChange = 1;
        }

        if (key=="password" && (data != settingMenu.password)) {
            Storage.setKeyValue(key, settingMenu.password);
            loginChange = 1;
        }

        // Robustness, minimum timer == 2
        if (key=="timer" && (data != settingMenu.timer)) {
            if (settingMenu.timer=="0") {
                Storage.setKeyValue(key, "2");
                settingMenu.timer="2";
            }
            else { Storage.setKeyValue(key, settingMenu.timer); }
        }

        if (key=="pfoStats" && (data != settingMenu.usePortfolioView)) {
            Storage.setKeyValue(key, settingMenu.usePortfolioView);
        }

        if (key=="activePos" && (data != pfoMenu.activePos)) {
            Storage.setKeyValue(key, pfoMenu.activePos);
        }

        if (key=="pfoActiveIdx" && (data != pfoMenu.selectedIdx)) {
            Storage.setKeyValue(key, pfoMenu.selectedIdx);
        }

        if (loginChange) { mainWindow.state = "inLogin"; }
        else             { mainWindow.state = "inMainView"; }
    }

    Row {
        anchors { right: parent.right; bottom: parent.bottom; margins: 20 }
        spacing: 10

        TextButton {
            id: menuButton
            text: "Menu"
            onClicked: {
                mainWindow.state = "inSettingView";
            }
        }

        TextButton {
            id: funcButton
            text: "View"
            onClicked: {
                //mainWindow.state = "inPfoView";
                settingMenu.usePortfolioView =! settingMenu.usePortfolioView;
            }
        }

        TextButton {
            id: refreshButton
            text: "Refresh"
            onClicked: {
                Script.loadOnePosition(activePos);
                mainWindow.state = "inMainView";
            }
        }
    }

    // Timer to control update interval
    Timer {
        id: updateTimer
        interval: settingMenu.timer*1000
        repeat: true
        onTriggered: {
            //console.log("Timer="+settingMenu.timer+"s");
            Script.loadOnePosition(activePos);
        }
    }

    Item {
        id: waitState

        states: [
            State {
                name: "hidden"
                StateChangeScript {
                    name: "hideIndicator"; script: { mainWindow.showWaitIndicator=false; }
                }
            },
            State {
                name: "shown"
                StateChangeScript {
                    name: "showIndicator"; script: { mainWindow.showWaitIndicator=true }
                }
            }
        ]
    }

    // Error dialog
    BusyDialog {
        id: errorDialog
        anchors.centerIn: parent
        busyIndicatorOn: false
        opacity: 0
        message: errorState.reason

        MouseArea {
            id: errorMouseArea
            anchors.fill: parent
            onClicked: {
                errorState.state = "hidden";
            }
        }
    }

//    Image {
//        id: reloadButton
//        anchors { top: errorDialog.bottom; horizontalCenter: errorDialog.horizontalCenter }
//        source: "gfx/reload.png"
//        opacity: 0
//        width: 60
//        height: 60

//        MouseArea {
//            id: reloadMouseArea
//            anchors.fill: parent
//            onClicked: {
//                errorState.state = "hidden";
//                mainWindow.state = "inLogin";
//            }
//        }
//    }


    // Error state is reset whenever not in mainView
    Item {
        id: errorState

        property string reason: ""

        states: [
            State {
                name: "hidden"
                PropertyChanges { target: errorDialog; opacity: 0 }
//                PropertyChanges { target: reloadButton; opacity: 0 }
            },
            State {
                name: "shown"
                PropertyChanges { target: errorDialog; opacity: 1 }
//                PropertyChanges { target: reloadButton; opacity: 1 }
                StateChangeScript {
                    name: "startErrorTimer"
                    script: { errorTimer.restart(); }
                }
            },
            State {
                name: "shownStatic"
                PropertyChanges { target: errorDialog; opacity: 1 }
//                PropertyChanges { target: reloadButton; opacity: 1 }

            }
        ]
    }

    Timer {
        id: errorTimer
        interval: 3000
        repeat: false
        onTriggered: {
            errorState.state = "hidden"
        }
    }


    // Fixed reference timer to control asynchronous network activity,
    // cannot re-use so have multiple instances for each stage.
    Timer {
        id: dlyTimerLogin
        // 3 second delay
        interval: 2500
        onTriggered: {
            //console.log("dlyTimerLogin triggered"
            mainWindow.state = "inLoadPortfolio"
        }
    }

    Timer {
        id: dlyTimerPfo
        // 2 second delay
        interval: 2000
        onTriggered: {
            //console.log("dlyTimerPfo triggered")
            mainWindow.state = "inMainView"
        }
    }

    // FSM
    states: [
        State {
            name: "inLogin";

            StateChangeScript {
                name: "loginStep"
                script: {
                    console.log("State.login");
                    Script.login(settingMenu.username, settingMenu.password);
                    mainWindow.showMainView();

                    dlyTimerLogin.restart();
                }
            }
        },

        State {
            name: "inLoadPortfolio";

            StateChangeScript {
                name: "PortfolioStep"
                script: {
                    console.log("State.loadPortfolio");
                    // clear posModel here
                    Script.cleanModel(posModel);
                    Script.loadAllPortfolios();
                    mainWindow.showMainView();

                    dlyTimerPfo.restart();
                }
            }
        },
		
        State {
            name: "inMainView";
            PropertyChanges { target: mainView; visible: false }
            // Hide error dialog when out of mainView
            PropertyChanges { target: errorState; state: "hidden" }

            StateChangeScript {
                name: "mainView"
                script: {
                    console.log("State.main");
                    Script.loadOnePosition(activePos);
                    mainWindow.showMainView();

                    updateTimer.start();
                }
            }
        },

        State {
            name: "inPfoView";
            PropertyChanges { target: pfoMenu; visible: false }
            StateChangeScript {
                name: "pfoView"
                script: {
                    // Re-load portfolio if loading failed earlier.
                    if (pfoModel.count==0) { Script.loadAllPortfolios(); }
                    mainWindow.showPfoMenu();

                    updateTimer.stop();
                    //dlyTimerPfo.stop();
                }
            }
        },

        State {
            name: "inSettingView";

            PropertyChanges { target: settingMenu; visible: false }
            //PropertyChanges { target: settingMenu; y: -mainWindows.height }

            StateChangeScript {
                name: "settingView"
                script: {
                    console.log("State.settingView");
                    mainWindow.showSettingMenu();

                    updateTimer.stop();
                    dlyTimerPfo.stop();
                }
            }
        },

        State {
            name: "inNewsView";

            PropertyChanges { target: mainView; visible: false }

            StateChangeScript {
                name: "newsView"
                script: {
                    console.log("State.newsView");
                    mainWindow.showNewsView();
                }
            }
        },

        State {
            name: "inStatsView";

            StateChangeScript {
                name: "statsView"
                script: {
                    console.log("State.statsView");
                    mainWindow.showStatsView();
                }
            }
        }
    ]

    function showPfoMenu() {
        pfoMenu.visible=true;

        menuButton.visible=false;
        refreshButton.visible=false;
        funcButton.visible=false;
    }

    function showSettingMenu() {
        mainView.visible=false;
        settingMenu.visible=true;
        settingMenu.y=0;

        menuButton.visible=false;
        refreshButton.visible=false;
        funcButton.visible=false;

        // Stop timers to avoid jumping back to main view
        dlyTimerPfo.stop();
        dlyTimerLogin.stop();
    }

    function showMainView() {
        mainView.visible=true;

        menuButton.visible=true;
        refreshButton.visible=true;
        funcButton.visible=true;
    }

    function showStatsView() {
        menuButton.visible=false;
        refreshButton.visible=false;
        funcButton.visible=false;
    }


    function showNewsView() {
        newsView.visible=true;

        menuButton.visible=false;
        refreshButton.visible=false;
        funcButton.visible=false;

        updateTimer.stop();
    }
}
