/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Pekka Nissinen - pekka.nissinen@ixonos.com
       Kaj Wallin - kaj.wallin@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#include <QDebug>
#include <QPainter>
#include <QSettings>

#include "zoombuttonpanel.h"
#include "panelcommon.h"
#include "zoombutton.h"
#include "common.h"

ZoomButtonPanel::ZoomButtonPanel(QWidget *parent)
    : QWidget(parent),
      m_isDraggable(false),
      m_panelLayout(this),
      m_zoomInButton(0),
      m_zoomOutButton(0)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_zoomInButton = new ZoomButton(this, ":/res/images/zoom_in.png");
    m_zoomOutButton = new ZoomButton(this, ":/res/images/zoom_out.png");

    m_panelLayout.setMargin(0);
    m_panelLayout.setSpacing(0);
    m_panelLayout.setVerticalSpacing(ZOOM_BUTTON_PANEL_BUTTON_SPACING);
    m_panelLayout.setSizeConstraint(QLayout::SetFixedSize);

    m_panelLayout.addWidget(m_zoomInButton, 0, 0);
    m_panelLayout.addWidget(m_zoomOutButton, 1, 0);

    QSettings settings(DIRECTORY_NAME, FILE_NAME);
    move(settings.value(ZOOMPANEL_POSITION,
                        QPoint(ZOOM_BUTTON_PANEL_POSITION_X,
                               ZOOM_BUTTON_PANEL_POSITION_Y)).toPoint());

    QPalette pal = palette();
    pal.setColor(QPalette::Background, QColor(0, 0, 0, 128));
    setPalette(pal);

    m_dragStartTimer = new QTimer(this);
    m_dragStartTimer->setSingleShot(true);
    m_dragStartTimer->setInterval(DRAG_INIT_TIME);

    m_forceReleaseTimer = new QTimer(this);
    m_forceReleaseTimer->setSingleShot(true);
    m_forceReleaseTimer->setInterval(FORCE_RELEASE_TIME);

    connect(m_zoomInButton, SIGNAL(pressed()),
            m_dragStartTimer, SLOT(start()));
    connect(m_zoomInButton, SIGNAL(released()),
            m_dragStartTimer, SLOT(stop()));
    connect(m_zoomOutButton, SIGNAL(pressed()),
            m_dragStartTimer, SLOT(start()));
    connect(m_zoomOutButton, SIGNAL(released()),
            m_dragStartTimer, SLOT(stop()));

    connect(m_dragStartTimer, SIGNAL(timeout()),
            this, SLOT(timerExpired()));
    connect(m_forceReleaseTimer, SIGNAL(timeout()),
            this, SLOT(forceMouseRelease()));
}

void ZoomButtonPanel::mouseMoveEvent(QMouseEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__;

    if(m_isDraggable) {
        if (event->buttons() & Qt::LeftButton) {
            QPoint newLocation = mapToParent(event->pos()) - m_dragPosition;

            if (newLocation.x() < SIDEBAR_WIDTH)
                newLocation.rx() = SIDEBAR_WIDTH;
            else if (newLocation.x() > m_screenSize.width() - width() - SIDEBAR_WIDTH)
                newLocation.rx() =  m_screenSize.width() - width() - SIDEBAR_WIDTH;

            if (newLocation.y() < 0)
                newLocation.ry() = 0;
            else if (newLocation.y() > m_screenSize.height() - height())
                newLocation.ry() = m_screenSize.height() - height();

            move(newLocation);
        }
    } else {
        if(!rect().contains(event->pos()))
            m_dragStartTimer->stop();
    }
    QWidget::mouseMoveEvent(event);
}

void ZoomButtonPanel::mousePressEvent(QMouseEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (event->button() == Qt::LeftButton) {
        m_dragPosition = event->pos();
    }
    m_dragStartTimer->start();
    QWidget::mousePressEvent(event);
}

void ZoomButtonPanel::mouseReleaseEvent(QMouseEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__;
    
    m_dragStartTimer->stop();

    Q_UNUSED(event);
    if(m_isDraggable) {
        setDraggable(false);
        QSettings settings(DIRECTORY_NAME, FILE_NAME);
        settings.setValue(ZOOMPANEL_POSITION, pos());
        releaseMouse();
        m_zoomInButton->setDown(false);
        m_zoomOutButton->setDown(false);
    }
    QWidget::mouseReleaseEvent(event);
}

const ZoomButton* ZoomButtonPanel::zoomInButton()
{
    qDebug() << __PRETTY_FUNCTION__;

    return m_zoomInButton;
}

const ZoomButton* ZoomButtonPanel::zoomOutButton()
{
    qDebug() << __PRETTY_FUNCTION__;

    return m_zoomOutButton;
}

void ZoomButtonPanel::disableZoomInButton()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_zoomInButton->setMode(QIcon::Disabled);
}

void ZoomButtonPanel::disableZoomOutButton()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_zoomOutButton->setMode(QIcon::Disabled);
}

void ZoomButtonPanel::resetButtons()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_zoomInButton->setMode(QIcon::Normal);
    m_zoomOutButton->setMode(QIcon::Normal);
}

void ZoomButtonPanel::setDraggable(bool mode, QPoint eventPosition)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_isDraggable = mode;

    if(mode) {
        setAutoFillBackground(true);

        m_zoomInMode = m_zoomInButton->mode();
        m_zoomOutMode = m_zoomOutButton->mode();

        m_zoomInButton->setMode(QIcon::Disabled);
        m_zoomOutButton->setMode(QIcon::Disabled);
        grabMouse();
        m_forceReleaseTimer->start();
        m_dragPosition = eventPosition;
    } else {
        setAutoFillBackground(false);
        if(m_zoomInMode == QIcon::Selected)
            m_zoomInButton->setMode(QIcon::Normal);
        else
            m_zoomInButton->setMode(m_zoomInMode);
        if(m_zoomOutMode == QIcon::Selected)
            m_zoomOutButton->setMode(QIcon::Normal);
        else
            m_zoomOutButton->setMode(m_zoomOutMode);
        releaseMouse();
        m_forceReleaseTimer->stop();
        m_zoomInButton->setDown(false);
        m_zoomOutButton->setDown(false);
    }
}

void ZoomButtonPanel::screenResized(const QSize &newSize)
{
    qDebug() << __PRETTY_FUNCTION__;
    
    m_screenSize = newSize;

    QPoint resizedPosition = pos();
    if(resizedPosition.x() > (newSize.width() - rect().width()))
        resizedPosition.rx() = newSize.width() - rect().width();
    else if (resizedPosition.x() < SIDEBAR_WIDTH)
        resizedPosition.rx() = SIDEBAR_WIDTH;
    if(resizedPosition.y() > (newSize.height() - rect().height()))
        resizedPosition.ry() = newSize.height() - rect().height();
    else if (resizedPosition.y() < 0)
        resizedPosition.ry() = 0;
    move(resizedPosition);
}

void ZoomButtonPanel::forceMouseRelease()
{
    qDebug() << __PRETTY_FUNCTION__;

    releaseMouse();
    setDraggable(false);
}

void ZoomButtonPanel::timerExpired()
{
    qDebug() << __PRETTY_FUNCTION__;
    
    if(m_zoomInButton->isDown())
        m_dragPosition = m_zoomInButton->eventPosition();
    if(m_zoomOutButton->isDown())
        m_dragPosition = m_zoomOutButton->eventPosition();

    setDraggable(true, m_dragPosition);
}
