/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

      Katri Kaikkonen - katri.kaikkonen@ixonos.com
      Henri Lampela - henri.lampela@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#include <QDebug>
#include "updatelocationdialog.h"

const int MESSAGE_MAX_LENGTH = 255;

UpdateLocationDialog::UpdateLocationDialog(const QString &userMessage, bool publishOnFacebook,
                                           QWidget *parent)
    : QDialog(parent)
{
    qDebug() << __PRETTY_FUNCTION__;

    setWindowTitle(tr("Update Location"));

    QGridLayout *gridLayout = new QGridLayout();

    m_textEdit = new QTextEdit();
    m_charCountLabel = new QLabel();

    if (userMessage.isEmpty())
    {
        m_textEdit->setText(tr("Message:"));
        m_textEdit->selectAll();
        m_charCountLabel->setNum(MESSAGE_MAX_LENGTH);

        connect(m_textEdit, SIGNAL(selectionChanged()),
                this, SLOT(textSelectionChanged()));
    }
    else
    {
        m_textEdit->setText(userMessage);
        m_textEdit->document()->setModified(true);
        textChanged();
    }

    QScrollArea *scrollArea = new QScrollArea();

    QVBoxLayout* scrollAreaLayout = new QVBoxLayout;
    scrollAreaLayout->setSpacing(0);
    scrollAreaLayout->setMargin(0);
    scrollAreaLayout->addWidget(m_textEdit);

    QWidget* scrollAreaWidget = new QWidget;
    scrollAreaWidget->setLayout(scrollAreaLayout);

    scrollArea->setWidget(scrollAreaWidget);

    scrollArea->setWidgetResizable(true);
    scrollArea->setFrameShape(QScrollArea::NoFrame);

    m_locationLabel = new QLabel();
    m_locationLabel->setWordWrap(true);
    m_checkBox = new QCheckBox(tr("Publish on Facebook"));
    m_checkBox->setChecked(publishOnFacebook);

    QDialogButtonBox *buttonBox = new QDialogButtonBox(Qt::Vertical);
    QPushButton *sendButton = buttonBox->addButton(QDialogButtonBox::Ok);
    QPushButton *cancelButton = buttonBox->addButton(QDialogButtonBox::Cancel);
    sendButton->setText(tr("Send"));

    gridLayout->addWidget(new QLabel(tr("Location:")), 0, 0, 1, 1, Qt::AlignTop | Qt::AlignLeft);
    gridLayout->addWidget(m_locationLabel, 0, 1, 1, 3);
    gridLayout->addWidget(scrollArea,      1, 0, 1, 4);
    gridLayout->addWidget(m_checkBox,      2, 0, 1, 2);
    gridLayout->addWidget(m_charCountLabel,2, 2, 1, 1, Qt::AlignCenter);
    gridLayout->addWidget(buttonBox,       2, 3, 2, 1, Qt::AlignTop);

    gridLayout->setColumnMinimumWidth(2, 2 * m_charCountLabel->fontMetrics().width(
            m_charCountLabel->text()));

    setLayout(gridLayout);

    const int LAYOUT_COLUMN_ONE_STRETCH = 10;
    const int LAYOUT_TOP_MARGIN = 15;
    gridLayout->setColumnStretch(1, LAYOUT_COLUMN_ONE_STRETCH);

    QMargins layoutMargins = gridLayout->contentsMargins();
    layoutMargins.setTop(LAYOUT_TOP_MARGIN);
    gridLayout->setContentsMargins(layoutMargins);

    connect(sendButton, SIGNAL(clicked()),
            this, SLOT(sendUpdate()));

    connect(cancelButton, SIGNAL(clicked()),
            this, SLOT(reject()));

    connect(m_textEdit, SIGNAL(textChanged()),
            this, SLOT(textChanged()));

#ifdef Q_WS_MAEMO_5
    if (QAbstractKineticScroller *scroller = scrollArea->property("kineticScroller").value<
                                              QAbstractKineticScroller *>()) {
        scroller->setOvershootPolicy(QAbstractKineticScroller::OvershootAlwaysOff);
    }

    new TextEditAutoResizer(m_textEdit);
    setAttribute(Qt::WA_Maemo5StackedWindow);
    setWindowFlags(Qt::Window);
#endif
}

UpdateLocationDialog::~UpdateLocationDialog()
{
    qDebug() << __PRETTY_FUNCTION__;
}

void UpdateLocationDialog::setAddress(const QString &address)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_locationLabel->setText(address);
}

void UpdateLocationDialog::sendUpdate()
{
    qDebug() << __PRETTY_FUNCTION__;

    // coordinates for this call will be get from somewhere, map etc...
    emit statusUpdate(m_textEdit->document()->isModified() ? m_textEdit->toPlainText() : QString(),
                      m_checkBox->isChecked());

    close();
}

void UpdateLocationDialog::textChanged()
{
    qDebug() << __PRETTY_FUNCTION__;

    QString msgText = m_textEdit->toPlainText();

    if (msgText.length() > MESSAGE_MAX_LENGTH)
    {
        int cursorPos = m_textEdit->textCursor().position();

        QTextCursor textCursor = m_textEdit->textCursor();
        int removedChars = msgText.length() - MESSAGE_MAX_LENGTH;
        m_textEdit->blockSignals(true);

        m_textEdit->setText( msgText.left(cursorPos - removedChars) + msgText.mid(cursorPos));
        m_textEdit->blockSignals(false);

        textCursor.setPosition(cursorPos - removedChars);
        m_textEdit->setTextCursor(textCursor);
    }

    m_charCountLabel->setNum(MESSAGE_MAX_LENGTH - m_textEdit->toPlainText().length());
}

void UpdateLocationDialog::textSelectionChanged()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (!m_textEdit->document()->isModified())
    {
        m_textEdit->clear();
    }
    disconnect(m_textEdit, SIGNAL(selectionChanged()), this, SLOT(textSelectionChanged()));
}

