/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Kaj Wallin - kaj.wallin@ixonos.com
        Pekka Nissinen - pekka.nissinen@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#ifndef SIDEPANEL_H
#define SIDEPANEL_H

#include <QWidget>
#include <QtGui>

class PanelSliderBar;
class SidePanelBase;

/**
 * @brief Base class for sliding side panels
 *
 * @author Kaj Wallin - kaj.wallin (at) ixonos.com
 * @author Pekka Nissinen - pekka.nissinen@ixonos.com
 *
 * @class SidePanel sidepanel.h "ui/sidepanel.h"
 */
class SidePanel : public QWidget
{
    Q_OBJECT

public:
    /**
     * @brief Constructor
     *
     * @param parent
     */
    SidePanel(QWidget *parent = 0);

    /**
     * @brief Enumerator for the panel type
     * Defines values: UserPanel, FriendPanel, None
     *
     * @sa setType
     */
    enum PanelType {UserPanel, FriendPanel, None};

/******************************************************************************
 * MEMBER FUNCTIONS AND SLOTS
 ******************************************************************************/
public:
    /**
     * @brief Type setter for the panel. Also sets panel visible
     *
     * Use to set panel type as UserPanel or FriendPanel. Panel type determines
     * which side the panel will be rendered. UserPanel will always be rendered
     * on the left side of the screen and FriendPanel on the right side. Type
     * cannot be set twice.
     *
     * @param type Type of the panel, either UserPanel or FriendPanel
     * @sa PanelType
     */
    void setType(SidePanel::PanelType type);

public slots:
    /**
     * @brief Public slot that will open the panel unless already open
     */
    void openPanel();

    /**
     * @brief Public slot that will close the panel unless already closed
     */
    void closePanel();

    /**
     * @brief Slot to redraw the panel after window resize event
     *
     * @param size Size of the new window
     */
    void resizePanel(const QSize &size);

private slots:
    /**
     * @brief Internal slot used to track statemachine state
     */
    void stateChangedToClosed();

    /**
     * @brief Internal slot used to track statemachine state
     */
    void stateChangedToOpen();

/******************************************************************************
 * SIGNALS
 ******************************************************************************/
signals:
    /**
     * @brief Signal that is sent to state machine when state must be changed
     *
     * @sa openPanel
     * @sa closePanel
     */
    void toggleState();

    /**
     * @brief Signal that is sent when panel is closed
     *
     * @sa openPanel
     * @sa closePanel
     */
    void panelClosed();

    /**
     * @brief Signal that is sent when panel is opened
     *
     * @sa openPanel
     * @sa closePanel
     */
    void panelOpened();

/*******************************************************************************
 * DATA MEMBERS
 *******************************************************************************/
protected:
    QVBoxLayout *m_panelVBox; ///< Vertical layout inside the panel

private:
    bool m_isOpen; ///< Boolean used to track the current state of the statemachine

    QSignalTransition *m_panelTransitionClose;  ///< Transition signal for closing the panel
    QSignalTransition *m_panelTransitionOpen;   ///< Transition signal for opening the panel
    QState *m_panelStateClosed;                 ///< State of the closed panel
    QState *m_panelStateOpened;                 ///< State of the opened panel
    QStateMachine *m_panelStateMachine;         ///< State machine for sliding the panel

    SidePanelBase *m_panelBase;         ///< Widget for panel base
    PanelType currentType;              ///< Holder for the type of this panel
    PanelSliderBar *m_panelSlidingBar;  ///< Widget for panel sliding bar item
};

#endif // SIDEPANEL_H
