/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

      Katri Kaikkonen - katri.kaikkonen@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#ifdef Q_WS_MAEMO_5
#include <QMaemo5TimePickSelector>
#include <QMaemo5ValueButton>
#endif

#include <QtGui>
#include <QDebug>
#include <QTime>
#include "common.h"
#include "settingsdialog.h"

const int LIST_MINUTES_STEP = 5;
const int LIST_MINUTES_MAX = 60;
const int LIST_HOURS_MAX = 1;

SettingsDialog::SettingsDialog(QWidget *parent)
    : QDialog(parent)
{
    qDebug() << __PRETTY_FUNCTION__;
    setWindowTitle(tr("Settings"));
    setAttribute(Qt::WA_DeleteOnClose, true);

    QScrollArea *scrollArea = new QScrollArea(this);
    QGridLayout *gridLayout = new QGridLayout(this);
    QGroupBox *groupBox = new QGroupBox(scrollArea);

    m_automaticLocationUpdate = new QCheckBox(tr("Use automatic location update"));

    QDialogButtonBox *buttonBox = new QDialogButtonBox(Qt::Vertical);
    m_saveButton = buttonBox->addButton(QDialogButtonBox::Save);
    QPushButton *cancelButton = buttonBox->addButton(QDialogButtonBox::Cancel);

#ifdef Q_WS_MAEMO_5
    m_automaticLocationUpdateIntervalButton = new QMaemo5ValueButton(tr("Update interval"), this);
    m_automaticLocationUpdateIntervalButton->setDisabled(true);
    m_timePick = new QMaemo5ListPickSelector;
    m_automaticLocationUpdateIntervalButton->setPickSelector(m_timePick);
    m_automaticLocationUpdateIntervalButton->setValueLayout(QMaemo5ValueButton::ValueBesideText);
    QStandardItemModel *updateIntervalListModel = new QStandardItemModel(0, 1, this);
    populateUpdateIntervalList(updateIntervalListModel);
    m_timePick->setModel(updateIntervalListModel);
    m_automaticLocationUpdateIntervalButton->setValueText(
            updateIntervalListModel->item(0, 0)->text());
    Q_UNUSED(cancelButton);
#else
    m_automaticLocationUpdateInterval = new QTimeEdit();
    m_automaticLocationUpdateInterval->setTimeRange(QTime(0, LIST_MINUTES_STEP),
                                                    QTime(LIST_HOURS_MAX, 0));
    m_automaticLocationUpdateInterval->setDisplayFormat("hh:mm");
    m_automaticLocationUpdateInterval->setDisabled(true);

    connect(cancelButton, SIGNAL(clicked()), this, SLOT(reject()));
#endif

    connect(m_automaticLocationUpdate, SIGNAL(toggled(bool)),
            this, SLOT(toggleAutomaticLocationUpdate(bool)));
    connect(m_saveButton, SIGNAL(clicked()), this, SLOT(saveValues()));

    QFormLayout *form = new QFormLayout();
    form->setRowWrapPolicy(QFormLayout::WrapAllRows);
    form->addWidget(m_automaticLocationUpdate);

#ifdef Q_WS_MAEMO_5
    form->addWidget(m_automaticLocationUpdateIntervalButton);
#else
    form->addRow(tr("Update interval"), m_automaticLocationUpdateInterval);
#endif

    groupBox->setLayout(form);
    scrollArea->setWidget(groupBox);
    scrollArea->setWidgetResizable(true);
    gridLayout->addWidget(scrollArea, 0, 0, 2, 1);
    gridLayout->addWidget(buttonBox, 0, 1, 1, 1);
    setLayout(gridLayout);

    scrollArea->show();

    readSettings();
}

void SettingsDialog::enableSituareSettings(bool enabled)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_saveButton->setEnabled(enabled);
    m_automaticLocationUpdate->setEnabled(enabled);

    if (enabled)
        toggleAutomaticLocationUpdate(m_automaticLocationUpdate->isChecked());
    else
        toggleAutomaticLocationUpdate(false);
}

void SettingsDialog::populateUpdateIntervalList(QStandardItemModel *model)
{
    qDebug() << __PRETTY_FUNCTION__;

    for (int i = LIST_MINUTES_STEP; i <= LIST_MINUTES_MAX; i+=LIST_MINUTES_STEP) {
        QStandardItem *item = new QStandardItem(QString(tr("%1 min")).arg(i));
        item->setTextAlignment(Qt::AlignCenter);
        item->setEditable(false);
        model->appendRow(item);
    }
}

void SettingsDialog::readSettings()
{
    qDebug() << __PRETTY_FUNCTION__;

    QSettings settings(DIRECTORY_NAME, FILE_NAME);
    bool automaticUpdateEnabled = settings.value(SETTINGS_AUTOMATIC_UPDATE_ENABLED, false).toBool();
    QTime automaticUpdateInterval = settings.value(SETTINGS_AUTOMATIC_UPDATE_INTERVAL, QTime())
                                      .toTime();

    m_automaticLocationUpdate->setChecked(automaticUpdateEnabled);

    if (automaticUpdateInterval.isValid())
        setTime(automaticUpdateInterval);
    else
        setTime(QTime(0, LIST_MINUTES_STEP));
}

void SettingsDialog::saveValues()
{
    qDebug() << __PRETTY_FUNCTION__;

    QSettings settings(DIRECTORY_NAME, FILE_NAME);
    settings.setValue(SETTINGS_AUTOMATIC_UPDATE_ENABLED, m_automaticLocationUpdate->isChecked());
    settings.setValue(SETTINGS_AUTOMATIC_UPDATE_INTERVAL, time());

    accept();
}

void SettingsDialog::setTime(const QTime &time)
{
    qDebug() << __PRETTY_FUNCTION__;

#ifdef Q_WS_MAEMO_5
        // Convert time to index in list
        int index = time.minute()/LIST_MINUTES_STEP - 1;

        if (index < 0)
            index = 0;
        if (index >= m_timePick->model()->rowCount())
            index = m_timePick->model()->rowCount() - 1;

        m_timePick->setCurrentIndex(index);
#else
        m_automaticLocationUpdateInterval->setTime(time);
#endif
}

QTime SettingsDialog::time()
{
    qDebug() << __PRETTY_FUNCTION__;

    QTime time;

#ifdef Q_WS_MAEMO_5
    time = time.addSecs((m_timePick->currentIndex()+1)*LIST_MINUTES_STEP*60);
#else
    time = m_automaticLocationUpdateInterval->time();
#endif

    if (time < QTime(0, LIST_MINUTES_STEP))
        time = QTime(0, LIST_MINUTES_STEP);
    if (time > QTime(LIST_HOURS_MAX, 0))
        time = QTime(LIST_HOURS_MAX, 0);

    return time;
}

void SettingsDialog::toggleAutomaticLocationUpdate(bool enabled)
{
    qDebug() << __PRETTY_FUNCTION__;

#ifdef Q_WS_MAEMO_5
    m_automaticLocationUpdateIntervalButton->setEnabled(enabled);
#else
    m_automaticLocationUpdateInterval->setEnabled(enabled);
#endif
}
