 /*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Kaj Wallin - kaj.wallin@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
 */

#include <QPainter>
#include <QPen>
#include <QLine>
#include <QDebug>
#include "mapscale.h"
#include <math.h>

const int TARGET_WIDTH = 140;                   ///< Target width of the scale in pixels
const qreal M_TO_FT = 3.2808399;                ///< Meter to feet conversion value
const qreal FT_TO_MI = 5280;                    ///< Feet to mile conversion
const qreal M_TO_KM = 1000;                     ///< Meters to kilometers conversion
const int MAPSCALE_HEIGHT = 31;                 ///< Height of the the map scale
const int CENTERLINE_Y = MAPSCALE_HEIGHT / 2;   ///< Y position of the centerline
const int MAPSCALE_STOP_HEIGHT = 7;             ///< Height of each perpendicular stop on the scale
const int SCALE_TEXT_X = 3;                     ///< X coordinate for the scale texts

MapScale::MapScale(QWidget *parent) :
    QWidget(parent),
    m_centerLineImperial(0),
    m_centerLineMetric(0),
    m_imperialText(""),
    m_metricText("")
{
    qDebug() << __PRETTY_FUNCTION__;

//    updateMapScale(10.0);
    setAttribute(Qt::WA_TransparentForMouseEvents, true);
}

void MapScale::paintEvent(QPaintEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__;

    Q_UNUSED(event);

    resize(ceil(fmax(m_centerLineMetric,m_centerLineImperial)) + 1, MAPSCALE_HEIGHT);

    QLineF centerLine(0, CENTERLINE_Y, fmax(m_centerLineMetric, m_centerLineImperial), CENTERLINE_Y);
    QLineF startKm(1, CENTERLINE_Y, 1, CENTERLINE_Y - MAPSCALE_STOP_HEIGHT);
    QLineF stopKm(m_centerLineMetric, CENTERLINE_Y,
                  m_centerLineMetric, CENTERLINE_Y - MAPSCALE_STOP_HEIGHT);

    QLineF startMi(1, CENTERLINE_Y, 1, CENTERLINE_Y + MAPSCALE_STOP_HEIGHT);
    QLineF stopMi(m_centerLineImperial, CENTERLINE_Y,
                  m_centerLineImperial, CENTERLINE_Y + MAPSCALE_STOP_HEIGHT);

    QPainter painter(this);
    QPen pen(Qt::black, 2);

    painter.setFont(QFont("Nokia Sans", 13, QFont::Normal));
    painter.setPen(pen);
    painter.drawLine(centerLine);
    painter.drawLine(startKm);
    painter.drawLine(stopKm);
    painter.drawText(SCALE_TEXT_X, MAPSCALE_HEIGHT / 2 - 2, m_metricText);

    painter.drawLine(startMi);
    painter.drawLine(stopMi);
    painter.drawText(SCALE_TEXT_X, MAPSCALE_HEIGHT, m_imperialText);
}

qreal MapScale::roundToBaseScale(qreal value)
{
    qDebug() << __PRETTY_FUNCTION__;

    int scale = 0;
    qreal baseLine;
    while(value > 1){
        value = value/10;
        scale++;
    }
    if(value < 0.15)
        baseLine = 1;
    else if (value < 0.35)
        baseLine = 2;
    else if (value < 0.75)
        baseLine = 5;
    else
        baseLine = 10;
    baseLine = baseLine * (pow(10,scale-1));
    return baseLine;
}

void MapScale::updateMapResolution(const qreal &resolution)
{
    qDebug() << __PRETTY_FUNCTION__;

    //Calculate distance scale in metric units
    qreal genericMetricScale = TARGET_WIDTH * resolution;
    qreal baseMetricScale = roundToBaseScale(genericMetricScale);
    if(resolution != 0)
        m_centerLineMetric = baseMetricScale / resolution;
    else
        m_centerLineMetric = baseMetricScale / 0.000001;

    if(baseMetricScale < M_TO_KM)
    {
        m_metricText.setNum(baseMetricScale);
        m_metricText.append(" m");
    } else {
        m_metricText.setNum(baseMetricScale/M_TO_KM);
        m_metricText.append(" km");
    }

    //Calculate distance scale in imperial units
    qreal imperialScaleResolution = resolution * M_TO_FT;
    qreal genericImperialScale = TARGET_WIDTH * imperialScaleResolution;
    qreal baseImperialScale;

    if(genericImperialScale < FT_TO_MI) {
        baseImperialScale = roundToBaseScale(genericImperialScale);
        if(imperialScaleResolution != 0)
            m_centerLineImperial = baseImperialScale / imperialScaleResolution;
        else
            m_centerLineImperial = baseImperialScale / 0.000001;
        m_imperialText.setNum(baseImperialScale);
        m_imperialText.append(" ft");
    } else {
        baseImperialScale = roundToBaseScale(genericImperialScale / FT_TO_MI);
        if(imperialScaleResolution != 0)
            m_centerLineImperial = (baseImperialScale*FT_TO_MI) / imperialScaleResolution;
        else
            m_centerLineImperial = baseImperialScale / 0.000001;
        m_imperialText.setNum(baseImperialScale);
        m_imperialText.append(" mi");
    }

    update();
}
