/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

      Henri Lampela - henri.lampela@ixonos.com
      Kaj Wallin - kaj.wallin@ixonos.com
      Sami Rämö - sami.ramo@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#ifndef MAINWINDOW_H
#define MAINWINDOW_H

#include <QtGui/QMainWindow>
#include <QUrl>

#include "network/networkcookiejar.h"
#include "panelsidebar.h"

class QGraphicsScene;
class QLabel;
class QWebView;
class QNetworkReply;

class FacebookAuthentication;
class FullScreenButton;
class FriendListPanel;
class MapScale;
class MapScene;
class MapView;
class SituareService;
class User;
class UserInfoPanel;
class ZoomButtonPanel;
class SettingsDialog;
class QToolButton;
class QMessageBox;

/**
* @brief Main Window Class
*/
class MainWindow : public QMainWindow
{
    Q_OBJECT

public:
    /**
    * @brief Constructor
    *
    * @param parent Parent
    */
    MainWindow(QWidget *parent = 0);

    /**
    * @brief Destructor
    *
    */
    ~MainWindow();

/*******************************************************************************
 * BASE CLASS INHERITED AND REIMPLEMENTED MEMBER FUNCTIONS
 ******************************************************************************/
private:
    /**
      * @brief HW increase and decrease key presses are grabbed and used for zooming the map.
      */
    void keyPressEvent(QKeyEvent* event);

/*******************************************************************************
 * MEMBER FUNCTIONS AND SLOTS
 ******************************************************************************/
public:
    /**
    * @brief Clears cookie jar
    *
    */
    void clearCookieJar();

    /**
    * @brief
    *
    * @param logged
    */
    void loggedIn(bool logged);

    /**
    * @brief Gets the login state (logged in/logged out)
    *
    * @return bool Login state
    */
    bool loginState();

    /**
    * @brief Reads automatic location update settings.
    */
    void readAutomaticLocationUpdateSettings();

    /**
    * @brief Enable / disable auto centering button.
    *
    * @param enabled true if shoud be enabled, false otherwise
    */
    void setAutoCenteringButtonEnabled(bool enabled);

    /**
    * @brief Enable / disable GPS button.
    *
    * Does set visibilities for manual location cursor and auto centering menu button.
    *
    * @param enabled true if enabled, false otherwise
    */
    void setGPSButtonEnabled(bool enabled);

    /**
      * @brief Set scene for MapView
      *
      * @param scene Scene to be set
      */
    void setMapViewScene(QGraphicsScene *scene);

    /**
    * Shows dialog with enable automatic location update question.
    *
    * @param text text to show in dialog
    */
    void showEnableAutomaticUpdateLocationDialog(const QString &text);

    /**
    * @brief Gets the username from member variable for saving purposes
    *
    * @return QString Username
    */
    const QString username();

public slots:
    /**
    * @brief Builds information box with message.
    *
    * @param message Information message
    * @param modal Modal = true, non-modal false
    */
    void buildInformationBox(const QString &message, bool modal=false);

    /**
    * @brief Slot for failed login
    *
    */
    void loginFailed();

    /**
    * @brief Slot to intercept signal when login with cookies is requested
    *
    */
    void loginUsingCookies();

    /**
    * @brief Public slot, which open settings dialog
    */
    void openSettingsDialog();

    /**
      * @brief Set own location crosshair visibility
      *
      * @param visible
      */
    void setOwnLocationCrosshairVisibility(bool visible);

    /**
    * @brief Sets username to member variable for login dialog
    *
    * @param username Username to be set
    */
    void setUsername(const QString &username);

    /**
    * @brief Method to show panels
    *
    */
    void showPanels();

    /**
    * @brief Public slot to intercept signal when old cerdentials are invalid or credentials
    *        doesn't exist yet
    *
    */
    void startLoginProcess();

    /**
    * @brief Toggle progress indicator.
    *
    * @param state true if progress indicator should be shown, false otherwise
    */
    void toggleProgressIndicator(bool state);

    /**
    * @brief Shows / hides Situare related UI items
    *
    */
    void updateItemVisibility();

private:
    /**
      * @brief Build fullscreen toggle button and connect slots
      */
    void buildFullScreenButton();

    /**
      * @brief Build friend list panel and connect slots
      */
    void buildFriendListPanel();

    /**
      * @brief Build manual location setting cross hair and connect slots
      */
    void buildManualLocationCrosshair();

    /**
      * @brief Build map and connect slots
      */
    void buildMap();

    /**
     * @brief Build map scale and connect slots
     */
    void buildMapScale();

    /**
      * @brief Build OSM license and connect slots
      */
    void buildOsmLicense();

    /**
      * @brief Build user info panel and connect slots
      */
    void buildUserInfoPanel();

    /**
    * @brief Build webview and connect slots
    *
    */
    void buildWebView();

    /**
      * @brief Build zoom button panel and connect slots
      */
    void buildZoomButtonPanel();

    /**
    * @brief Private method to create the Menu items
    */
    void createMenus();

    /**
      * @brief Grab or release HW increase and decrease buttons.
      *
      * @param grab Use true for grabbing and false for releasing the keys
      */
    void grabZoomKeys(bool grab);

    /**
    * @brief Queues dialog/information box
    *
    * @param dialog Dialog to be added into queue
    */
    void queueDialog(QDialog *dialog);

    /**
    * @brief Shows queued error information box
    *
    */
    void showErrorInformationBox();

    /**
    * @brief Shows queued information box
    *
    * @fn showInformationBox
    */
    void showInformationBox();

private slots:
    /**
    * @brief Slot for automatic update dialog finished.
    *
    * @result result code
    */
    void automaticUpdateDialogFinished(int result);

    /**
    * @brief Slot to intercept signal when dialog/information note is processed
    *
    * @param status Status of the dialog
    */
    void dialogFinished(int status);

    /**
    * @brief Slot for drawing the fullscreen toggle button
    *
    * @param size Size of the screen
    */
    void drawFullScreenButton(const QSize &size);

    /**
    * @brief Slot for drawing the map distance scale
    *
    * @param size Size of the screen
    */
    void drawMapScale(const QSize &size);

    /**
    * @brief Slot for drawing the Open Street Map license text
    *
    * @param size Size of the screen
    */
    void drawOsmLicense(const QSize &size);

    /**
    * @brief Slot for drawing the own location crosshair
    *
    * @param size Size of the screen
    */
    void drawOwnLocationCrosshair(const QSize &size);

    /**
    * @brief Slot to intercept signal when error dialog/information note is processed
    *
    * @param status Status of the dialog
    */
    void errorDialogFinished(int status);

    /**
    * @brief Slot for gps timeout.
    *
    * Called when request timeout occurs.
    */
    void gpsTimeout();

    /**
    * @brief Slot to load cookies from settings
    *
    */
    void loadCookies();

    /**
    * @brief Slot to intercept signal when webview has finished loading webpage
    *
    * @param done Status of the loading
    */
    void loadDone(bool done);

    /**
    * @brief Slot to save cookies to settings
    *
    */
    void saveCookies();

    /**
    * @brief Slot for settings dialog accepted.
    */
    void settingsDialogAccepted();

    /**
    * @brief Set correnct view port size to datamembers
    *
    * @param size Size of the screen
    */
    void setViewPortSize(const QSize &size);

    /**
    * @brief Toggle between fullscreen and normal window mode
    */
    void toggleFullScreen();

    /**
    * @brief Slot to intercept signal from webview's networkaccessmanager
    *
    * @param reply Network reply (contains errors)
    */
    void webViewRequestFinished(QNetworkReply* reply);

/*******************************************************************************
 * SIGNALS
 ******************************************************************************/
signals:
    /**
    * @brief Automatic centering setting changed by user
    *
    * @param enabled True if automatic centering is enabled, otherwise false
    */
    void autoCenteringTriggered(bool enabled);

    /**
    * @brief Signal that indicates when user has cancelled login process
    *
    */
    void cancelLoginProcess();

    /**
    * @brief View should be centered to new location
    *
    * @param sceneCoordinate Scene coordinates of the new center point
    */
    void centerToSceneCoordinates(QPoint sceneCoordinate);

    /**
    * @brief Signal for enabling automatic location update.
    *
    * @param enabled true if enabled, false otherwise
    * @param updateIntervalMsecs update interval in milliseconds
    */
    void enableAutomaticLocationUpdate(bool enabled, int updateIntervalMsecs = 0);

    /**
    * @brief Signals error
    *
    * @param context error context
    * @param error error code
    */
    void error(const int context, const int error);

    /**
    * @brief Signal for requesting username from settings
    *
    */
    void fetchUsernameFromSettings();

    /**
    * @brief Signal for finding user.
    *
    * @param coordinates user geo coordinates
    */
    void findUser(const QPointF &coordinates);

    /**
    * @brief GPS setting changed
    *
    * @param enabled True if GPS is enabled, otherwise false
    */
    void gpsTriggered(bool enabled);

    /**
    * @brief Signal for finding friend.
    *
    * @param coordinates friend's geo coordinates
    */
    void findFriend(const QPointF &coordinates);

    /**
    * @brief Signal for friend location ready.
    *
    * @param friendsList
    */
    void friendsLocationsReady(QList<User *> &friendsList);

    /**
    * @brief Signal is emitted when location item is clicked.
    *
    * @param userIDs list of friends user IDs in the group
    */
    void locationItemClicked(const QList<QString> &userIDs);

    /**
    * @brief Signals when Login/Logout action is pressed
    *
    */
    void loginActionPressed();

    /**
    * @brief MapView has been resized
    *
    * @param size view size
    */
    void mapViewResized(const QSize &size);

    /**
      * @brief Forwarding signal from MapView to MapEngine
      *
      * @param sceneCoordinate
      */
    void mapViewScrolled(QPoint sceneCoordinate);

    /**
      * @brief Forwarding signal from MapEngine to MapView
      */
    void maxZoomLevelReached();

    /**
    * @brief Signal that informs that user's message/location failed to update on Situare server
    *        This signal is originally sended from SituareService with name error
    *        Signal is renamed on MainWindow
    *
    * @param error error code
    */
    void messageSendingFailed(const int error);

    /**
      * @brief Forwarding signal from MapEngine to MapView
      */
    void minZoomLevelReached();

    /**
     * @brief Forwarding signal from MapEngine to MapScale
     */
    void newMapResolution(qreal scale);

    /**
    * @brief Signal for refreshing user data.
    *
    */
    void refreshUserData();

    /**
    * @brief Signal for requesting reverseGeo from SituareEngine
    *
    */
    void requestReverseGeo();

    /**
    * @brief Signals, when address data is ready
    *
    * @param address Street address
    */
    void reverseGeoReady(const QString &address);

    /**
    * @brief Signal to save username to settings
    *
    * @param username Username
    */
    void saveUsername(const QString &username);

    /**
    * @brief Signal for requestLocationUpdate from SituareEngine
    *
    * @param status Status message
    * @param publish Publish on Facebook
    */
    void statusUpdate(const QString &status, const bool &publish);

    /**
    * @brief Signals when webview's urlChanged signal is emitted
    *
    * @param url New url
    */
    void updateCredentials(const QUrl &url);

    /**
    * @brief Signals when updateLocationDialog's data must be cleared
    *
    */
    void clearUpdateLocationDialogData();

    /**
    * @brief MapView has finished zooming
    */
    void viewZoomFinished();

    /**
    * @brief Signal for use location ready.
    *
    * @param user User object
    */
    void userLocationReady(User *user);

    /**
    * @brief Map zoom in request
    */
    void zoomIn();

    /**
      * @brief Forwarding signal from MapEngine to MapView
      */
    void zoomLevelChanged(int zoomLevel);

    /**
    * @brief Map zoom out request
    */
    void zoomOut();

/*******************************************************************************
 * DATA MEMBERS
 ******************************************************************************/
private:
    bool m_drawOwnLocationCrosshair;        ///< Flag for making ownLocationCrosshair visible or not
    bool m_errorShown;                      ///< Indicates if error dialog/note is shown
    bool m_loggedIn;                        ///< Indicates login state
    bool m_refresh;                         ///< Indicates when webpage is refreshed

    int m_progressIndicatorCount;           ///< Indicates the number of progress indicator calls

    QAction *m_autoCenteringAct;            ///< Action to auto center map using gps position
    QAction *m_gpsToggleAct;                ///< Action to trigger gps toggle
    QAction *m_loginAct;                    ///< Action to Login/Logout
    QAction *m_toSettingsAct;               ///< Action to trigger switch to settings dialog

    QLabel *m_osmLicense;                   ///< Label for Open Street Map license
    QLabel *m_ownLocationCrosshair;         ///< Label that show ownLocationCrosshair

    QList<QDialog *> m_error_queue;         ///< QList type error dialog queue
    QList<QDialog *> m_queue;               ///< QList type dialog queue

    QMenu *m_viewMenu;                      ///< Object that hold the view menu items

    QMessageBox *m_automaticUpdateLocationDialog;   ///< Automatic update location dialog

    QSize m_viewPortSize;                 ///< Size of the viewport

    QString m_email;                        ///< Placeholder for email
    QString m_password;                     ///< Placeholder for password

    FullScreenButton *m_fullScreenButton;   ///< Instance of the fullscreen toggle button

    QWebView *m_webView;                    ///< Shows facebook login page

    FriendListPanel *m_friendsListPanel;    ///< Instance of friends list panel
    MapScale *m_mapScale;                   ///< Instance of the map scale
    MapView *m_mapView;                     ///< Instance of the map view
    NetworkCookieJar *m_cookieJar;          ///< Placeholder for QNetworkCookies
    PanelSideBar *m_userPanelSidebar;       ///< User panel side bar
    PanelSideBar *m_friendsListPanelSidebar;///< Friends panel side bar
    UserInfoPanel *m_userPanel;             ///< Instance of the user information panel
    ZoomButtonPanel *m_zoomButtonPanel;     ///< Instance of zoom button panel
};

#endif // MAINWINDOW_H
