/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

      Henri Lampela - henri.lampela@ixonos.com
      Kaj Wallin - kaj.wallin@ixonos.com
      Jussi Laitinen jussi.laitinen@ixonos.com
      Sami Rämö - sami.ramo@ixonos.com
      Ville Tiensuu - ville.tiensuu@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#include <QtGui>
#include <QtWebKit>
#include <QtAlgorithms>

#include "facebookservice/facebookauthentication.h"
#include "map/mapcommon.h"
#include "map/mapview.h"
#include "common.h"
#include "friendlistpanel.h"
#include "fullscreenbutton.h"
#include "logindialog.h"
#include "mapscale.h"
#include "settingsdialog.h"
#include "userinfopanel.h"
#include "zoombuttonpanel.h"

#include "mainwindow.h"

// These MUST BE HERE, compiling for Maemo fails if moved
#ifdef Q_WS_MAEMO_5
#include <QtMaemo5/QMaemo5InformationBox>
#include <QtGui/QX11Info>
#include <X11/Xatom.h>
#include <X11/Xlib.h>
#endif // Q_WS_MAEMO_5

MainWindow::MainWindow(QWidget *parent)
    : QMainWindow(parent),
    m_drawOwnLocationCrosshair(false),
    m_errorShown(false),
    m_loggedIn(false),
    m_refresh(false),
    m_progressIndicatorCount(0),
    m_ownLocationCrosshair(0),
    m_email(),    
    m_password(),
    m_fullScreenButton(0),
    m_webView(0),
    m_mapScale(0),
    m_cookieJar(0)
{
    qDebug() << __PRETTY_FUNCTION__;

    buildMap();

    // build main layout
    QHBoxLayout *layout = new QHBoxLayout;
    layout->addWidget(m_mapView);
    layout->setMargin(0);
    layout->setSpacing(0);

    setCentralWidget(new QWidget());
    centralWidget()->setLayout(layout);

    buildFriendListPanel();
    buildUserInfoPanel();

    createMenus();
    setWindowTitle(tr("Situare"));

    // set stacking order of widgets
    m_zoomButtonPanel->stackUnder(m_userPanel);
    if(m_fullScreenButton) {
        m_fullScreenButton->stackUnder(m_zoomButtonPanel);
        m_osmLicense->stackUnder(m_fullScreenButton);
    } else {
        m_osmLicense->stackUnder(m_zoomButtonPanel);
    }
    m_ownLocationCrosshair->stackUnder(m_osmLicense);
    m_mapScale->stackUnder(m_ownLocationCrosshair);
    m_mapView->stackUnder(m_mapScale);

    grabZoomKeys(true);

    // Set default screen size
    resize(DEFAULT_SCREEN_WIDTH, DEFAULT_SCREEN_HEIGHT);
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5StackedWindow);
#endif
}

MainWindow::~MainWindow()
{
    qDebug() << __PRETTY_FUNCTION__;

    grabZoomKeys(false);

    if(m_webView)
        delete m_webView;

    qDeleteAll(m_queue.begin(), m_queue.end());
    m_queue.clear();

    qDeleteAll(m_error_queue.begin(), m_error_queue.end());
    m_error_queue.clear();
}

void MainWindow::automaticUpdateDialogFinished(int result)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (result == QMessageBox::Yes) {
        readAutomaticLocationUpdateSettings();
    } else {
        QSettings settings(DIRECTORY_NAME, FILE_NAME);
        settings.setValue(SETTINGS_AUTOMATIC_UPDATE_ENABLED, false);
        readAutomaticLocationUpdateSettings();
    }

    m_automaticUpdateLocationDialog->deleteLater();
}

void MainWindow::buildFullScreenButton()
{
    qDebug() << __PRETTY_FUNCTION__;
#ifdef Q_WS_MAEMO_5
    m_fullScreenButton = new FullScreenButton(this);
    connect(m_fullScreenButton, SIGNAL(clicked()),
            this, SLOT(toggleFullScreen()));
#endif // Q_WS_MAEMO_5
}

void MainWindow::buildFriendListPanel()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_friendsListPanel = new FriendListPanel(this);

    m_friendsListPanelSidebar = new PanelSideBar(this, RIGHT);

    m_friendsListPanel->stackUnder(m_friendsListPanelSidebar);

    connect(this, SIGNAL(friendsLocationsReady(QList<User*>&)),
            m_friendsListPanel, SLOT(friendInfoReceived(QList<User*>&)));

    connect(this, SIGNAL(locationItemClicked(QList<QString>)),
            m_friendsListPanel, SLOT(showFriendsInList(QList<QString>)));

    connect(m_mapView, SIGNAL(viewResized(QSize)),
            m_friendsListPanel, SLOT(resizePanel(QSize)));

    connect(m_mapView, SIGNAL(viewResized(QSize)),
            m_friendsListPanelSidebar, SLOT(resizeSideBar(QSize)));

    connect(m_friendsListPanel, SIGNAL(findFriend(QPointF)),
            this, SIGNAL(findFriend(QPointF)));
}

void MainWindow::buildInformationBox(const QString &message, bool modal)
{
    qDebug() << __PRETTY_FUNCTION__;

    QString errorMessage = message;

#ifdef Q_WS_MAEMO_5

    QMaemo5InformationBox *msgBox = new QMaemo5InformationBox(this);

    if(modal) {
        msgBox->setTimeout(QMaemo5InformationBox::NoTimeout);
        // extra line changes are needed to make error notes broader
        errorMessage.prepend("\n");
        errorMessage.append("\n");
    } else {
        msgBox->setTimeout(QMaemo5InformationBox::DefaultTimeout);
    }
    QLabel *label = new QLabel(msgBox);
    label->setAlignment(Qt::AlignCenter);
    label->setText(errorMessage);
    msgBox->setWidget(label);
#else
    QMessageBox *msgBox = new QMessageBox(this);
    msgBox->button(QMessageBox::Ok);
    msgBox->setText(errorMessage);
    msgBox->setModal(modal);
#endif

    queueDialog(msgBox);
}

void MainWindow::buildManualLocationCrosshair()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_ownLocationCrosshair = new QLabel(this);
    QPixmap crosshairImage(":/res/images/sight.png");
    m_ownLocationCrosshair->setPixmap(crosshairImage);
    m_ownLocationCrosshair->setFixedSize(crosshairImage.size());
    m_ownLocationCrosshair->hide();
    m_ownLocationCrosshair->setAttribute(Qt::WA_TransparentForMouseEvents, true);

    connect(m_mapView, SIGNAL(viewResized(QSize)),
            this, SLOT(drawOwnLocationCrosshair(QSize)));
}

void MainWindow::buildMap()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_mapView = new MapView(this);

    buildZoomButtonPanel();
    buildOsmLicense();
    buildManualLocationCrosshair();
    buildFullScreenButton();
    buildMapScale();

    connect(m_mapView, SIGNAL(viewScrolled(QPoint)),
            this, SIGNAL(mapViewScrolled(QPoint)));

    connect(this, SIGNAL(centerToSceneCoordinates(QPoint)),
            m_mapView, SLOT(centerToSceneCoordinates(QPoint)));

    connect(m_mapView, SIGNAL(viewResized(QSize)),
            this, SIGNAL(mapViewResized(QSize)));

    connect(m_mapView, SIGNAL(viewResized(QSize)),
            this, SLOT(drawFullScreenButton(QSize)));

    connect(m_mapView, SIGNAL(viewResized(QSize)),
            this, SLOT(drawMapScale(QSize)));

    connect(m_mapView, SIGNAL(viewResized(QSize)),
             this, SLOT(setViewPortSize(QSize)));

    connect(this, SIGNAL(zoomLevelChanged(int)),
            m_mapView, SLOT(setZoomLevel(int)));

    connect(m_mapView, SIGNAL(viewZoomFinished()),
            this, SIGNAL(viewZoomFinished()));

    connect(m_mapView, SIGNAL(zoomIn()),
            this, SIGNAL(zoomIn()));
}

void MainWindow::buildMapScale()
{
    m_mapScale = new MapScale(this);
    connect(this, SIGNAL(newMapResolution(qreal)),
            m_mapScale, SLOT(updateMapResolution(qreal)));
}

void MainWindow::buildOsmLicense()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_osmLicense = new QLabel(this);
    m_osmLicense->setAttribute(Qt::WA_TranslucentBackground, true);
    m_osmLicense->setAttribute(Qt::WA_TransparentForMouseEvents, true);
    m_osmLicense->setText("<font color='black'>" + OSM_LICENSE + "</font>");
    m_osmLicense->setFont(QFont("Nokia Sans", 9));
    m_osmLicense->resize(m_osmLicense->fontMetrics().width(OSM_LICENSE),
                         m_osmLicense->fontMetrics().height());

    connect(m_mapView, SIGNAL(viewResized(QSize)),
            this, SLOT(drawOsmLicense(QSize)));
}

void MainWindow::buildUserInfoPanel()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_userPanel = new UserInfoPanel(this);

    m_userPanelSidebar = new PanelSideBar(this, LEFT);

    m_userPanelSidebar->stackUnder(m_friendsListPanel);
    m_userPanel->stackUnder(m_userPanelSidebar);

    connect(this, SIGNAL(userLocationReady(User*)),
            m_userPanel, SLOT(userDataReceived(User*)));

    connect(this, SIGNAL(reverseGeoReady(QString)),
            m_userPanel, SIGNAL(reverseGeoReady(QString)));

    connect(this, SIGNAL(clearUpdateLocationDialogData()),
            m_userPanel, SIGNAL(clearUpdateLocationDialogData()));

    connect(this, SIGNAL(messageSendingFailed(int)),
            m_userPanel, SIGNAL(messageSendingFailed(int)));

    connect(m_mapView, SIGNAL(viewResized(QSize)),
            m_userPanel, SLOT(resizePanel(QSize)));

    connect(m_mapView, SIGNAL(viewResized(QSize)),
            m_userPanelSidebar, SLOT(resizeSideBar(QSize)));

    connect(m_userPanel, SIGNAL(findUser(QPointF)),
            this, SIGNAL(findUser(QPointF)));

    connect(m_userPanel, SIGNAL(requestReverseGeo()),
            this, SIGNAL(requestReverseGeo()));

    connect(m_userPanel, SIGNAL(statusUpdate(QString,bool)),
            this, SIGNAL(statusUpdate(QString,bool)));

    connect(m_userPanel, SIGNAL(refreshUserData()),
            this, SIGNAL(refreshUserData()));

    connect(m_userPanel, SIGNAL(notificateUpdateFailing(QString, bool)),
            this, SLOT(buildInformationBox(QString, bool)));
}

void MainWindow::buildWebView()
{
    qDebug() << __PRETTY_FUNCTION__;

    if(!m_webView) {
        m_webView = new QWebView;

        if(!m_cookieJar)
            m_cookieJar = new NetworkCookieJar(new QNetworkCookieJar(this));

        m_webView->page()->networkAccessManager()->setCookieJar(m_cookieJar);

        connect(m_webView->page()->networkAccessManager(), SIGNAL(finished(QNetworkReply*)),
                this, SLOT(webViewRequestFinished(QNetworkReply*)));
        connect(m_webView, SIGNAL(urlChanged(const QUrl &)),
                this, SIGNAL(updateCredentials(QUrl)));
        connect(m_webView, SIGNAL(loadFinished(bool)),
                this, SLOT(loadDone(bool)));

        m_webView->hide();
    }
}

void MainWindow::buildZoomButtonPanel()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_zoomButtonPanel = new ZoomButtonPanel(this);

    connect(m_zoomButtonPanel->zoomInButton(), SIGNAL(clicked()),
            this, SIGNAL(zoomIn()));

    connect(m_zoomButtonPanel->zoomOutButton(), SIGNAL(clicked()),
            this, SIGNAL(zoomOut()));

    connect(this, SIGNAL(zoomLevelChanged(int)),
            m_zoomButtonPanel, SLOT(resetButtons()));

    connect(this, SIGNAL(maxZoomLevelReached()),
            m_zoomButtonPanel, SLOT(disableZoomInButton()));

    connect(this, SIGNAL(minZoomLevelReached()),
            m_zoomButtonPanel, SLOT(disableZoomOutButton()));

    connect(m_mapView, SIGNAL(viewResized(QSize)),
            m_zoomButtonPanel, SLOT(screenResized(QSize)));
}

void MainWindow::clearCookieJar()
{
    qDebug() << __PRETTY_FUNCTION__;

    buildWebView();

    m_webView->stop();

    if(!m_cookieJar) {
        m_cookieJar = new NetworkCookieJar(new QNetworkCookieJar(this));
    }
    QList<QNetworkCookie> emptyList;
    emptyList.clear();

    m_cookieJar->setAllCookies(emptyList);
    m_webView->page()->networkAccessManager()->setCookieJar(m_cookieJar);
}

void MainWindow::createMenus()
{
    qDebug() << __PRETTY_FUNCTION__;

    // login/logout
    m_loginAct = new QAction(tr("Login"), this);
    connect(m_loginAct, SIGNAL(triggered()),
            this, SIGNAL(loginActionPressed()));

    // settings
    m_toSettingsAct = new QAction(tr("Settings"), this);
    connect(m_toSettingsAct, SIGNAL(triggered()),
        this, SLOT(openSettingsDialog()));

    // GPS
    m_gpsToggleAct = new QAction(tr("GPS"), this);
    m_gpsToggleAct->setCheckable(true);

    connect(m_gpsToggleAct, SIGNAL(triggered(bool)),
            this, SIGNAL(gpsTriggered(bool)));

    // automatic centering
    m_autoCenteringAct = new QAction(tr("Auto centering"), this);
    m_autoCenteringAct->setCheckable(true);
    connect(m_autoCenteringAct, SIGNAL(triggered(bool)),
        this, SIGNAL(autoCenteringTriggered(bool)));

    // build the actual menu
    m_viewMenu = menuBar()->addMenu(tr("Main"));
    m_viewMenu->addAction(m_loginAct);
    m_viewMenu->addAction(m_toSettingsAct);
    m_viewMenu->addAction(m_gpsToggleAct);
    m_viewMenu->addAction(m_autoCenteringAct);
    m_viewMenu->setObjectName(tr("Menu"));
}

void MainWindow::dialogFinished(int status)
{
    qDebug() << __PRETTY_FUNCTION__;

    QDialog *dialog = m_queue.takeFirst();
    LoginDialog *loginDialog = qobject_cast<LoginDialog *>(dialog);
    if(loginDialog) {
        if(status != 0) {
            buildWebView();
            loginDialog->userInput(m_email, m_password);

            QStringList urlParts;
            urlParts.append(FACEBOOK_LOGINBASE);
            urlParts.append(SITUARE_PUBLIC_FACEBOOKAPI_KEY);
            urlParts.append(INTERVAL1);
            urlParts.append(SITUARE_LOGIN_SUCCESS);
            urlParts.append(INTERVAL2);
            urlParts.append(SITUARE_LOGIN_FAILURE);
            urlParts.append(FACEBOOK_LOGIN_ENDING);

            emit saveUsername(m_email);
            m_refresh = true;
            m_webView->load(QUrl(urlParts.join(EMPTY)));
            toggleProgressIndicator(true);
        } else {
            emit cancelLoginProcess();
        }
    }

    dialog->deleteLater();

    if(!m_error_queue.isEmpty() && m_errorShown == false) {
        showErrorInformationBox();
    } else {
        if(!m_queue.isEmpty()) {
            showInformationBox();
        }
    }
}

void MainWindow::drawFullScreenButton(const QSize &size)
{
    qDebug() << __PRETTY_FUNCTION__ << size.width() << "x" << size.height();

    if(m_fullScreenButton) {
        if(m_loggedIn) {
            m_fullScreenButton->move(size.width() - m_fullScreenButton->size().width()
                                     - PANEL_PEEK_AMOUNT,
                                     size.height() - m_fullScreenButton->size().height());
        } else {
            m_fullScreenButton->move(size.width() - m_fullScreenButton->size().width(),
                                     size.height() - m_fullScreenButton->size().height());
        }
    }
}

void MainWindow::drawMapScale(const QSize &size)
{
    const int LEFT_SCALE_MARGIN = 10;
    const int BOTTOM_SCALE_MARGIN = 2;
    qDebug() << __PRETTY_FUNCTION__ << size.width() << "x" << size.height();

    m_mapScale->move(PANEL_PEEK_AMOUNT + LEFT_SCALE_MARGIN,
                     size.height() - m_mapScale->size().height() - BOTTOM_SCALE_MARGIN);
}

void MainWindow::drawOsmLicense(const QSize &size)
{
    qDebug() << __PRETTY_FUNCTION__ << size.width() << "x" << size.height();

    m_osmLicense->move(size.width() - m_osmLicense->fontMetrics().width(OSM_LICENSE)
                       - PANEL_PEEK_AMOUNT,
                       size.height() - m_osmLicense->fontMetrics().height());
}

void MainWindow::drawOwnLocationCrosshair(const QSize &size)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_drawOwnLocationCrosshair && m_ownLocationCrosshair != 0) {
        m_ownLocationCrosshair->move(size.width()/2 - m_ownLocationCrosshair->pixmap()->width()/2,
                            size.height()/2 - m_ownLocationCrosshair->pixmap()->height()/2);
    }
}

void MainWindow::errorDialogFinished(int status)
{
    qDebug() << __PRETTY_FUNCTION__;

    qDebug() << status;
    QDialog *dialog = m_error_queue.takeFirst();

    dialog->deleteLater();
    m_errorShown = false;

    if(!m_error_queue.isEmpty())
        showErrorInformationBox();
    else if(!m_queue.isEmpty())
        showInformationBox();
}

void MainWindow::gpsTimeout()
{
    qDebug() << __PRETTY_FUNCTION__;

    buildInformationBox(tr("GPS timeout"));
}

void MainWindow::grabZoomKeys(bool grab)
{
    qDebug() << __PRETTY_FUNCTION__;

#ifdef Q_WS_MAEMO_5
    // Can't grab keys unless we have a window id
    if (!winId())
        return;

    unsigned long val = (grab) ? 1 : 0;
    Atom atom = XInternAtom(QX11Info::display(), "_HILDON_ZOOM_KEY_ATOM", False);
    if (!atom)
        return;

    XChangeProperty (QX11Info::display(),
                     winId(),
                     atom,
                     XA_INTEGER,
                     32,
                     PropModeReplace,
                     reinterpret_cast<unsigned char *>(&val),
                     1);
#else
    Q_UNUSED(grab);
#endif // Q_WS_MAEMO_5
}

void MainWindow::keyPressEvent(QKeyEvent* event)
{
    qDebug() << __PRETTY_FUNCTION__;

    switch (event->key()) {
    case Qt::Key_F7:
        event->accept();
        emit zoomIn();
        break;

    case Qt::Key_F8:
        event->accept();
        emit zoomOut();
        break;
    }
    QWidget::keyPressEvent(event);
}

void MainWindow::loadCookies()
{
    qDebug() << __PRETTY_FUNCTION__;

    QSettings settings(DIRECTORY_NAME, FILE_NAME);

    QStringList list = settings.value(COOKIES, EMPTY).toStringList();

    if(!list.isEmpty()) {
        QList<QNetworkCookie> cookieList;
        for(int i=0;i<list.count();i++) {
            cookieList.append(QNetworkCookie::parseCookies(list.at(i).toAscii()));
        }

        if(!m_cookieJar)
               m_cookieJar = new NetworkCookieJar(new QNetworkCookieJar(this));

        m_cookieJar->setAllCookies(cookieList);
        m_webView->page()->networkAccessManager()->setCookieJar(m_cookieJar);

    }
}

void MainWindow::loadDone(bool done)
{
    qDebug() << __PRETTY_FUNCTION__;

    // for the first time the login page is opened, we need to refresh it to get cookies working
    if(m_refresh) {
        m_webView->reload();
        m_refresh = false;
    }

    if (done)
    {
        QWebFrame* frame = m_webView->page()->currentFrame();
        if (frame!=NULL)
        {
            // set email box
            QWebElementCollection emailCollection = frame->findAllElements("input[name=email]");

            foreach (QWebElement element, emailCollection) {
                element.setAttribute("value", m_email.toAscii());
            }
            // set password box
            QWebElementCollection passwordCollection = frame->findAllElements("input[name=pass]");
            foreach (QWebElement element, passwordCollection) {
                element.setAttribute("value", m_password.toAscii());
            }
            // find connect button
            QWebElementCollection buttonCollection = frame->findAllElements("input[name=login]");
            foreach (QWebElement element, buttonCollection)
            {
                QPoint pos(element.geometry().center());

                // send a mouse click event to the web page
                QMouseEvent event0(QEvent::MouseButtonPress, pos, Qt::LeftButton, Qt::LeftButton,
                                   Qt::NoModifier);
                QApplication::sendEvent(m_webView->page(), &event0);
                QMouseEvent event1(QEvent::MouseButtonRelease, pos, Qt::LeftButton, Qt::LeftButton,
                                   Qt::NoModifier);
                QApplication::sendEvent(m_webView->page(), &event1);
            }
        }
    }
}

void MainWindow::loggedIn(bool logged)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_loggedIn = logged;

    if(logged) {
        m_loginAct->setText(tr("Logout"));
    } else {
        clearCookieJar();
        m_email.clear();
        m_password.clear();

        m_loginAct->setText(tr("Login"));
    }
    updateItemVisibility();
}

void MainWindow::loginFailed()
{
    qDebug() << __PRETTY_FUNCTION__;

    clearCookieJar();
    startLoginProcess();
}

bool MainWindow::loginState()
{
    qDebug() << __PRETTY_FUNCTION__;

    return m_loggedIn;
}

void MainWindow::loginUsingCookies()
{
    qDebug() << __PRETTY_FUNCTION__;

    toggleProgressIndicator(true);

    buildWebView();
    loadCookies();
    
    QStringList urlParts;
    urlParts.append(FACEBOOK_LOGINBASE);
    urlParts.append(SITUARE_PUBLIC_FACEBOOKAPI_KEY);
    urlParts.append(INTERVAL1);
    urlParts.append(SITUARE_LOGIN_SUCCESS);
    urlParts.append(INTERVAL2);
    urlParts.append(SITUARE_LOGIN_FAILURE);
    urlParts.append(FACEBOOK_LOGIN_ENDING);

    m_webView->load(QUrl(urlParts.join(EMPTY)));

}

void MainWindow::openSettingsDialog()
{
    qDebug() << __PRETTY_FUNCTION__;

    SettingsDialog *settingsDialog = new SettingsDialog(this);
    settingsDialog->enableSituareSettings((m_loggedIn && m_gpsToggleAct->isChecked()));
    connect(settingsDialog, SIGNAL(accepted()), this, SLOT(settingsDialogAccepted()));

    settingsDialog->show();
}

void MainWindow::readAutomaticLocationUpdateSettings()
{
    qDebug() << __PRETTY_FUNCTION__;

    QSettings settings(DIRECTORY_NAME, FILE_NAME);
    bool automaticUpdateEnabled = settings.value(SETTINGS_AUTOMATIC_UPDATE_ENABLED, false).toBool();
    QTime automaticUpdateInterval = settings.value(SETTINGS_AUTOMATIC_UPDATE_INTERVAL, QTime())
                                      .toTime();

    if (automaticUpdateEnabled && automaticUpdateInterval.isValid()) {
        QTime time;
        emit enableAutomaticLocationUpdate(true, time.msecsTo(automaticUpdateInterval));
    } else {
        emit enableAutomaticLocationUpdate(false);
    }
}

void MainWindow::queueDialog(QDialog *dialog)
{
    qDebug() << __PRETTY_FUNCTION__;

    // check is dialog is modal, for now all modal dialogs have hihger priority i.e. errors
    if(dialog->isModal()) {
        m_error_queue.append(dialog);
    } else {
        m_queue.append(dialog);
    }

    // show error dialog if there is only one error dialog in the queue and no error dialog is shown
    if(m_error_queue.count() == 1 && m_errorShown == false)
        showErrorInformationBox();
    else if(m_queue.count() == 1 && m_errorShown == false)
        showInformationBox();
}

void MainWindow::saveCookies()
{
    qDebug() << __PRETTY_FUNCTION__;

    if(!m_cookieJar)
        m_cookieJar = new NetworkCookieJar(new QNetworkCookieJar(this));

    QList<QNetworkCookie> cookieList = m_cookieJar->allCookies();
    QStringList list;

    for(int i=0;i<cookieList.count();i++) {
        QNetworkCookie cookie = cookieList.at(i);
        QByteArray byteArray = cookie.toRawForm(QNetworkCookie::Full);
        list.append(QString(byteArray));
    }
    list.removeDuplicates();

    QSettings settings(DIRECTORY_NAME, FILE_NAME);
    settings.setValue(COOKIES, list);
}

void MainWindow::setAutoCenteringButtonEnabled(bool enabled)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_autoCenteringAct->setChecked(enabled);
}

void MainWindow::setGPSButtonEnabled(bool enabled)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_gpsToggleAct->setChecked(enabled);

    setOwnLocationCrosshairVisibility(!enabled);

    m_autoCenteringAct->setVisible(enabled);
}

void MainWindow::setMapViewScene(QGraphicsScene *scene)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_mapView->setScene(scene);
}

void MainWindow::setOwnLocationCrosshairVisibility(bool visibility)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (visibility && m_loggedIn) {
        m_ownLocationCrosshair->show();
        m_drawOwnLocationCrosshair = true;
        drawOwnLocationCrosshair(m_viewPortSize);
    } else {
        m_ownLocationCrosshair->hide();
        m_drawOwnLocationCrosshair = false;
    }
}

void MainWindow::settingsDialogAccepted()
{
    qDebug() << __PRETTY_FUNCTION__;

    readAutomaticLocationUpdateSettings();
}

void MainWindow::setUsername(const QString &username)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_email = username;
}

void MainWindow::setViewPortSize(const QSize &size)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_viewPortSize = size;
}

void MainWindow::showEnableAutomaticUpdateLocationDialog(const QString &text)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_automaticUpdateLocationDialog = new QMessageBox(QMessageBox::Question,
                                                      tr("Automatic location update"), text,
                                                      QMessageBox::Yes | QMessageBox::No |
                                                      QMessageBox::Cancel, this);
    connect(m_automaticUpdateLocationDialog, SIGNAL(finished(int)),
            this, SLOT(automaticUpdateDialogFinished(int)));

    m_automaticUpdateLocationDialog->show();
}

void MainWindow::toggleFullScreen()
{
    qDebug() << __PRETTY_FUNCTION__;

    if(windowState() == Qt::WindowNoState)
        showFullScreen();
    else
        showNormal();
}

void MainWindow::showErrorInformationBox()
{
    qDebug() << __PRETTY_FUNCTION__;

    if(m_error_queue.count()) {
        m_errorShown = true;
        QDialog *dialog = m_error_queue.first();
        connect(dialog, SIGNAL(finished(int)),
                this, SLOT(errorDialogFinished(int)));
        dialog->show();
    }
}

void MainWindow::showInformationBox()
{
    qDebug() << __PRETTY_FUNCTION__;

    if(m_queue.count()) {
        QDialog *dialog = m_queue.first();
        connect(dialog, SIGNAL(finished(int)),
                this, SLOT(dialogFinished(int)));
        dialog->show();
    }
}

void MainWindow::showPanels()
{
    qDebug() << __PRETTY_FUNCTION__;

    drawFullScreenButton(m_viewPortSize);

    if(m_loggedIn) {
        if(!m_friendsListPanel->isVisible()) {
            m_friendsListPanel->show();
            m_friendsListPanelSidebar->show();
        }

        if(!m_userPanel->isVisible()) {
            m_userPanel->show();
            m_userPanelSidebar->show();
        }
    }
}

void MainWindow::startLoginProcess()
{
    qDebug() << __PRETTY_FUNCTION__;

    LoginDialog *loginDialog = new LoginDialog();

    emit fetchUsernameFromSettings();

    loginDialog->clearTextFields();

    if(!m_email.isEmpty())
        loginDialog->setEmailField(m_email);

    queueDialog(loginDialog);
}

void MainWindow::toggleProgressIndicator(bool value)
{
    qDebug() << __PRETTY_FUNCTION__;

#ifdef Q_WS_MAEMO_5
    if(value) {
        m_progressIndicatorCount++;
        setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
    } else {
        if(m_progressIndicatorCount > 0)
            m_progressIndicatorCount--;

        if(m_progressIndicatorCount == 0)
            setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
    }
#else
    Q_UNUSED(value);
#endif // Q_WS_MAEMO_5
}

void MainWindow::updateItemVisibility()
{
    qDebug() << __PRETTY_FUNCTION__;
    
    if(m_loggedIn) {       

        if(!m_gpsToggleAct->isChecked())
            setOwnLocationCrosshairVisibility(true);
    } else {
        m_friendsListPanel->closePanel();
        m_friendsListPanel->hide();
        m_friendsListPanelSidebar->hide();
        m_userPanel->closePanel();
        m_userPanel->hide();
        m_userPanelSidebar->hide();
        setOwnLocationCrosshairVisibility(false);
    }
}

const QString MainWindow::username()
{
    qDebug() << __PRETTY_FUNCTION__;
    
    return m_email;
}

void MainWindow::webViewRequestFinished(QNetworkReply *reply)
{
    qDebug() << __PRETTY_FUNCTION__;

    // omit QNetworkReply::OperationCanceledError due to it's nature to be called when ever
    // qwebview starts to load a new page while the current page loading is not finished
    if(reply->error() != QNetworkReply::OperationCanceledError &&
       reply->error() != QNetworkReply::NoError) {
        emit error(ErrorContext::NETWORK, reply->error());
    }
}

