/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

      Henri Lampela - henri.lampela@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#include <QtAlgorithms>
#include <QDebug>
#include <QtGlobal>
#include <QStringList>
#include <QPixmap>
#include <QNetworkReply>
#include "situareservice.h"
#include "situarecommon.h"
#include "common.h"
#include "parser.h"
#include "ui/avatarimage.h"
#include "network/networkaccessmanager.h"

SituareService::SituareService(QObject *parent)
        : QObject(parent),
        m_user(0)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_networkManager = NetworkAccessManager::instance();
    connect(m_networkManager, SIGNAL(finished(QNetworkReply*)),
            this, SLOT(requestFinished(QNetworkReply*)), Qt::QueuedConnection);

    m_imageFetcher = new ImageFetcher(NetworkAccessManager::instance(), this);
    connect(this, SIGNAL(fetchImage(QUrl)),
            m_imageFetcher, SLOT(fetchImage(QUrl)));
    connect(m_imageFetcher, SIGNAL(imageReceived(QUrl,QPixmap)),
            this, SLOT(imageReceived(QUrl, QPixmap)));
    connect(m_imageFetcher, SIGNAL(error(int, int)),
            this, SIGNAL(error(int, int)));
}

SituareService::~SituareService()
{
    qDebug() << __PRETTY_FUNCTION__;

    if(m_user) {
        delete m_user;
        m_user = 0;
    }

    qDeleteAll(m_friendsList.begin(), m_friendsList.end());
    m_friendsList.clear();
}

void SituareService::fetchLocations()
{
    qDebug() << __PRETTY_FUNCTION__;

    QString cookie = formCookie(API_KEY, m_credentials.expires(), m_credentials.userID(),
                                m_credentials.sessionKey(), m_credentials.sessionSecret(),
                                m_credentials.sig(), EN_LOCALE);

    QUrl url = formUrl(SITUARE_URL, GET_LOCATIONS);
    sendRequest(url, COOKIE, cookie);
}

void SituareService::reverseGeo(const QPointF &coordinates)
{
    qDebug() << __PRETTY_FUNCTION__;

    QString cookie = formCookie(API_KEY, m_credentials.expires(),m_credentials.userID(),
                                m_credentials.sessionKey(), m_credentials.sessionSecret(),
                                m_credentials.sig(), EN_LOCALE);

    QString urlParameters = formUrlParameters(coordinates);
    urlParameters.append(JSON_FORMAT);
    QUrl url = formUrl(SITUARE_URL, REVERSE_GEO, urlParameters);

    sendRequest(url, COOKIE, cookie);
}

void SituareService::updateLocation(const QPointF &coordinates, const QString &status,
                                    const bool &publish)
{
    qDebug() << __PRETTY_FUNCTION__;

    QString cookie = formCookie(API_KEY, m_credentials.expires(), m_credentials.userID(),
                                m_credentials.sessionKey(), m_credentials.sessionSecret(),
                                m_credentials.sig(), EN_LOCALE);


    QString publishValue;
    if(publish) {
        publishValue = PUBLISH_TRUE;
    }
    else {
        publishValue = PUBLISH_FALSE;
    }
    QString urlParameters = formUrlParameters(coordinates, status, publishValue);
    QUrl url = formUrl(SITUARE_URL, UPDATE_LOCATION, urlParameters);

    sendRequest(url, COOKIE, cookie);
}

QString SituareService::formCookie(const QString &apiKeyValue, QString expiresValue,
                                   QString userValue, QString sessionKeyValue,
                                   QString sessionSecretValue, const QString &signatureValue,
                                   const QString &localeValue)
{
    qDebug() << __PRETTY_FUNCTION__;

    QString cookie;
    QString apiKey;
    QString user;
    QString expires;
    QString sessionKey;
    QString sessionSecret;
    QString locale;
    QString variable;
    QString signature = EQUAL_MARK;
    QStringList variableList;

    signature.append(signatureValue);
    apiKey.append(apiKeyValue);
    apiKey.append(UNDERLINE_MARK);

    user.append(USER);
    user.append(EQUAL_MARK);
    expires.append(EXPIRES);
    expires.append(EQUAL_MARK);
    sessionKey.append(SESSION_KEY);
    sessionKey.append(EQUAL_MARK);
    sessionSecret.append(SESSION_SECRET);
    sessionSecret.append(EQUAL_MARK);
    locale.append(LOCALE);
    locale.append(EQUAL_MARK);
    locale.append(localeValue);

    variableList.append(expires.append(expiresValue.append(BREAK_MARK)));
    variableList.append(sessionKey.append(sessionKeyValue.append(BREAK_MARK)));
    variableList.append(user.append(userValue).append(BREAK_MARK));
    variableList.append(sessionSecret.append(sessionSecretValue.append(BREAK_MARK)));

    cookie.append(BREAK_MARK);

    foreach(variable, variableList) {
        cookie.append(apiKey);
        cookie.append(variable);
    }
    apiKey.remove(UNDERLINE_MARK);
    cookie.append(apiKey);
    cookie.append(signature);
    cookie.append(BREAK_MARK);
    cookie.append(locale);

    qDebug() << cookie;

    return cookie;
}

QUrl SituareService::formUrl(const QString &baseUrl, const QString &phpScript,
                             QString urlParameters)
{
    qDebug() << __PRETTY_FUNCTION__;
    QString urlString;

    urlString.append(baseUrl);
    urlString.append(phpScript);
    if(!urlParameters.isEmpty())
        urlString.append(urlParameters);

    QUrl url = QUrl(urlString);

    qDebug() << url;

    return url;
}

QString SituareService::formUrlParameters(const QPointF &coordinates, QString status,
                                          QString publish)
{
    QString parameters;

    parameters.append(QUESTION_MARK);
    parameters.append(LATITUDE);
    parameters.append(EQUAL_MARK);
    parameters.append(QString::number(coordinates.y()));
    parameters.append(AMBERSAND_MARK);
    parameters.append(LONGTITUDE);
    parameters.append(EQUAL_MARK);
    parameters.append(QString::number(coordinates.x()));

    if(publish.compare(PUBLISH_TRUE) == 0) {
        parameters.append(AMBERSAND_MARK);
        parameters.append(PUBLISH);
        parameters.append(EQUAL_MARK);
        parameters.append(PUBLISH_TRUE);
    } else if(publish.compare(PUBLISH_FALSE) == 0) {
        parameters.append(AMBERSAND_MARK);
        parameters.append(PUBLISH);
        parameters.append(EQUAL_MARK);
        parameters.append(PUBLISH_FALSE);
    }

    if(!status.isEmpty()) {
        parameters.append(AMBERSAND_MARK);
        parameters.append(DATA);
        parameters.append(EQUAL_MARK);
        parameters.append(status);
    }

    return parameters;
}

void SituareService::sendRequest(const QUrl &url, const QString &cookieType, const QString &cookie)
{
    qDebug() << __PRETTY_FUNCTION__;

    QNetworkRequest request;

    request.setUrl(url);
    request.setAttribute(QNetworkRequest::CacheSaveControlAttribute, false);
    request.setRawHeader(cookieType.toAscii(), cookie.toUtf8());

    QNetworkReply *reply = m_networkManager->get(request, true);

    m_currentRequests.append(reply);
}

void SituareService::requestFinished(QNetworkReply *reply)
{
    qDebug() << __PRETTY_FUNCTION__;

    //Reply from situare
    if (m_currentRequests.contains(reply)) {

        qDebug() << "BytesAvailable: " << reply->bytesAvailable();

        if (reply->error()) {
            emit error(ErrorContext::NETWORK, reply->error());
        } else {
            QByteArray replyArray = reply->readAll();
            qDebug() << "Reply from: " << reply->url() << "reply " << replyArray;

            if(replyArray == ERROR_LAT.toAscii()) {
                qDebug() << "Error: " << ERROR_LAT;
                emit error(ErrorContext::SITUARE, SituareError::UPDATE_FAILED);
            } else if(replyArray == ERROR_LON.toAscii()) {
                qDebug() << "Error: " << ERROR_LON;
                emit error(ErrorContext::SITUARE, SituareError::UPDATE_FAILED);
            } else if(replyArray.contains(ERROR_SESSION.toAscii())) {
                qDebug() << "Error: " << ERROR_SESSION;
                emit error(ErrorContext::SITUARE, SituareError::SESSION_EXPIRED);
            } else if(replyArray.startsWith(OPENING_BRACE_MARK.toAscii())) {
                qDebug() << "JSON string";
                parseUserData(replyArray);
            } else if(replyArray.isEmpty()) {
                if(reply->url().toString().contains(UPDATE_LOCATION.toAscii())) {
                    emit updateWasSuccessful();
                } else {
                    // session credentials are invalid
                    emit error(ErrorContext::SITUARE, SituareError::SESSION_EXPIRED);
                }
            } else {
                // unknown reply
                emit error(ErrorContext::SITUARE, SituareError::ERROR_GENERAL);
            }
        }
        m_currentRequests.removeAll(reply);
        reply->deleteLater();
    }
}

void SituareService::credentialsReady(const FacebookCredentials &credentials)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_credentials = credentials;    
}

void SituareService::parseUserData(const QByteArray &jsonReply)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_visited = 0;
    m_nbrOfImages = 0;
    m_defaultImage = false;
    qDeleteAll(m_friendsList.begin(), m_friendsList.end());
    m_friendsList.clear();

    if(m_user) {
        delete m_user;
        m_user = 0;
    }

    QJson::Parser parser;
    bool ok;

    QVariantMap result = parser.parse (jsonReply, &ok).toMap();
    if (!ok) {
        emit error(ErrorContext::SITUARE, SituareError::INVALID_JSON);
        return;
    } else {

        if(result.contains("ErrorCode")) {
            QVariant errorVariant = result.value("ErrorCode");
            emit error(ErrorContext::SITUARE, errorVariant.toInt());
            return;
        } else if(result.contains("user")) {

            QVariant userVariant = result.value("user");
            QMap<QString, QVariant> userMap = userVariant.toMap();

            QPointF coordinates(userMap["longitude"].toReal(), userMap["latitude"].toReal());

            QUrl imageUrl = userMap[NORMAL_SIZE_PROFILE_IMAGE].toUrl();

            if(imageUrl.isEmpty()) {
                // user doesn't have profile image, so we need to get him a silhouette image
                m_defaultImage = true;
            }

            QString address = userMap["address"].toString();
            if(address.isEmpty()) {
                QStringList location;
                location.append(QString::number(coordinates.y()));
                location.append(QString::number(coordinates.x()));
                address = location.join(", ");
            }

            m_user = new User(address, coordinates, userMap["name"].toString(),
                          userMap["note"].toString(), imageUrl, userMap["timestamp"].toString(),
                          true, userMap["uid"].toString());

            foreach (QVariant friendsVariant, result["friends"].toList()) {
              QMap<QString, QVariant> friendMap = friendsVariant.toMap();
              QVariant distance = friendMap["distance"];
              QMap<QString, QVariant> distanceMap = distance.toMap();

              QPointF coordinates(friendMap["longitude"].toReal(), friendMap["latitude"].toReal());

              QUrl imageUrl = friendMap["profile_pic"].toUrl();

              if(imageUrl.isEmpty()) {
                  // friend doesn't have profile image, so we need to get him a silhouette image
                  m_defaultImage = true;
              }

              QString address = friendMap["address"].toString();
              if(address.isEmpty()) {
                  QStringList location;
                  location.append(QString::number(coordinates.y()));
                  location.append(QString::number(coordinates.x()));
                  address = location.join(", ");
              }

              User *user = new User(address, coordinates,
                                    friendMap["name"].toString(),
                                    friendMap["note"].toString(), imageUrl,
                                    friendMap["timestamp"].toString(),
                                    false, friendMap["uid"].toString(),
                                    distanceMap["units"].toString(),
                                    distanceMap["value"].toDouble());

              m_friendsList.append(user);
            }
            addProfileImages();
        } else {
            QVariant address = result.value("address");
            if(!address.toString().isEmpty()) {
                emit reverseGeoReady(address.toString());
            } else {
                QStringList coordinates;
                coordinates.append(result.value("lat").toString());
                coordinates.append(result.value("lon").toString());

                emit error(ErrorContext::SITUARE, SituareError::ADDRESS_RETRIEVAL_FAILED);
                emit reverseGeoReady(coordinates.join(", "));
            }
        }
    }
}

void SituareService::imageReceived(const QUrl &url, const QPixmap &image)
{
    qDebug() << __PRETTY_FUNCTION__;
    qDebug() << "Image URL: " << url << " size :" << image.size();

    // assign facebook silhouette image to all who doesn't have a profile image
    if(url == QUrl(SILHOUETTE_URL)) {
        if(m_user->profileImageUrl().isEmpty()) {
            m_user->setProfileImage(AvatarImage::create(image, AvatarImage::Large));
        }
        for(int i=0;i < m_friendsList.count();i++) {
            if(m_friendsList.at(i)->profileImageUrl().isEmpty()) {
                m_friendsList.at(i)->setProfileImage(AvatarImage::create(image,
                                                                         AvatarImage::Small));
            }
        }
    }

    if (m_user->profileImageUrl() == url) {
        m_user->setProfileImage(AvatarImage::create(image, AvatarImage::Large));
    }

    for(int i=0;i<m_friendsList.count();i++) {
        if(m_friendsList.at(i)->profileImageUrl() == url) {
            m_friendsList.at(i)->setProfileImage(AvatarImage::create(image, AvatarImage::Small));
            m_nbrOfImages++; // indicates how many friend profile images has been downloaded
        }
    }

    if(m_nbrOfImages == m_visited) {
        qDebug() << "m_nbrOfImages: " << m_nbrOfImages << " m_visited: " << m_visited;
        qDebug() << "emit userDataChanged";
        emit userDataChanged(m_user, m_friendsList);
    }
}

void SituareService::addProfileImages()
{
    qDebug() << __PRETTY_FUNCTION__;

    // reduce net traffic by sending only one download request for facebook silhouette image
    if(m_defaultImage) {
        emit fetchImage(QUrl(SILHOUETTE_URL));
    }

    if(!m_user->profileImageUrl().isEmpty() && m_user->profileImageUrl().isValid())
        emit fetchImage(m_user->profileImageUrl());

    for(int i=0;i<m_friendsList.count();i++) {
        if(!m_friendsList.at(i)->profileImageUrl().isEmpty() &&
           m_friendsList.at(i)->profileImageUrl().isValid()) {
            m_visited++; // indicates how many friends that have profile image
            emit fetchImage(m_friendsList.at(i)->profileImageUrl());
        }
    }
}

void SituareService::clearUserData()
{
    qDebug() << __PRETTY_FUNCTION__;

    qDeleteAll(m_friendsList.begin(), m_friendsList.end());
    m_friendsList.clear();

    if(m_user) {
        delete m_user;
        m_user = 0;
    }
    emit userDataChanged(m_user, m_friendsList);
}
