/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Sami Rämö - sami.ramo@ixonos.com
       Pekka Nissinen - pekka.nissinen@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#include <cmath>

#include <QDebug>
#include <QMouseEvent>
#include <QParallelAnimationGroup>

#include "mapcommon.h"
#include "mapscroller.h"

#include "mapview.h"

const qreal MS_PER_S = 1000;

// const values for tuning the kinetic scroll effect
const int KINETIC_MIN_DRAG_LENGTH_VIEW_PIXELS = 30;
const int KINETIC_MAX_TIME_FROM_LAST_MOUSE_EVENT_MS = 100;
const qreal KINETIC_MAX_VIEW_DISTANCE_FACTOR = 0.8;
const int KINETIC_SCROLL_TIME_MS = 750;
const qreal KINETIC_SPEED_TO_DISTANCE_FACTOR = 0.15 * sqrt(KINETIC_SCROLL_TIME_MS / MS_PER_S);

const qreal ZOOM_TIME_MS = 350; ///< Length of the zoom effect (ms)

MapView::MapView(QWidget *parent)
    : QGraphicsView(parent),
      m_doubleTapZoomRunning(false)
{
    qDebug() << __PRETTY_FUNCTION__;

    setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    m_zoomAnimation = new QPropertyAnimation(this, "viewScale", this);
    connect(m_zoomAnimation, SIGNAL(finished()),
        this, SIGNAL(viewZoomFinished()));
    setOptimizationFlag(QGraphicsView::DontAdjustForAntialiasing);

    m_scroller = &MapScroller::getInstance();

    m_scrollAndZoomAnimation = new QParallelAnimationGroup();
    m_scrollAndZoomAnimation->addAnimation(m_scroller);
    m_scrollAndZoomAnimation->addAnimation(m_zoomAnimation);
    connect(m_scrollAndZoomAnimation, SIGNAL(finished()),
            this, SLOT(doubleTapZoomFinished()));
}

void MapView::centerToSceneCoordinates(QPoint sceneCoordinate)
{
    qDebug() << __PRETTY_FUNCTION__ << "sceneCoordinate" << sceneCoordinate;

    centerOn(sceneCoordinate);
}

void MapView::doubleTapZoomFinished()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_doubleTapZoomRunning = false;
    emit zoomIn();
}

void MapView::mouseDoubleClickEvent(QMouseEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_zoomLevel + 1 <= MAX_MAP_ZOOM_LEVEL) {
        QPoint pressPosition = mapToScene(event->pos()).toPoint();
        QPoint viewCenterPosition = mapToScene(width() / 2 - 1, height() / 2 - 1).toPoint();
        QPoint zoomPosition = viewCenterPosition - ((viewCenterPosition - pressPosition) / 2);

        m_scrollAndZoomAnimation->stop();
        m_doubleTapZoomRunning = true;

        m_scroller->setEasingCurve(QEasingCurve::Linear);
        m_scroller->setDuration(ZOOM_TIME_MS);
        m_scroller->setStartValue(m_scenePosition);
        m_scroller->setEndValue(zoomPosition);

        m_zoomAnimation->setEasingCurve(QEasingCurve::InQuad);
        m_zoomAnimation->setDuration(ZOOM_TIME_MS);
        m_zoomAnimation->setStartValue(viewScale());
        m_zoomAnimation->setEndValue(pow(2, m_zoomLevel+1 - MAX_MAP_ZOOM_LEVEL));

        m_scrollAndZoomAnimation->start();
    }
}

void MapView::mouseMoveEvent(QMouseEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_doubleTapZoomRunning)
        return;

    m_scenePosition += m_mouseLastScenePosition - mapToScene(event->pos()).toPoint();

    if (m_index >= VALUES)
        m_index = 0;

    m_dragMovement[m_index] = m_mouseLastViewPosition - event->pos();
    m_dragTime[m_index] = m_time.elapsed();
    m_time.start();
    m_index++;

    emit viewScrolled(m_scenePosition);

    m_mouseLastScenePosition = mapToScene(event->pos()).toPoint();
    m_mouseLastViewPosition = event->pos();
}

void MapView::mousePressEvent(QMouseEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_doubleTapZoomRunning)
        return;

    m_time.start();

    m_scroller->stop();

    QGraphicsView::mousePressEvent(event);

    m_mouseLastScenePosition = mapToScene(event->pos()).toPoint();
    m_mouseLastViewPosition = event->pos();
    m_scenePosition = mapToScene(width() / 2 - 1, height() / 2 - 1).toPoint();

    for (int i = 0; i < VALUES; i++) {
        m_dragMovement[i] = QPoint();
        m_dragTime[i] = 0;
    }
    m_index = 0;
}

void MapView::mouseReleaseEvent(QMouseEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_doubleTapZoomRunning)
        return;

    int elapsed = m_time.elapsed();

    QGraphicsView::mouseReleaseEvent(event);

    // start kinetic scroll only if there isn't too much time elapsed from the last mouse move event
    if (elapsed <= KINETIC_MAX_TIME_FROM_LAST_MOUSE_EVENT_MS) {
        QPointF dragViewSpeed;
        int dragLength = 0;
        int values = 0;
        for (int i = 0; i < VALUES; i++) {
            if (m_dragTime[i] > 0) {
                dragViewSpeed += m_dragMovement[i] / (m_dragTime[i] / MS_PER_S);
                dragLength += m_dragMovement[i].manhattanLength();
                values++;
            }
        }

        if (dragLength >= KINETIC_MIN_DRAG_LENGTH_VIEW_PIXELS) {
            dragViewSpeed /= values;
            QPointF effectViewDistance = dragViewSpeed * KINETIC_SPEED_TO_DISTANCE_FACTOR;

            // limit the scroll distance in screen pixels
            qreal biggerDistance = qMax(abs(effectViewDistance.x()), abs(effectViewDistance.y()));
            if (biggerDistance > m_kineticMaxViewDistance)
                effectViewDistance /= biggerDistance / m_kineticMaxViewDistance;

            QPointF effectSceneDistance = effectViewDistance
                                          * (1 << (MAX_MAP_ZOOM_LEVEL - m_zoomLevel));

            m_scroller->setEasingCurve(QEasingCurve::OutCirc);
            m_scroller->setDuration(KINETIC_SCROLL_TIME_MS);
            m_scroller->setStartValue(m_scenePosition);
            m_scroller->setEndValue(m_scenePosition + effectSceneDistance.toPoint());
            m_scroller->start();
        }
    }
}

void MapView::resizeEvent(QResizeEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__ << "Resize:" << event->size();

    m_kineticMaxViewDistance = qMax(width(), height()) * KINETIC_MAX_VIEW_DISTANCE_FACTOR;

    emit viewResized(event->size());
}

void MapView::setViewScale(qreal viewScale)
{
    qDebug() << __PRETTY_FUNCTION__;

    QTransform transform;
    transform.scale(viewScale, viewScale);
    setTransform(transform);
}

void MapView::setZoomLevel(int zoomLevel)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_zoomLevel = zoomLevel;

    if (m_zoomAnimation) {
        m_zoomAnimation->stop();
        m_zoomAnimation->setEasingCurve(QEasingCurve::InQuad);
        m_zoomAnimation->setDuration(ZOOM_TIME_MS);
        m_zoomAnimation->setStartValue(viewScale());
        m_zoomAnimation->setEndValue(pow(2, zoomLevel - MAX_MAP_ZOOM_LEVEL));

        m_zoomAnimation->start();
    }
}

qreal MapView::viewScale()
{
    qDebug() << __PRETTY_FUNCTION__;

    return transform().m11();
}

MapView::~MapView()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_scrollAndZoomAnimation->removeAnimation(m_scroller);
    delete m_scrollAndZoomAnimation;
}
