/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Jussi Laitinen - jussi.laitinen@ixonos.com
       Sami Rämö - sami.ramo@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#ifndef MAPFETCHER_H
#define MAPFETCHER_H

#include <QtCore>
#include <QNetworkAccessManager>

#include "maptilerequest.h"
#include "network/networkaccessmanager.h"

class QNetworkReply;
class QUrl;

/**
* @brief MapFetcher handles requests to get map tiles.
*
* @author Jussi Laitinen jussi.laitinen@ixonos.com
* @author Sami Rämö sami.ramo@ixonos.com
*/
class MapFetcher : public QObject
{
    Q_OBJECT

public:
    /**
    * @brief Constructor for MapFetcher.
    *
    * @param manager Network access manager
    * @param parent parent object
    */
    MapFetcher(NetworkAccessManager *manager, QObject *parent = 0);

/*******************************************************************************
 * CLASS SPECIFIC MEMBER FUNCTIONS AND SLOTS
 ******************************************************************************/
public:
    /**
    * @brief Set size of the download queue
    *
    * @param size New size
    */
    void setDownloadQueueSize(int size);

public slots:
    /**
    * @brief Enqueue fetching of map image
    *
    * Image fetching is triggered after events have been processed
    * so the UI stays responsive.
    *
    * @param zoomLevel Zoom level
    * @param x Tile x index
    * @param y Tile y index
    */
    void enqueueFetchMapImage(int zoomLevel, int x, int y);

private:
    /**
    * @brief Build URL for donwloading single map tile from OpenStreetMap tile server
    *
    * @param zoomLevel Zoom level
    * @param tileNumbers Tile x & y numbers
    * @return URL for the required tile
    */
    QUrl buildURL(int zoomLevel, QPoint tileNumbers);

    /**
    * @brief Limit pending requests list size to m_pendingRequestsSize
    *
    */
    void limitPendingRequestsListSize();

    /**
    * @brief Loads image from cache
    *
    * Tries to load requested, or upper zoom level image from the cache. Emits imageReveived signal
    * if any image was found. If image found was from requested zoom level then expidation date is
    * checked.
    *
    * @param url URL of the requested image
    * @return True if requested zoom level image was found and was not expired. Otherwise false.
    */
    bool loadImageFromCache(const QUrl &url);

    /**
    * @brief Find first item based on criteria if the request is already checked from the cache
    *
    * If cacheChecked is true, then returns index of the first request which is already
    * checked from the cache. If cacheChecked is false then returns first item which
    * isn't checked from the cache. Returns -1 if the item is not found.
    *
    * @param cacheChecked Search criteria
    * @return Index of the first matching request, or -1 if not found.
    */
    int newestRequestIndex(bool cacheChecked);

    /**
    * @brief Parse given URL to zoom, x and y values. Parsed values are
    * placed in variables given as parameters.
    *
    * @param url url to parse
    * @param [out] zoom zoom variable
    * @param [out] x x variable
    * @param [out] y y variable
    */
    void parseURL(const QUrl &url, int *zoom, int *x, int *y);

    /**
    * @brief Translate indexes to matching upper level map tile indexes
    * @param[in,out] zoomLevel Zoom level
    * @param[in,out] x x index
    * @param[in,out] y y index
    * @return true if translation succeeded, otherwise false
    */
    bool translateIndexesToUpperLevel(int &zoomLevel, int &x, int &y);

private slots:

    /**
    * @brief This slot is called when network manager has finished
    * the download. Loads image and emits imageReceived signal with
    * url and image. If there was a error in reply emits error-signal.
    *
    * @param reply
    */
    void downloadFinished(QNetworkReply *reply);

    /**
    * @brief Check next request if it is found from cache
    *
    * Next queued request, which is not already checked against cache, is taken
    * from the queue and checked against cache. If not found from cache, then
    * cache checked flag is set. New download is started if there aren't too
    * many simultaneus downloads already running.
    */
    void checkNextRequestFromCache();

    /**
    * @brief This slot is called when next download is started. Takes url
    * from queue, sends request and puts request to download queue.
    */
    void startNextDownload();

/*******************************************************************************
 * SIGNALS
 ******************************************************************************/
signals:    
    /**
    * @brief Signal which is emitted when a map tile
    * is received from the server and loaded to pixmap.
    *
    * @param zoomLevel Zoom level
    * @param x Tile x index
    * @param y Tile y index
    * @param image image pixmap
    */
    void mapImageReceived(int zoomLevel, int x, int y, const QPixmap &image);

    /**
    * @brief Signals error
    *
    * @param context error context
    * @param error error code
    */
    void error(const int context, const int error);

/*******************************************************************************
 * DATA MEMBERS
 ******************************************************************************/
private:
    QList<QNetworkReply*> m_currentDownloads; ///< List of current downloads
    int m_pendingRequestsSize; ///< Max number of pending requests
    bool m_fetchMapImagesTimerRunning; ///< is the singleshot timer already running
    NetworkAccessManager *m_manager; ///< Network access manager
    QList<MapTileRequest> m_pendingRequests; ///< List of map image fetching requests
};

#endif
