/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Sami Rämö - sami.ramo@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QDebug>
#include <QList>
#include <QPainter>
#include <QRect>
#include <QGraphicsSceneMouseEvent>
#include <QIcon>

#include "friendlocationitem.h"
#include "friendgroupitem.h"
#include "mapcommon.h"
#include "common.h"

FriendGroupItem::FriendGroupItem(FriendLocationItem *item)
    : m_clickEvent(false)
{
    qDebug() << __PRETTY_FUNCTION__;

    setPixmap(QPixmap(":/res/images/friend_group.png"));

    setOffset(-pixmap().width() / 2, -pixmap().height() / 2);
    setZValue(FRIEND_LOCATION_ICON_Z_LEVEL);

    joinFriend(item);
    setPos(item->pos());
}

void FriendGroupItem::dropFriend(FriendLocationItem *item)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_friends.removeOne(item))
        item->setPartOfGroup(false);
}

bool FriendGroupItem::dropFriends(int zoomLevel)
{
    qDebug() << __PRETTY_FUNCTION__;

    foreach (FriendLocationItem *friendItem, m_friends) {
        if (!friendItem->sceneTransformedBoundingRect(zoomLevel)
            .intersects(sceneTransformedBoundingRect(zoomLevel)))
            dropFriend(friendItem);
    }

    // group can be deleted if it includes max one friend
    if (m_friends.size() <= 1) {
        // drop remaining items
        foreach (FriendLocationItem *friendItem, m_friends)
            dropFriend(friendItem);

        return true;
    }
    return false;
}

void FriendGroupItem::joinFriend(FriendLocationItem *item)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_friends.append(item);
    item->setPartOfGroup(true);
}

void FriendGroupItem::mergeWithGroup(FriendGroupItem *group)
{
    qDebug() << __PRETTY_FUNCTION__;

    foreach (FriendLocationItem *item, m_friends) {
        m_friends.removeOne(item);
        group->joinFriend(item);
    }
}

void FriendGroupItem::paint(QPainter *painter, const QStyleOptionGraphicsItem *option,
                            QWidget *widget)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_clickEvent) {
        QIcon icon;
        icon.addPixmap(pixmap());
        painter->drawPixmap(offset(), icon.pixmap(pixmap().size(), QIcon::Selected,
                                                                 QIcon::On));
    } else {
        QGraphicsPixmapItem::paint(painter, option, widget);
    }

    painter->setPen(Qt::white);
    painter->setFont(NOKIA_FONT_SMALL);
    painter->drawText(QRect(GROUP_ITEM_FRIENDS_COUNT_X, GROUP_ITEM_FRIENDS_COUNT_Y,
                            GROUP_ITEM_FRIENDS_COUNT_WIDTH, GROUP_ITEM_FRIENDS_COUNT_HEIGHT),
                      Qt::AlignCenter, QString::number(m_friends.count()));
}

void FriendGroupItem::mouseMoveEvent(QGraphicsSceneMouseEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_clickEvent) {
        if ((event->pos() - m_mousePressPosition).manhattanLength() > PRESS_MANHATTAN_LENGTH) {
            m_clickEvent = false;
            update();
        }
    }
}

void FriendGroupItem::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_mousePressPosition = event->pos();
    m_clickEvent = true;
    update();
}

void FriendGroupItem::mouseReleaseEvent(QGraphicsSceneMouseEvent *event)
{
    Q_UNUSED(event);

    qDebug() << __PRETTY_FUNCTION__;

    if (m_clickEvent) {
        m_clickEvent = false;
        QList<QString> userIDs;

        foreach (FriendLocationItem *item, m_friends)
            userIDs.append(item->userId());

        emit locationItemClicked(userIDs);
    }

    update();
}
