/*
 * Copyright (C) 2010 Collabora Ltd.
 *   @author Marco Barisione <marco.barisione@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "config.h"

#include "call.h"
#include "call-private.h"
#include <gst/gst.h>
#include <libringtoned/ringtoned.h>
#include <libringtoned/utils.h>
#include <pulse/pulseaudio.h>
#include <telepathy-glib/util.h>
#include "ringtoned-enums.h"


#define RINGTONED_CALL_GET_PRIVATE(obj) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((obj), RINGTONED_TYPE_CALL, \
                                  RingtonedCallPrivate))

#define ID_RINGTONE 101

enum
{
    PROP_0,
    PROP_CHANNEL,
    PROP_STATUS,
    PROP_MUTE,
    PROP_IDENTIFIER,
    PROP_VCARD_FIELD,
    PROP_ACCOUNT,
    PROP_CONTACTS,
    PROP_RINGTONE_PATH,
};

struct _RingtonedCallPrivate
{
    TpChannel *channel;
    RingtonedCallStatus status;
    gboolean mute;

    gchar *identifier;
    gchar *vcard_field;
    McAccount *account;
    GList *contacts;

    gchar *ringtone_path;
    GstElement *pipeline;
    guint playing_timeout_id;
};

G_DEFINE_TYPE (RingtonedCall, ringtoned_call, G_TYPE_OBJECT)

RingtonedCall *
ringtoned_call_new (TpChannel   *channel,
                    const gchar *vcard_field,
                    const gchar *identifier,
                    McAccount   *account)
{
    return g_object_new (RINGTONED_TYPE_CALL,
            "channel", channel,
            "vcard-field", vcard_field,
            "identifier", identifier,
            "account", account,
            NULL);
}

static gboolean
bus_watch_cb (GstBus     *bus,
              GstMessage *msg,
              gpointer    user_data)
{
    RingtonedCall *self = user_data;

    switch (GST_MESSAGE_TYPE (msg))
    {
        case GST_MESSAGE_EOS:
            DEBUG ("Repeating");
            gst_element_seek_simple (self->priv->pipeline, GST_FORMAT_TIME,
                    GST_SEEK_FLAG_FLUSH | GST_SEEK_FLAG_KEY_UNIT, 0);
            break;

        case GST_MESSAGE_ERROR:
        {
            GError *err = NULL;
            gchar *dbg_info = NULL;

            gst_message_parse_error (msg, &err, &dbg_info);

            g_printerr ("Error from element %s: %s\n",
                    GST_OBJECT_NAME (msg->src), err->message);
            g_printerr ("Debugging info:\n%s\n",
                    dbg_info ? dbg_info : "none");

            gst_object_unref (self->priv->pipeline);
            self->priv->pipeline = NULL;

            g_error_free (err);
            g_free (dbg_info);

            break;
        }

        default:
            break;
    }

    return TRUE;
}

static gboolean
play_real (gpointer user_data)
{
    RingtonedCall *self = user_data;

    g_assert (self->priv->pipeline);
    gst_element_set_state (self->priv->pipeline, GST_STATE_PLAYING);

    self->priv->playing_timeout_id = 0;

    return FALSE;
}

void
ringtoned_call_play_ringtone (RingtonedCall *self)
{
    pa_proplist *proplist;
    gchar *playable;
    GstElement *src;
    GstElement *sink;
    GstBus *bus;
    GError *error = NULL;

    if (self->priv->status != RINGTONED_CALL_STATUS_RINGING)
    {
        DEBUG ("ringtoned_call_play_ringtone() called, but not in "
                "ringing status");
        return;
    }

    if (self->priv->pipeline)
    {
        DEBUG ("Already playing");
        return;
    }

    if (IS_EMPTY (self->priv->ringtone_path))
    {
        DEBUG ("No ringtone set, not playing");
        return;
    }

    if (ringtoned_profile_get_current_volume () == 0)
    {
        DEBUG ("Volume muted, not playing anything");
        return;
    }

    DEBUG ("Playing %s", self->priv->ringtone_path);

    self->priv->pipeline = gst_parse_launch (
            "filesrc name=src ! wavparse ! pulsesink name=sink", &error);
    if (!self->priv->pipeline)
    {
        g_critical ("Cannot create the ringtone pipeline: %s",
                error ? error->message : "unknown error");
        g_clear_error (&error);
        return;
    }

    /* Magical bits to make the ringtone play at the right volume */
    proplist = pa_proplist_new();
    pa_proplist_sets (proplist, "event.id", "phone-incoming-call");
    pa_proplist_sets (proplist, "module-stream-restore.id",
            "x-maemo-hildon-notify");

    sink = gst_bin_get_by_name (GST_BIN (self->priv->pipeline), "sink");
    g_assert (sink);
    g_object_set (sink, "proplist", proplist, NULL);

    src = gst_bin_get_by_name (GST_BIN (self->priv->pipeline), "src");
    g_assert (src);
    playable = ringtoned_decoder_get_playable_path (self->priv->ringtone_path);
    g_object_set (src, "location", playable, NULL);

    bus = gst_pipeline_get_bus (GST_PIPELINE (self->priv->pipeline));
    gst_bus_add_watch (bus, bus_watch_cb, self);

    DEBUG ("Actually playing %s", playable);

    /* Ringtoned is faster than the call-ui and we want to avoid playing the
     * ringtone too long before the UI is displayed */
    self->priv->playing_timeout_id = g_timeout_add (500, play_real, self);

    gst_object_unref (bus);
    g_free (playable);
    gst_object_unref (sink);
    gst_object_unref (src);
}

static void
stop_ringtone (RingtonedCall *self)
{
    if (self->priv->pipeline)
    {
        DEBUG ("Stop playing %s", self->priv->ringtone_path);

        if (self->priv->playing_timeout_id)
        {
            g_source_remove (self->priv->playing_timeout_id);
            self->priv->playing_timeout_id = 0;
        }

        gst_element_set_state (self->priv->pipeline, GST_STATE_NULL);
        gst_object_unref (self->priv->pipeline);
        self->priv->pipeline = NULL;
    }
}

static gboolean
unref_self_cb (gpointer user_data)
{
    RingtonedCall *self = user_data;

    g_object_unref (self);

    return FALSE;
}

static void
set_status (RingtonedCall       *self,
            RingtonedCallStatus  new_status)
{
    /* We cannot go back to ringing */
    g_return_if_fail (new_status != RINGTONED_CALL_STATUS_RINGING);

    if (self->priv->status == new_status)
        return;

    DEBUG ("Changing status from %s to %s (call: %p)",
            ringtoned_call_status_get_nick (self->priv->status),
            ringtoned_call_status_get_nick (new_status),
            self);

    if (self->priv->status == RINGTONED_CALL_STATUS_RINGING)
    {
        stop_ringtone (self);

        /* We increased the ref count when we started ringing and now
         * we drop it. This way if no plugin is using this object it will
         * be finalized and removed from the list of calls in the dispatcher.
         * We do the unref in the idle so the caller of this function doesn't
         * suddenly find that the object it was using was destroyed. */
        g_idle_add (unref_self_cb, self);
    }

    self->priv->status = new_status;
    g_object_notify (G_OBJECT (self), "status");
}

static gint
find_handle (GArray   *handles,
             TpHandle  handle)
{
    gint i;

    for (i = 0; i < (gint) handles->len; i++)
    {
        if (g_array_index (handles, TpHandle, i) == handle)
            return i;
    }

    return -1;
}

static void
group_members_changed_cb (TpChannel     *channel,
                          gchar         *message,
                          GArray        *added,
                          GArray        *removed,
                          GArray        *local_pending,
                          GArray        *remote_pending,
                          guint          actor,
                          guint          reason,
                          RingtonedCall *self)
{
    TpHandle self_handle;

    g_return_if_fail (self->priv->channel == channel);

    self_handle = tp_channel_group_get_self_handle (self->priv->channel);

    if (find_handle (added, self_handle) >= 0)
    {
        DEBUG ("Call accepted (call: %p, channel: %p)",
                self, self->priv->channel);
        set_status (self, RINGTONED_CALL_STATUS_ACTIVE);
    }
    else if (find_handle (removed, self_handle) >= 0)
    {
        DEBUG ("Call rejected (call: %p, channel: %p)",
                self, self->priv->channel);
        set_status (self, RINGTONED_CALL_STATUS_CLOSED);
    }
}

static void
channel_invalidated_cb (TpProxy       *proxy,
                        guint          domain,
                        gint           code,
                        gchar         *message,
                        RingtonedCall *self)
{
    DEBUG ("Channel invalidated (call: %p, channel: %p)",
            self, self->priv->channel);

    /* Usually the self handle is removed from the channel and then the
     * channel is invalidated, but we also have to handle the connection
     * manager misbehaving or crashing */
    set_status (self, RINGTONED_CALL_STATUS_CLOSED);
}

static void
set_mute (RingtonedCall *self,
          gboolean       new_mute)
{
    if (self->priv->mute == new_mute)
        return;

    if (!new_mute)
    {
        g_warning ("You cannot unmute an already muted call");
        return;
    }

    if (self->priv->status != RINGTONED_CALL_STATUS_RINGING)
    {
        g_warning ("Cannot mute a non-ringing call");
        return;
    }

    DEBUG ("Muting call %p", self);

    stop_ringtone (self);

    self->priv->mute = new_mute;
    g_object_notify (G_OBJECT (self), "mute");
}

static void
ringtoned_call_get_property (GObject    *object,
                             guint       param_id,
                             GValue     *value,
                             GParamSpec *pspec)
{
    RingtonedCall *self = RINGTONED_CALL (object);

    switch (param_id)
    {
        case PROP_CHANNEL:
            g_value_set_object (value, self->priv->channel);
            break;
        case PROP_STATUS:
            g_value_set_enum (value, self->priv->status);
            break;
        case PROP_MUTE:
            g_value_set_boolean (value, self->priv->mute);
            break;
        case PROP_IDENTIFIER:
            g_value_set_string (value, self->priv->identifier);
            break;
        case PROP_VCARD_FIELD:
            g_value_set_string (value, self->priv->vcard_field);
            break;
        case PROP_ACCOUNT:
            g_value_set_object (value, self->priv->account);
            break;
        case PROP_CONTACTS:
            g_value_set_boxed (value, self->priv->contacts);
            break;
        case PROP_RINGTONE_PATH:
            g_value_set_string (value, self->priv->ringtone_path);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
            break;
    }
}

static void
set_channel (RingtonedCall *self,
             TpChannel     *channel)
{
    g_return_if_fail (self->priv->channel == NULL);

    self->priv->channel = g_object_ref (channel);

    g_signal_connect (channel, "group-members-changed",
            G_CALLBACK (group_members_changed_cb), self);

    g_signal_connect (channel, "invalidated",
            G_CALLBACK (channel_invalidated_cb), self);
    if (tp_proxy_get_invalidated (channel))
        /* This should not happen, but... */
        channel_invalidated_cb (TP_PROXY (channel), 0, 0, NULL, self);
}

static void
ringtoned_call_set_property (GObject      *object,
                             guint         param_id,
                             const GValue *value,
                             GParamSpec   *pspec)
{
    RingtonedCall *self = RINGTONED_CALL (object);

    switch (param_id)
    {
        case PROP_CHANNEL:
            set_channel (self, g_value_get_object (value));
            break;
        case PROP_IDENTIFIER:
            g_return_if_fail (self->priv->identifier == NULL);
            self->priv->identifier = g_value_dup_string (value);
            break;
        case PROP_MUTE:
            set_mute (self, g_value_get_boolean (value));
            break;
        case PROP_VCARD_FIELD:
            g_return_if_fail (self->priv->vcard_field == NULL);
            self->priv->vcard_field = g_value_dup_string (value);
            break;
        case PROP_ACCOUNT:
            g_return_if_fail (self->priv->account == NULL);
            self->priv->account = g_value_dup_object (value);
            break;
        case PROP_RINGTONE_PATH:
            g_free (self->priv->ringtone_path);
            self->priv->ringtone_path = g_value_dup_string (value);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
            break;
    }
}

static void
ringtoned_call_init (RingtonedCall *self)
{
    self->priv = RINGTONED_CALL_GET_PRIVATE (self);

    self->priv->status = RINGTONED_CALL_STATUS_RINGING;
    /* Keep a ref as long as we are ringing */
    g_object_ref (self);
}

static void
ringtoned_call_constructed (GObject *object)
{
    RingtonedCall *self = RINGTONED_CALL (object);
    OssoABookAggregator *aggregator;
    GList *contacts;

    g_assert (self->priv->channel);

    aggregator = OSSO_ABOOK_AGGREGATOR (osso_abook_aggregator_get_default (NULL));

    /* We could use a generic function for searching, but I *think* that
     * this is the only way to get the fastest possible matching */
    if (!tp_strdiff (self->priv->vcard_field,  EVC_TEL))
        contacts = osso_abook_aggregator_find_contacts_for_phone_number (
                aggregator, self->priv->identifier, TRUE);
    else if (!tp_strdiff (self->priv->vcard_field, EVC_X_SIP))
        contacts = osso_abook_aggregator_find_contacts_for_sip_address (
                aggregator, self->priv->identifier);
    else
        contacts = osso_abook_aggregator_find_contacts_for_im_contact (
                aggregator, self->priv->identifier, self->priv->account);

    g_list_foreach (contacts, (GFunc) g_object_ref, NULL);
    self->priv->contacts = contacts;
}

static void
ringtoned_call_finalize (GObject *object)
{
    RingtonedCall *self = RINGTONED_CALL (object);

    stop_ringtone (self);

    if (self->priv->channel) {
        g_signal_handlers_disconnect_by_func (self->priv->channel,
                G_CALLBACK (group_members_changed_cb), self);
        g_signal_handlers_disconnect_by_func (self->priv->channel,
                G_CALLBACK (channel_invalidated_cb), self);
        g_object_unref (self->priv->channel);
    }

    if (self->priv->account)
        g_object_unref (self->priv->account);

    g_list_foreach (self->priv->contacts, (GFunc) g_object_unref, NULL);
    g_list_free (self->priv->contacts);

    g_free (self->priv->identifier);
    g_free (self->priv->ringtone_path);

    G_OBJECT_CLASS (ringtoned_call_parent_class)->finalize (object);
}

static void
ringtoned_call_class_init (RingtonedCallClass *class)
{
    GObjectClass *object_class = G_OBJECT_CLASS (class);

    object_class->constructed = ringtoned_call_constructed;
    object_class->finalize = ringtoned_call_finalize;
    object_class->get_property = ringtoned_call_get_property;
    object_class->set_property = ringtoned_call_set_property;

    g_object_class_install_property (
            object_class,
            PROP_CHANNEL,
            g_param_spec_object (
                "channel",
                "Channel",
                "The TpChannel for this call",
                TP_TYPE_CHANNEL,
                G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY |
                G_PARAM_STATIC_STRINGS));

    g_object_class_install_property (
            object_class,
            PROP_STATUS,
            g_param_spec_enum (
                "status",
                "Status",
                "The status of the call",
                RINGTONED_TYPE_CALL_STATUS,
                RINGTONED_CALL_STATUS_RINGING,
                G_PARAM_READABLE | G_PARAM_STATIC_STRINGS));

    g_object_class_install_property (
            object_class,
            PROP_MUTE,
            g_param_spec_boolean (
                "mute",
                "Mute",
                "Whether the call was muted",
                FALSE,
                G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));

    g_object_class_install_property (
            object_class,
            PROP_IDENTIFIER,
            g_param_spec_string (
                "identifier",
                "Identifier",
                "The identifier of the caller",
                "",
                G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY |
                G_PARAM_STATIC_STRINGS));

    g_object_class_install_property (
            object_class,
            PROP_VCARD_FIELD,
            g_param_spec_string (
                "vcard-field",
                "VCard field",
                "The VCard field for the identifier",
                "",
                G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY |
                G_PARAM_STATIC_STRINGS));

    g_object_class_install_property (
            object_class,
            PROP_ACCOUNT,
            g_param_spec_object (
                "account",
                "Account",
                "The McAccount for the call",
                MC_TYPE_ACCOUNT,
                G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY |
                G_PARAM_STATIC_STRINGS));

    g_object_class_install_property (
            object_class,
            PROP_CONTACTS,
            g_param_spec_boxed (
                "contacts",
                "Contacts",
                "A GList of OssoABookContacts matching the caller identifier",
                RINGTONED_TYPE_CONTACTS_LIST,
                G_PARAM_READABLE | G_PARAM_STATIC_STRINGS));

    g_object_class_install_property (
            object_class,
            PROP_RINGTONE_PATH,
            g_param_spec_string (
                "ringtone-path",
                "Ringtone path",
                "The file path of the ringtone",
                "",
                G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));

    g_type_class_add_private (object_class, sizeof (RingtonedCallPrivate));
}

GType
ringtoned_contacts_list_get_type (void)
{
    static GType our_type = 0;

    if (G_UNLIKELY (our_type == 0))
        our_type = g_boxed_type_register_static ("RingtonedContactsList",
                (GBoxedCopyFunc) ringtoned_contacts_list_copy,
                (GBoxedFreeFunc) ringtoned_contacts_list_free);

    return our_type;
}

void
ringtoned_contacts_list_free (GList *contacts)
{
    g_list_foreach (contacts, (GFunc) g_object_unref, NULL);
    g_list_free (contacts);
}

GList *
ringtoned_contacts_list_copy (GList *contacts)
{
    GList *copy;

    copy = g_list_copy (contacts);
    g_list_foreach (copy, (GFunc) g_object_ref, NULL);

    return copy;
}
