//    QtLockscreen
//    Copyright (C) 2011 Paolo Iommarini
//    sakya_tg@yahoo.it
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
#include "lsbattery.h"
#ifdef Q_WS_MAEMO_5
    #include <mce/dbus-names.h>
    #include <mce/mode-names.h>
#endif
#include "math.h"
#include <QProcess>
#include <QFile>

LsBattery::LsBattery(QString name) :
        LsWidget(name.isEmpty() ? "Battery": name)
{
    m_PercSource = LsBattery::SourceDefault;
    m_IsCharging = false;
    m_Percentage = 100;
    applySettings();

#ifndef Q_WS_MAEMO_5
    m_Percentage = 100;
    setImage();
#endif
}

void LsBattery::paint(QPainter *painter, const QStyleOptionGraphicsItem* option,
                      QWidget*)
{
    painter->setClipRect(option->exposedRect);
    if (!m_Image.isNull())
        painter->drawPixmap(0, 0, m_Image);

    if (!m_IsCharging){
        QRectF rect = boundingRect();
        painter->setFont(m_PercFont);
        painter->setPen(m_PercColor);
        painter->drawText(rect, Qt::AlignCenter | Qt::TextSingleLine, QString("%1%").arg(QString::number(m_Percentage, 'f', 1)));
    }
}

void LsBattery::stopUpdate()
{
#ifdef Q_WS_MAEMO_5
    QDBusConnection::systemBus().disconnect(QString(), "/org/freedesktop/Hal/devices/bme", "org.freedesktop.Hal.Device",
                                            "PropertyModified", this, SLOT(bmePropertyModifiedSlot(QDBusMessage)));
#endif
}

void LsBattery::startUpdate()
{
#ifdef Q_WS_MAEMO_5
    QDBusConnection::systemBus().connect(QString(), "/org/freedesktop/Hal/devices/bme", "org.freedesktop.Hal.Device",
                                         "PropertyModified", this, SLOT(bmePropertyModifiedSlot(QDBusMessage)));
    bmePropertyModifiedSlot(QDBusMessage());
#endif
}

void LsBattery::applySettings()
{
    setSizeAndPosition(QSize(64, 40),QPoint(0, 0));

    QString type = getSetting("PercentageType", "default").toString().toLower();
    if (type == "default" || type.isEmpty())
        m_PercSource = LsBattery::SourceDefault;
    else if (type == "dr_frost_dk")
        m_PercSource = LsBattery::SourceDrFrost;
    else if (type == "bq27200")
        m_PercSource = LsBattery::SourceBq27200;

    QString images = getSetting("Images",
                                "/opt/qtlockscreen/default/battery_0.png;/opt/qtlockscreen/default/battery_1.png;/opt/qtlockscreen/default/battery_2.png;/opt/qtlockscreen/default/battery_3.png;/opt/qtlockscreen/default/battery_4.png;/opt/qtlockscreen/default/battery_5.png;/opt/qtlockscreen/default/battery_6.png").toString();

    m_Images = images.split(';');

    m_ChargingImage = getSetting("ChargingImage", "/opt/qtlockscreen/default/battery_charging.png").toString();
    m_PercColor = getColorSetting("PercentageColor", QColor(255,255,255));
    m_PercFont = getFontSetting("PercentageFont", QFont("", 15, QFont::Normal, false));
}

#ifdef Q_WS_MAEMO_5
void LsBattery::bmePropertyModifiedSlot(QDBusMessage)
{
    qDebug() << "Battery state changed";
    QProcess* proc = new QProcess(this);
    proc->start("hal-device bme", QIODevice::ReadOnly);
    proc->waitForFinished();
    if (proc->exitCode() == 0){
        QByteArray ba = proc->readAll();
        QString output(ba);
        QStringList lines = output.split("\n", QString::SkipEmptyParts);

        float vcurrent = 0;
        float current = 0;
        float design = 0;
        float vdesign = 0;
        bool charging = false;

        foreach (const QString line, lines){
            QString trimmed = line.trimmed();

            QStringList e = trimmed.split("=", QString::SkipEmptyParts);
            if (e.count() > 1){
                QString name = e.at(0).trimmed();
                QString value = e.at(1).trimmed();
                int pos = value.indexOf(" ");
                if (pos != -1)
                    value = value.left(pos);

                if (name == "battery.reporting.current"){
                   current = value.toFloat();
                }else if (name == "battery.reporting.design"){
                   design = value.toFloat();
                }else if (name == "battery.voltage.current" && vcurrent == 0){
                   vcurrent = value.toFloat();
                }else if (name == "battery.voltage.design"){
                   vdesign = value.toFloat();
                }else if (name == "battery.rechargeable.is_charging"){
                   charging = value == "true";
                }
            }
        }

        m_IsCharging = charging;

        switch (m_PercSource)
        {
        case LsBattery::SourceDefault:
            if (design > 0)
                m_Percentage = current / design * 100.0;
            else
                m_Percentage = 0;
            break;
        case LsBattery::SourceDrFrost:
            if (vcurrent >= 4050)
                m_Percentage = 85.0 + (vcurrent - 4050.0) / 12.0;
            else if (vcurrent >= 3900)
                m_Percentage = 70.0 + (vcurrent - 3900.0) / 10.0;
            else if (vcurrent >= 3800)
                m_Percentage = 50.0 + (vcurrent - 3800.0) / 5.0;
            else if (vcurrent >= 3660)
                m_Percentage = 15.0 + (vcurrent - 3660.0) / 4.0;
            else if (vcurrent >= 3600)
                m_Percentage = 5.0 + (vcurrent - 3600.0) / 6.0;
            else
                m_Percentage = (vcurrent - 3300.0) / 60.0;
            break;
        case LsBattery::SourceBq27200:
            m_Percentage = GetValue("/sys/class/power_supply/bq27200-0/capacity");
        }
    }
    delete proc;

    qDebug() << "Battery percentage" << m_Percentage;

    setImage();
}
#endif

void LsBattery::setImage()
{
    if (m_IsCharging){
        if (m_CurrImage != m_ChargingImage){
            m_CurrImage = m_ChargingImage;
            m_Image = QPixmap(m_CurrImage);
            update();
        }
    }else if (m_Images.count() > 0){
        int idx = ceil(m_Percentage / (100.0 / (float)m_Images.count()) - 1.0);
        if (idx < 0)
            idx = 0;

        if (m_CurrImage != m_Images.at(idx)){
            m_CurrImage = m_Images.at(idx);
            m_Image = QPixmap(m_CurrImage);
            update();
        }
    }
}

float LsBattery::GetValue(QString file)
{
    float value = 0.0;
    QFile in(file);
    if (in.open(QIODevice::ReadOnly)){
        QString temp(in.readAll());
        value = temp.trimmed().toFloat();
        in.close();
    }
    return value;
}
