#include <time.h>

#include "reversigame.h"
#include "engine.h"

ReversiGame::ReversiGame()
{
    engine_ = new Engine();
    init(0, 1);
}

ReversiGame::~ReversiGame()
{
    delete engine_;
}

void ReversiGame::init(int mode, int strength, bool isNetworkGame)
{
    unsigned i, j;
    mode_ = mode;
    for (i = 0; i < FIELDSIZE; ++i)
        for (j = 0; j < FIELDSIZE; ++j)
            cells_[i][j] = Empty;
    //central cells
    cells_[3][3] = cells_[4][4] = White;
    cells_[3][4] = cells_[4][3] = Black;
    currentPlayer_ = Black;
    opponentPlayer_ = White;
    score_[0] = score_[1] = 2;

    engine_->init(strength, time(0));
    //engine_->init(strength);

    if (mode_ == 2)
        computerColor_ = Black;
    else if (mode_ == 1)
        computerColor_ = White;
    
    emit newGame(isNetworkGame);
    startNextTurn();
}//init

void ReversiGame::switchPlayer()
{
    opponentPlayer_ = currentPlayer_;
    currentPlayer_ = currentPlayer_ == White ? Black : White;
}

void ReversiGame::makeStep(unsigned x, unsigned y)
{
    changedCells_.clear();
    if (x < FIELDSIZE && y < FIELDSIZE && cells_[x][y] == Empty)
    {
        changedCells_.append(QPoint(x, y));
        changedCells_.append(checkVertical(x, y, -1, -1));
        changedCells_.append(checkVertical(x, y, FIELDSIZE, 1));
        changedCells_.append(checkHorizontal(x, -1, -1, y));
        changedCells_.append(checkHorizontal(x, FIELDSIZE, 1, y));
        changedCells_.append(checkDiagonal(x, -1, -1, y, -1, -1));
        changedCells_.append(checkDiagonal(x, -1, -1, y, FIELDSIZE, 1));
        changedCells_.append(checkDiagonal(x, FIELDSIZE, 1, y, -1, -1));
        changedCells_.append(checkDiagonal(x, FIELDSIZE, 1, y, FIELDSIZE, 1));
    }//if (x < FIELDSIZE && y < FIELDSIZE && cells_[x][y] == Empty)

    if (changedCells_.size() > 1)
    {
        foreach(QPoint p, changedCells_)
            cells_[p.x()][p.y()] = currentPlayer_;
        unsigned ind = int(currentPlayer_);
        score_[ind] += changedCells_.size();
        score_[1 - ind] -= changedCells_.size() - 1;

        switchPlayer();
        emit updateInfo();
        emit stepComplete();
    }//if (changedCells_.size() > 1)
    else
    {
        changedCells_.clear();
        emit stepImpossible();
    }
}

QList<QPoint> ReversiGame::checkVertical(unsigned x, unsigned startY, unsigned endY, unsigned stepY)
{
    QList<QPoint> changedCells;
    unsigned y;
    for(y = startY + stepY; y != endY && cells_[x][y] == opponentPlayer_; y += stepY)
    {
        changedCells.append(QPoint(x, y));
    }
    if (y == endY || cells_[x][y] != currentPlayer_)  //it beyond the board
        changedCells.clear();
    return changedCells;
}//checkHorizontal

QList<QPoint> ReversiGame::checkHorizontal(unsigned startX, unsigned endX, unsigned stepX, unsigned y)
{
    QList<QPoint> changedCells;
    unsigned x;
    for(x = startX + stepX; x != endX && cells_[x][y] == opponentPlayer_; x += stepX)
    {
        changedCells.append(QPoint(x, y));
    }
    if (x == endX || cells_[x][y] != currentPlayer_)  //it beyond the board
        changedCells.clear();
    return changedCells;
}//checkHorizontal

QList<QPoint> ReversiGame::checkDiagonal(unsigned startX, unsigned endX, unsigned stepX, unsigned startY, unsigned endY, unsigned stepY)
{
    QList<QPoint> changedCells;
    unsigned x, y;
    for(x = startX + stepX, y = startY + stepY; x != endX && y != endY && cells_[x][y] == opponentPlayer_; x += stepX, y += stepY)
    {
        changedCells.append(QPoint(x, y));
    }
    if (x == endX || y == endY || cells_[x][y] != currentPlayer_)  //it beyond the board
        changedCells.clear();
    return changedCells;
}

bool ReversiGame::isVerticalAvailable(unsigned x, unsigned startY, unsigned endY, unsigned stepY) const
{
    unsigned y;
    unsigned numChips = 0;
    for(y = startY + stepY; y != endY && cells_[x][y] == opponentPlayer_; y += stepY)
        ++numChips;
    return (y != endY && cells_[x][y] == currentPlayer_ && numChips);
}//isVerticalAvailable

bool ReversiGame::isHorizontalAvailable(unsigned startX, unsigned endX, unsigned stepX, unsigned y) const
{
    unsigned numChips = 0;
    unsigned x;
    for(x = startX + stepX; x != endX && cells_[x][y] == opponentPlayer_; x += stepX)
        ++numChips;
    return (x != endX && cells_[x][y] == currentPlayer_ && numChips); 
}//isHorizontalAvailable

bool ReversiGame::isDiagonalAvailable(unsigned startX, unsigned endX, unsigned stepX, 
        unsigned startY, unsigned endY, unsigned stepY) const
{
    unsigned numChips = 0;
    unsigned x, y;
    for(x = startX + stepX, y = startY + stepY; x != endX && y != endY && cells_[x][y] == opponentPlayer_; x += stepX, y += stepY)
        ++numChips;
    return (x != endX && y != endY && cells_[x][y] == currentPlayer_ && numChips);
}//isDiagonalAvailable

bool ReversiGame::isStepAvailable() const
{
    for(unsigned x = 0; x < FIELDSIZE; ++x)
        for(unsigned y = 0; y < FIELDSIZE; ++y)
            if (cells_[x][y] == Empty)
            {
                if (isHorizontalAvailable(x, -1, -1, y))
                    return true;
                if (isHorizontalAvailable(x, FIELDSIZE, 1, y))
                    return true;
                if (isVerticalAvailable(x, y, -1, -1))
                    return true;
                if (isVerticalAvailable(x, y, FIELDSIZE, 1))
                    return true;
                if (isDiagonalAvailable(x, -1, -1, y, -1, -1))
                    return true;
                if (isDiagonalAvailable(x, -1, -1, y, FIELDSIZE, 1))
                    return true;
                if (isDiagonalAvailable(x, FIELDSIZE, 1, y, -1, -1))
                    return true;
                if (isDiagonalAvailable(x, FIELDSIZE, 1, y, FIELDSIZE, 1))
                    return true;
            }
    return false;
}//isStepAvailable

void
ReversiGame::startNextTurn()
{
    if (isGameOver())
    {
        emit gameOver();
        return;
    }

    if ((mode_ == 1 || mode_ == 2) && isComputersTurn())
    {
        if (isStepAvailable())
            computerMove();
        else
        {
            emit stepUnavailable();
            switchPlayer();
            emit updateInfo();
        } /*         else */
    }
    else
    {
        if (!isStepAvailable())
        {
            emit stepUnavailable();
            switchPlayer();
            emit updateInfo();
            if (mode_ == 1 || mode_ == 2)
                computerMove();
        } /*         if (!isStepAvailable()) */
    }
} /* ReversiGame::startNextTurn() */

void
ReversiGame::computerMove()
{
    ReversiPos pos = engine_->computeMove(*this);
    if (pos.first != FIELDSIZE + 1 && pos.second != FIELDSIZE + 1)
    {
        makeStep(pos.first, pos.second);
    } /*     if (pos.first != FIELDSIZE + 1 && pos.second != FIELDSIZE + 1) */
} /* ReversiGame::computerMove() */

bool 
ReversiGame::isGameOver()
{
    if (score(White) + score(Black) == CHIPS_COUNT)
        return true;

    bool c1 = isStepAvailable();
    ChipColor cur = currentPlayer_;
    currentPlayer_ = opponentPlayer_;
    opponentPlayer_ = cur;
    bool c2 = isStepAvailable();
    cur = currentPlayer_;
    currentPlayer_ = opponentPlayer_;
    opponentPlayer_ = cur;
    return !c1 && !c2;
} /* ReversiGame::isGameOver() const */
