/****************************************************************************
**
** Copyright (C) 2010 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: Nokia Corporation (qt-info@nokia.com)
**
** This file is part of the Qt Mobility Components.
**
** $QT_BEGIN_LICENSE:LGPL$
** No Commercial Usage
** This file contains pre-release code and may not be distributed.
** You may use this file in accordance with the terms and conditions
** contained in the Technology Preview License Agreement accompanying
** this package.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights.  These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** If you have questions regarding the use of this file, please contact
** Nokia at qt-info@nokia.com.
**
**
**
**
**
**
**
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include "qdeclarativegraphicsgeomap_p.h"

#include "qdeclarativecoordinate_p.h"
#include "qdeclarativegeoserviceprovider_p.h"

#include <qgeoserviceprovider.h>
#include <qgeomappingmanager.h>
#include <qgeomapdata.h>
#include <qgeomapobject.h>

#include <QGraphicsSceneMouseEvent>

QTM_BEGIN_NAMESPACE

/*!
    \qmlclass Map

    \brief The Map element displays a map.
    \inherits QDeclarativeItem

    \ingroup qml-location-maps

    The Map element can be used be used to display a map of the world.  The 
    bulk of the functionality is provided by a mapping plugin described 
    by the Plugin element associated with the Map.

    Various map objects can be added to the map.  These map objects are 
    specified in terms of coordinates and metres.


    The Map element is part of the \bold{QtMobility.location 1.1} module.
*/
QDeclarativeGraphicsGeoMap::QDeclarativeGraphicsGeoMap(QDeclarativeItem *parent)
    : QDeclarativeItem(parent),
    plugin_(0),
    serviceProvider_(0),
    mappingManager_(0),
    mapData_(0),
    mapType_(NoMap),
    connectivityMode_(NoConnectivity)
{
    setFlag(QGraphicsItem::ItemHasNoContents, false);

    center_ = new QDeclarativeCoordinate(this);

    connect(center_,
            SIGNAL(latitudeChanged(double)),
            this,
            SLOT(centerLatitudeChanged(double)));
    connect(center_,
            SIGNAL(longitudeChanged(double)),
            this,
            SLOT(centerLongitudeChanged(double)));
    connect(center_,
            SIGNAL(altitudeChanged(double)),
            this,
            SLOT(centerAltitudeChanged(double)));

    center_->setCoordinate(QGeoCoordinate(-27.0, 153.0));
    zoomLevel_ = 8;
    size_ = QSizeF(100.0, 100.0);
    setAcceptsHoverEvents(true);
    setAcceptHoverEvents(true);
    setAcceptedMouseButtons(Qt::LeftButton);
    setFlag(QGraphicsItem::ItemHasNoContents, false);
    setFlag(QGraphicsItem::ItemAcceptsInputMethod);
}

QDeclarativeGraphicsGeoMap::~QDeclarativeGraphicsGeoMap()
{
    qDeleteAll(objects_);

    if (mapData_)
        delete mapData_;

    if (serviceProvider_)
        delete serviceProvider_;
}

void QDeclarativeGraphicsGeoMap::paint(QPainter *painter,
                                       const QStyleOptionGraphicsItem *option,
                                       QWidget * /*widget*/)
{
    if (mapData_) {
        mapData_->paint(painter, option);
    }
}

void QDeclarativeGraphicsGeoMap::geometryChanged(const QRectF &newGeometry,
                                                 const QRectF & /*oldGeometry*/)
{
    setSize(newGeometry.size());
}

/*!
    \qmlproperty Plugin Map::plugin

    This property holds the plugin which provides the mapping functionality.

    This is write-once property.  Once the map has a plugin associated with 
    it any attempted modifications of the plugin will be ignored.
*/

void QDeclarativeGraphicsGeoMap::setPlugin(QDeclarativeGeoServiceProvider *plugin)
{
    if (plugin_)
        return;

    plugin_ = plugin;

    emit pluginChanged(plugin_);

    serviceProvider_ = new QGeoServiceProvider(plugin_->name(),
                                               plugin_->parameterMap());

    if (serviceProvider_->error() != QGeoServiceProvider::NoError) {
        qWarning() << serviceProvider_->errorString();
        delete serviceProvider_;
        serviceProvider_ = 0;
        return;
    }

    mappingManager_ = serviceProvider_->mappingManager();
    if (!mappingManager_ || serviceProvider_->error() != QGeoServiceProvider::NoError) {
        qWarning() << serviceProvider_->errorString();
        delete serviceProvider_;
        serviceProvider_ = 0;
        delete mappingManager_;
        mappingManager_ = 0;
        return;
    }

    mapData_ = mappingManager_->createMapData();
    mapData_->init();
    //mapData_->setParentItem(this);

    // setters
    mapData_->setWindowSize(size_);
    mapData_->setZoomLevel(zoomLevel_);
    mapData_->setCenter(center_->coordinate());
    mapData_->setMapType(QGraphicsGeoMap::MapType(mapType_));
    mapData_->setConnectivityMode(QGraphicsGeoMap::ConnectivityMode(connectivityMode_));

    for (int i = 0; i < objects_.size(); ++i)
        mapData_->addMapObject(objects_.at(i)->mapObject());

    // setup signals

    connect(mapData_,
            SIGNAL(updateMapDisplay(QRectF)),
            this,
            SLOT(updateMapDisplay(QRectF)));

    connect(mapData_,
            SIGNAL(centerChanged(QGeoCoordinate)),
            this,
            SLOT(internalCenterChanged(QGeoCoordinate)));

    connect(mapData_,
            SIGNAL(mapTypeChanged(QGraphicsGeoMap::MapType)),
            this,
            SLOT(internalMapTypeChanged(QGraphicsGeoMap::MapType)));

    connect(mapData_,
            SIGNAL(connectivityModeChanged(QGraphicsGeoMap::ConnectivityMode)),
            this,
            SLOT(internalConnectivityModeChanged(QGraphicsGeoMap::ConnectivityMode)));

    connect(mapData_,
            SIGNAL(windowSizeChanged(QSizeF)),
            this,
            SIGNAL(sizeChanged(QSizeF)));

    connect(mapData_,
            SIGNAL(zoomLevelChanged(qreal)),
            this,
            SIGNAL(zoomLevelChanged(qreal)));
}

void QDeclarativeGraphicsGeoMap::updateMapDisplay(const QRectF &target)
{
    update(target);
}

QDeclarativeGeoServiceProvider* QDeclarativeGraphicsGeoMap::plugin() const
{
    return plugin_;
}

/*!
    \qmlproperty qreal Map::minimumZoomLevel

    This property holds the minimum valid zoom level for the map.
*/
qreal QDeclarativeGraphicsGeoMap::minimumZoomLevel() const
{
    if (mappingManager_)
        return mappingManager_->minimumZoomLevel();
    else
        return -1.0;
}

/*!
    \qmlproperty qreal Map::maximumZoomLevel

    This property holds the maximum valid zoom level for the map.
*/
qreal QDeclarativeGraphicsGeoMap::maximumZoomLevel() const
{
    if (mappingManager_)
        return mappingManager_->maximumZoomLevel();
    else
        return -1.0;
}

// TODO make these more QML like
//QList<MapType> QDeclarativeGraphicsGeoMap::supportedMapTypes() const;
//QList<ConnectivityMode> QDeclarativeGraphicsGeoMap::supportedConnectivityModes() const;

/*!
    \qmlproperty QSizeF Map::size

    This property holds the size of the map viewport.
*/
void QDeclarativeGraphicsGeoMap::setSize(const QSizeF &size)
{
    if (mapData_) {
        setWidth(size.width());
        setHeight(size.height());
        mapData_->setWindowSize(size);
    } else {
        if (size_ == size)
            return;

        size_ = size;

        emit sizeChanged(size_);
    }
        
}

QSizeF QDeclarativeGraphicsGeoMap::size() const
{
    if (mapData_)
        return mapData_->windowSize();
    else
        return size_;
}

/*!
    \qmlproperty qreal Map::zoomLevel

    This property holds the zoom level for the map.

    Larger values for the zoom level provide more detail.

    The default value is 8.0.
*/
void QDeclarativeGraphicsGeoMap::setZoomLevel(qreal zoomLevel)
{
    if (mapData_) {
        mapData_->setZoomLevel(zoomLevel);
    } else {
        if (zoomLevel_ == zoomLevel)
            return;

        zoomLevel_ = zoomLevel;

        emit zoomLevelChanged(zoomLevel_);
    }
}

qreal QDeclarativeGraphicsGeoMap::zoomLevel() const
{
    if (mapData_) {
        return mapData_->zoomLevel();
    } else {
        return zoomLevel_;
    }
}

/*!
    \qmlproperty Coordinate Map::center

    This property holds the coordinate which occupies the center of the 
    mapping viewport.

    The default value is an arbitrary valid coordinate.
*/
void QDeclarativeGraphicsGeoMap::setCenter(const QDeclarativeCoordinate *center)
{
    if (mapData_) {
        mapData_->setCenter(center->coordinate());
    } else {
        if (center_->coordinate() == center->coordinate())
            return;

        center_->setCoordinate(center->coordinate());

        emit declarativeCenterChanged(center_);
    }
}

QDeclarativeCoordinate* QDeclarativeGraphicsGeoMap::center()
{
    if (mapData_)
        center_->setCoordinate(mapData_->center());
    return center_;
}

void QDeclarativeGraphicsGeoMap::centerLatitudeChanged(double /*latitude*/)
{
    if (mapData_)
        mapData_->setCenter(center_->coordinate());
}

void QDeclarativeGraphicsGeoMap::centerLongitudeChanged(double /*longitude*/)
{
    if (mapData_)
        mapData_->setCenter(center_->coordinate());
}

void QDeclarativeGraphicsGeoMap::centerAltitudeChanged(double /*altitude*/)
{
    if (mapData_)
        mapData_->setCenter(center_->coordinate());
}

/*!
    \qmlproperty enumeration Map::mapType

    This property holds the type of map to display.

    The type can be one of:
    \list
    \o Map.StreetMap
    \o Map.SatelliteMapDay
    \o Map.SatelliteMapNight
    \o Map.TerrainMap
    \endlist

    The default value is determined by the plugin.
*/
void QDeclarativeGraphicsGeoMap::setMapType(QDeclarativeGraphicsGeoMap::MapType mapType)
{
    if (mapData_) {
        mapData_->setMapType(QGraphicsGeoMap::MapType(mapType));
    } else {
        if (mapType_ == mapType)
            return;

        mapType_ = mapType;

        emit mapTypeChanged(mapType_);
    }
}

QDeclarativeGraphicsGeoMap::MapType QDeclarativeGraphicsGeoMap::mapType() const
{
    if (mapData_) {
        return QDeclarativeGraphicsGeoMap::MapType(mapData_->mapType());
    } else {
        return mapType_;
    }
}

/*!
    \qmlproperty enumeration Map::connectivityMode

    This property holds the connectivity mode used to fetch the map data.

    The mode can be one of:
    \list
    \o Map.OfflineMode
    \o Map.OnlineMode
    \o Map.HybridMode
    \endlist

    The default value is determined by the plugin.
*/
void QDeclarativeGraphicsGeoMap::setConnectivityMode(QDeclarativeGraphicsGeoMap::ConnectivityMode connectivityMode)
{
    if (mapData_) {
        mapData_->setConnectivityMode(QGraphicsGeoMap::ConnectivityMode(connectivityMode));
    } else {
        if (connectivityMode_ == connectivityMode)
            return;

        connectivityMode_ = connectivityMode;

        emit connectivityModeChanged(connectivityMode_);
    }
}

QDeclarativeGraphicsGeoMap::ConnectivityMode QDeclarativeGraphicsGeoMap::connectivityMode() const
{
    if (mapData_)
        return QDeclarativeGraphicsGeoMap::ConnectivityMode(mapData_->connectivityMode());
    else
        return connectivityMode_;
}

/*!
    \qmlproperty list<QGeoMapObject> Map::objects
    \default

    This property holds the list of objects associated with this map.

    The various objects that can be added include:
    \list
    \o MapRectangle
    \o MapCircle
    \o MapText
    \o MapImage
    \o MapPolygon
    \o MapPolyline
    \o MapGroup
    \endlist
*/
QDeclarativeListProperty<QDeclarativeGeoMapObject> QDeclarativeGraphicsGeoMap::objects()
{
    return QDeclarativeListProperty<QDeclarativeGeoMapObject>(this,
            0,
            object_append,
            object_count,
            object_at,
            object_clear);
}

void QDeclarativeGraphicsGeoMap::object_append(QDeclarativeListProperty<QDeclarativeGeoMapObject> *prop, QDeclarativeGeoMapObject *mapObject)
{
    QDeclarativeGraphicsGeoMap *map = static_cast<QDeclarativeGraphicsGeoMap*>(prop->object);

    if (map->mapData_)
        map->mapData_->addMapObject(mapObject->mapObject());
    map->objects_.append(mapObject);
    map->objectMap_.insert(mapObject->mapObject(), mapObject);
}

int QDeclarativeGraphicsGeoMap::object_count(QDeclarativeListProperty<QDeclarativeGeoMapObject> *prop)
{
    return static_cast<QDeclarativeGraphicsGeoMap*>(prop->object)->objects_.size();
}

QDeclarativeGeoMapObject *QDeclarativeGraphicsGeoMap::object_at(QDeclarativeListProperty<QDeclarativeGeoMapObject> *prop, int index)
{
    return static_cast<QDeclarativeGraphicsGeoMap*>(prop->object)->objects_.at(index);
}

void QDeclarativeGraphicsGeoMap::object_clear(QDeclarativeListProperty<QDeclarativeGeoMapObject> *prop)
{
    QDeclarativeGraphicsGeoMap *map = static_cast<QDeclarativeGraphicsGeoMap*>(prop->object);

    if (map->mapData_)
        map->mapData_->clearMapObjects();
    map->objects_.clear();
    map->objectMap_.clear();
}

/*!
    \qmlmethod Map::toCoordinate(QPointF screenPosition)

    Returns the coordinate which corresponds to the screen position 
    \a screenPosition.

    Returns an invalid coordinate if \a screenPosition is not within
    the current viewport.
*/

QDeclarativeCoordinate* QDeclarativeGraphicsGeoMap::toCoordinate(QPointF screenPosition) const
{
    QGeoCoordinate coordinate;

    if (mapData_)
        coordinate = mapData_->screenPositionToCoordinate(screenPosition);

    return new QDeclarativeCoordinate(coordinate,
                                      const_cast<QDeclarativeGraphicsGeoMap *>(this));
}

/*!
    \qmlmethod Map::toScreenPosition(Coordinate coordinate)

    Returns the screen position which corresponds to the coordinate 
    \a coordinate.

    Returns an invalid QPointF if \a coordinate is not within the 
    current viewport.
*/
QPointF QDeclarativeGraphicsGeoMap::toScreenPosition(QDeclarativeCoordinate* coordinate) const
{
    QPointF point;

    if (mapData_)
        point = mapData_->coordinateToScreenPosition(coordinate->coordinate());

    return point;
}

void QDeclarativeGraphicsGeoMap::pan(int dx, int dy)
{
    if (mapData_) {
        mapData_->pan(dx, dy);
        update();
    }
}

QDeclarativeGeoMapMouseEvent* QDeclarativeGraphicsGeoMap::createMapMouseEvent(QGraphicsSceneMouseEvent *event)
{
    if (!event || !mapData_)
        return 0;

    QDeclarativeGeoMapMouseEvent *mouseEvent = new QDeclarativeGeoMapMouseEvent(this);

    mouseEvent->setButtons(event->buttons());
    QGeoCoordinate coordinate = mapData_->screenPositionToCoordinate(event->pos());
    mouseEvent->setCoordinate(new QDeclarativeCoordinate(coordinate, this));
    mouseEvent->setModifiers(event->modifiers());
    mouseEvent->setX(event->pos().x());
    mouseEvent->setY(event->pos().y());

    return mouseEvent;
}

void QDeclarativeGraphicsGeoMap::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    if (!mapData_)
        return;

    QList<QGeoMapObject*> objects = mapData_->mapObjectsAtScreenPosition(event->pos());

    QDeclarativeGeoMapMouseEvent *mouseEvent = createMapMouseEvent(event);

    for (int i = 0; i < objects.size(); ++i) {
        QDeclarativeGeoMapObject* mapObject = objectMap_.value(objects.at(i), 0);
        if (mapObject) {
            mapObject->pressEvent(mouseEvent);
            if (mouseEvent->accepted()) {
                event->setAccepted(true);
                return;
            }
        }
    }
}

void QDeclarativeGraphicsGeoMap::mouseReleaseEvent(QGraphicsSceneMouseEvent *event)
{
    if (!mapData_)
        return;

    QList<QGeoMapObject*> objects = mapData_->mapObjectsAtScreenPosition(event->pos());

    QDeclarativeGeoMapMouseEvent *mouseEvent = createMapMouseEvent(event);

    for (int i = 0; i < objects.size(); ++i) {
        QDeclarativeGeoMapObject* mapObject = objectMap_.value(objects.at(i), 0);
        if (mapObject) {
            mapObject->releaseEvent(mouseEvent);
            if (mouseEvent->accepted()) {
                event->setAccepted(true);
                return;
            }
        }
    }
}

void QDeclarativeGraphicsGeoMap::mouseDoubleClickEvent(QGraphicsSceneMouseEvent *event)
{
    if (!mapData_)
        return;

    QList<QGeoMapObject*> objects = mapData_->mapObjectsAtScreenPosition(event->pos());

    QDeclarativeGeoMapMouseEvent *mouseEvent = createMapMouseEvent(event);

    for (int i = 0; i < objects.size(); ++i) {
        QDeclarativeGeoMapObject* mapObject = objectMap_.value(objects.at(i), 0);
        if (mapObject) {
            mapObject->doubleClickEvent(mouseEvent);
            if (mouseEvent->accepted()) {
                event->setAccepted(true);
                return;
            }
        }
    }
}

void QDeclarativeGraphicsGeoMap::mouseMoveEvent(QGraphicsSceneMouseEvent *event)
{
    if (!mapData_)
        return;

    QList<QGeoMapObject*> objectsThen = mapData_->mapObjectsAtScreenPosition(event->lastPos());
    QList<QGeoMapObject*> objectsNow = mapData_->mapObjectsAtScreenPosition(event->pos());

    QSet<QGeoMapObject*> enter = objectsNow.toSet();
    enter -= objectsThen.toSet();

    for (int i = 0; i < objectsNow.size(); ++i) {
        if (!enter.contains(objectsNow.at(i)))
            continue;

        QDeclarativeGeoMapObject* mapObject = objectMap_.value(objectsNow.at(i), 0);
        if (mapObject)
            mapObject->enterEvent();
    }

    QSet<QGeoMapObject*> exit = objectsThen.toSet();
    exit -= objectsNow.toSet();

    for (int i = 0; i < objectsThen.size(); ++i) {
        if (!exit.contains(objectsThen.at(i)))
            continue;

        QDeclarativeGeoMapObject* mapObject = objectMap_.value(objectsThen.at(i), 0);
        if (mapObject)
            mapObject->exitEvent();
    }

    QSet<QGeoMapObject*> move = objectsNow.toSet();
    move += objectsThen.toSet();

    QList<QGeoMapObject*> objects = mapData_->mapObjectsInViewport();
    for (int i = 0; i < objects.size(); ++i) {
        if (!move.contains(objects.at(i)))
            continue;

        QDeclarativeGeoMapMouseEvent *mouseEvent = createMapMouseEvent(event);

        QDeclarativeGeoMapObject* mapObject = objectMap_.value(objects.at(i), 0);
        if (mapObject)
            mapObject->moveEvent(mouseEvent);
    }

}

void QDeclarativeGraphicsGeoMap::internalCenterChanged(const QGeoCoordinate &coordinate)
{
    emit declarativeCenterChanged(new QDeclarativeCoordinate(coordinate, this));
}

void QDeclarativeGraphicsGeoMap::internalMapTypeChanged(QGraphicsGeoMap::MapType mapType)
{
    emit mapTypeChanged(QDeclarativeGraphicsGeoMap::MapType(mapType));
}

void QDeclarativeGraphicsGeoMap::internalConnectivityModeChanged(QGraphicsGeoMap::ConnectivityMode connectivityMode)
{
    emit connectivityModeChanged(QDeclarativeGraphicsGeoMap::ConnectivityMode(connectivityMode));
}

#include "moc_qdeclarativegraphicsgeomap_p.cpp"

QTM_END_NAMESPACE

