#include "album.h"
#include <QDebug>
#include "spotifysession.h"
#include "link.h"

namespace Spotify
{
    Album::Album( Spotify::Album& album, bool ref ):
            QObject( 0 ),
            m_hasRef( ref )
    {
        m_album = album.getAlbum();
        if( m_album )
        {
            if( m_hasRef )
            {
                qDebug() << "An album was copied and a reference was added.";
                sp_album_add_ref( m_album );
            }

            m_isLoaded = album.isLoaded();
            m_isAvailable = album.isAvailable();

            if( m_isLoaded )
            {
                if( album.getArtist() )
                    m_artist = new Spotify::Artist( *album.getArtist() );
                m_cover = album.getCover();
                m_name = album.getName();
                m_year = album.getYear();
                m_type = album.type();
            }

            setParent( album.parent() );
           // if( thread() != album.thread() )
           //     moveToThread( album.thread() );
        }
    }

    Album::Album( sp_album* album, bool ref, QObject *parent ):
            QObject(parent),
            m_album( album ),
            m_isLoaded( false ),
            m_isAvailable( false ),
            m_artist( 0 ),
            m_cover( 0 ),
            m_hasRef( ref )
    {
        if( m_album )
        {
            if( m_hasRef )
            {
                qDebug() << "An album was created with ref";
                sp_album_add_ref( m_album );
            }
            load();
        }
    }

    bool Album::load()
    {
        if( m_album && sp_album_is_loaded( m_album ) )
        {
           // sp_image_size image_size = SP_IMAGE_SIZE_LARGE;
            m_isAvailable = sp_album_is_available( m_album );
            m_artist = new Spotify::Artist( sp_album_artist( m_album ) );
            m_cover = sp_album_cover( m_album );
            const char* charData = sp_album_name( m_album );
            m_name = QString::fromUtf8( charData, strlen( charData ) );
            m_year = sp_album_year( m_album );
            sp_albumtype at = sp_album_type( m_album );
            if( at == SP_ALBUMTYPE_ALBUM ) m_type = Spotify::Album::ALBUM;
            if( at == SP_ALBUMTYPE_SINGLE ) m_type = Spotify::Album::SINGLE;
            if( at == SP_ALBUMTYPE_COMPILATION ) m_type = Spotify::Album::COMPILATION;
            if( at == SP_ALBUMTYPE_UNKNOWN ) m_type = Spotify::Album::UNKNOWN;
            m_isLoaded = true;
            emit loaded();
        }
        return m_isLoaded;
    }

    void Album::setType(Spotify::Album::Type type)
    { //set album type
        m_type = type;
    }

    QString Album::getTypeName()
    {
        //return textual form of album type

        if(m_type==Spotify::Album::ALBUM)
            return tr("Album");
        else if(m_type==Spotify::Album::SINGLE)
            return tr("Single");
        else if(m_type==Spotify::Album::COMPILATION)
            return tr("Compilation");
        else
            return tr("Appears on");
    }

    QString Album::getCategory()
    {
        //Return textual form of album category
        if(m_type==Spotify::Album::ALBUM)
            return tr("Albums");
        else if(m_type==Spotify::Album::SINGLE)
            return tr("Singles");
        else if(m_type==Spotify::Album::COMPILATION)
            return tr("Compilations");
        else
            return tr("Appears on");
    }


    Spotify::Album* Spotify::Album::fromLink( Spotify::Link& link, QObject* parent )
    {
        if( link.isValid() && link.type() == Spotify::Link::ALBUM )
        {
            sp_album* album = sp_link_as_album( link.getLink() );
            if( album )
                return new Spotify::Album( album, true, parent );
        }
        return 0;
    }

    Album::~Album()
    {
        if( m_artist )
            m_artist->deleteLater();

        if( m_album && m_hasRef )
        {
            qDebug() << "A reffed album was dereffed and deleted.";
            sp_album_release( m_album );
        }
    }
} // namespace Spotify

 bool albumsByTypeYearDesc(Spotify::Album* alb1, Spotify::Album* alb2)
 {
     //sort albums first by then, then by year (descending)

     if(alb1 && alb2)
     {
         if(alb1->type()==alb2->type())
         {
             return(alb1->getYear()>alb2->getYear());
         }
         else
         {
             return(alb1->type()<alb2->type());
         }
     }

    return false;
 }
