//Settings.cpp: Implementation of the SettingsDialog class

#include "SettingsDialog.h"

SettingsDialog::SettingsDialog(QSettings* pSettings, QWidget* pParent)
    : QDialog(pParent)
{
    //set up dialog UI

    m_pAppSettings = pSettings;

    setFixedSize(APP_WIDTH,APP_HEIGHT);

    //connect signals/slots
    connect(&m_SaveSettingsBtn,SIGNAL(clicked()),SLOT(OnSaveSettingsBtn()));
    connect(&m_EnableScrobblingSelection,SIGNAL(stateChanged(int)),SLOT(OnScrobblingStateChanged(int)));
    connect(QApplication::desktop(), SIGNAL(resized(int)),SLOT(OnOrientationChanged()));

    setCapsLandscape(); //set caps for landscape mode

    m_maxSearchTracksInput.setInputMask("9000");
    m_maxSearchAlbumsInput.setInputMask("9000");
    m_maxSearchArtistsInput.setInputMask("9000");

    m_SearchOptionLayout.addWidget(&m_maxSearchTracksCaption,0,0);
    m_SearchOptionLayout.addWidget(&m_maxSearchTracksInput,0,1);
    m_SearchOptionLayout.addWidget(&m_maxSearchAlbumsCaption,1,0);
    m_SearchOptionLayout.addWidget(&m_maxSearchAlbumsInput,1,1);
    m_SearchOptionLayout.addWidget(&m_maxSearchArtistsCaption,2,0);
    m_SearchOptionLayout.addWidget(&m_maxSearchArtistsInput,2,1);
    m_SearchOptions.setLayout(&m_SearchOptionLayout);

    m_MinBufferLenInput.setInputMask("900");

    m_BufferingOptionLayout.addWidget(&m_MinBufferLenCaption,0,0);
    m_BufferingOptionLayout.addWidget(&m_MinBufferLenInput,0,1);
    m_BufferingOptionLayout.addWidget(&m_PreBufferNextTrackSelection,1,0,1,-1);
    m_BufferingOptions.setLayout(&m_BufferingOptionLayout);

    //set up bitrate selection value button / selector

    m_bitrateList << tr("Low bitrate (96kbps)") << tr("Standard bitrate (160kbps)")
            << tr("High bitrate (320kbps)");

#ifdef Q_WS_MAEMO_5

    QStringListModel *pBitrateListModel = new QStringListModel(this);

    pBitrateListModel->setStringList(m_bitrateList);

    m_pPreferredBitrateSelector = new QMaemo5ListPickSelector(this);
    m_pPreferredBitrateSelector->setModel(pBitrateListModel);
    m_PreferredBitrateBtn.setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered); // Maemo5 style guide wants the menu entries centered.
    m_PreferredBitrateBtn.setPickSelector(m_pPreferredBitrateSelector);
    m_pPreferredSyncBitrateSelector = new QMaemo5ListPickSelector(this);
    m_pPreferredSyncBitrateSelector->setModel(pBitrateListModel);
    m_PreferredSyncBitrateBtn.setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered); // Maemo5 style guide wants the menu entries centered.
    m_PreferredSyncBitrateBtn.setPickSelector(m_pPreferredSyncBitrateSelector);

#else

    m_PreferredBitrateCombo.addItems(m_bitrateList);
    m_PreferredSyncBitrateCombo.addItems(m_bitrateList);

#endif

    //set up UI language value button / selector

    m_UILangList << "English" << "Suomi" << "Dutch" << "German" << "French" << "Spanish"; //possible UI language values

    m_UILangCodeList << "en" << "fi" << "nl" << "de" << "fr" << "es"; //corresponding internal language codes

#ifdef Q_WS_MAEMO_5

    QStringListModel *pLangListModel = new QStringListModel(this);
    pLangListModel->setStringList(m_UILangList);
    m_pUILangSelector = new QMaemo5ListPickSelector(this);
    m_pUILangSelector->setModel(pLangListModel);
    m_UILangBtn.setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered); // Maemo5 style guide wants the menu entries centered.
    m_UILangBtn.setPickSelector(m_pUILangSelector);

#else

    m_UILangCombo.addItems(m_UILangList);

#endif

    //set up screen orientation value button / selector

    m_orientationList << tr("Landscape") << tr("Portrait") << tr("Automatic");

#ifdef Q_WS_MAEMO_5

    QStringListModel *pListModel = new QStringListModel(this);
    pListModel->setStringList(m_orientationList);
    m_pOrientationSelector = new QMaemo5ListPickSelector(this);
    m_pOrientationSelector->setModel(pListModel);
    m_OrientationBtn.setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered); // Maemo5 style guide wants the menu entries centered.
    m_OrientationBtn.setPickSelector(m_pOrientationSelector);

#else

    m_OrientationCombo.addItems(m_orientationList);

#endif

    m_LastFMUserIDInput.setInputMethodHints(Qt::ImhNoAutoUppercase);
    m_LastFMPasswordInput.setEchoMode(QLineEdit::Password); //set password mode for input field

    //advanced settings

    m_pathLocationList << tr("Internal EMMC") << tr("Memory (SD) card");

#ifdef Q_WS_MAEMO_5

    QStringListModel *pPathListModel = new QStringListModel(this);
    pPathListModel->setStringList(m_pathLocationList);
    m_pPathSelector = new QMaemo5ListPickSelector(this);
    m_pPathSelector->setModel(pPathListModel);
    m_PathSelBtn.setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered); // Maemo5 style guide wants the menu entries centered.
    m_PathSelBtn.setPickSelector(m_pPathSelector);
    m_AdvancedOptionLayout.addWidget(&m_PathSelBtn,0,0,1,-1);

#endif

    m_AdvancedOptions.setLayout(&m_AdvancedOptionLayout);

    m_ScrobblingOptionLayout.addWidget(&m_EnableScrobblingSelection,0,0,1,2);
    m_ScrobblingOptionLayout.addWidget(&m_LastFMUserIDCaption,1,0);
    m_ScrobblingOptionLayout.addWidget(&m_LastFMUserIDInput,1,1);
    m_ScrobblingOptionLayout.addWidget(&m_LastFMPasswordCaption,2,0);
    m_ScrobblingOptionLayout.addWidget(&m_LastFMPasswordInput,2,1);

    m_ScrobblingOptions.setLayout(&m_ScrobblingOptionLayout);

    m_GridLayout.addWidget(&m_SearchOptions,0,0,1,-1);
    m_GridLayout.addWidget(&m_BufferingOptions,1,0,1,-1);
    m_GridLayout.addWidget(&m_HideCoverArt,2,0,1,-1);
#ifdef Q_WS_MAEMO_5
    m_GridLayout.addWidget(&m_PreferredBitrateBtn,3,0,1,-1);
    m_GridLayout.addWidget(&m_PreferredSyncBitrateBtn,4,0,1,-1);
    m_GridLayout.addWidget(&m_AllowMobileSyncSelection,5,0,1,-1);
    m_GridLayout.addWidget(&m_UILangBtn,6,0,1,-1);
    m_GridLayout.addWidget(&m_OrientationBtn,7,0,1,-1);
#else
    m_GridLayout.addWidget(&m_PreferredBitrateCombo,3,0,1,-1);
    m_GridLayout.addWidget(&m_PreferredSyncBitrateCombo,4,0,1,-1);
    m_GridLayout.addWidget(&m_AllowMobileSyncSelection,5,0,1,-1);
    m_GridLayout.addWidget(&m_UILangCombo,6,0,1,-1);
    m_GridLayout.addWidget(&m_OrientationCombo,7,0,1,-1);
#endif
    m_GridLayout.addWidget(&m_PauseOnHSDisconnSelection,8,0,1,-1);
    m_GridLayout.addWidget(&m_ScrobblingOptions,9,0,1,-1);
    m_GridLayout.addWidget(&m_AdvancedOptions,10,0,1,-1);

    m_MainAreaLayout.setLayout(&m_GridLayout);
    m_MainArea.setWidget(&m_MainAreaLayout);

    m_MainLayout.addWidget(&m_MainArea,0,0,1,1);
    m_MainLayout.addWidget(&m_SaveSettingsBtn,1,1,1,1,Qt::AlignRight);

    setLayout(&m_MainLayout);

    readSettings(); //read and set application setting values
}

SettingsDialog::~SettingsDialog()
{
}

void SettingsDialog::setCapsLandscape()
{
    //set dialog captions (landscape mode)

    setWindowTitle(tr("QSpot - Settings"));
    m_maxSearchTracksCaption.setText(tr("Max. tracks returned"));
    m_maxSearchAlbumsCaption.setText(tr("Max. albums returned"));
    m_maxSearchArtistsCaption.setText(tr("Max. artists returned"));
    m_SearchOptions.setTitle(tr("Search settings"));
    m_BufferingOptions.setTitle(tr("Buffering settings"));
    m_MinBufferLenCaption.setText(tr("Min. buffering length (secs)"));
    m_PreBufferNextTrackSelection.setText(tr("Enable track pre-buffering"));
#ifdef Q_WS_MAEMO_5
    m_PreferredBitrateBtn.setText(tr("Preferred bitrate"));
    m_PreferredSyncBitrateBtn.setText(tr("Preferred offline sync. bitrate"));
    m_UILangBtn.setText(tr("UI Language"));
    m_OrientationBtn.setText(tr("Display orientation"));
    m_PathSelBtn.setText(tr("Cache / settings location"));
#endif
    m_AllowMobileSyncSelection.setText(tr("Allow sync. over 3G"));
    m_PauseOnHSDisconnSelection.setText(tr("Pause on headset unplugged"));
    m_ScrobblingOptions.setTitle(tr("Scrobbling options"));
    m_EnableScrobblingSelection.setText(tr("Enable scrobbling to Last.fm"));
    m_LastFMUserIDCaption.setText(tr("Last.fm username"));
    m_LastFMPasswordCaption.setText(tr("Last.fm password"));
    m_AdvancedOptions.setTitle(tr("Advanced settings"));
    m_SaveSettingsBtn.setText(tr("Save"));
    m_HideCoverArt.setText(tr("Hide coverart in Now Playing"));
}

void SettingsDialog::setCapsPortrait()
{
    //set dialog captions (portrait mode)

    setWindowTitle(tr("QSpot - Settings"));
    m_maxSearchTracksCaption.setText(tr("Max. tracks"));
    m_maxSearchAlbumsCaption.setText(tr("Max. albums"));
    m_maxSearchArtistsCaption.setText(tr("Max. artists"));
    m_SearchOptions.setTitle(tr("Search settings"));
    m_BufferingOptions.setTitle(tr("Buffering settings"));
    m_MinBufferLenCaption.setText(tr("Min.buff.length (secs)"));
    m_PreBufferNextTrackSelection.setText(tr("Enable track pre-buff."));
#ifdef Q_WS_MAEMO_5
    m_PreferredBitrateBtn.setText(tr("Preferred bitrate"));
    m_PreferredSyncBitrateBtn.setText(tr("Preferred offline bitrate"));
    m_UILangBtn.setText(tr("UI Language"));
    m_OrientationBtn.setText(tr("Display orientation"));
    m_PathSelBtn.setText(tr("Cache / settings location"));
#endif
    m_AllowMobileSyncSelection.setText(tr("Allow sync. over 3G"));
    m_PauseOnHSDisconnSelection.setText(tr("Pause on headset unplugged"));
    m_ScrobblingOptions.setTitle(tr("Scrobbling options"));
    m_EnableScrobblingSelection.setText(tr("Enable Last.fm scrobbling"));
    m_LastFMUserIDCaption.setText(tr("Last.fm username"));
    m_LastFMPasswordCaption.setText(tr("Last.fm password"));
    m_AdvancedOptions.setTitle(tr("Advanced settings"));
    m_SaveSettingsBtn.setText(tr("Save"));
    m_HideCoverArt.setText(tr("Hide coverart in Now Playing"));
}

void SettingsDialog::readSettings()
{
    //read and set current  application settings from config file (if config file not exists, use default settings)

    m_MainArea.ensureVisible(0,0); //make sure the scrollarea is reset (top area displayed)
    m_SaveSettingsBtn.setFocus(); //default focus set to Save button

    if(m_pAppSettings)
    {
        m_maxSearchTracks = m_pAppSettings->value("QSpot/maxSearchTracks",MAX_TRACKS_DEFAULT).toInt();
        m_maxSearchTracksInput.setText(QString::number(m_maxSearchTracks));

        m_maxSearchAlbums = m_pAppSettings->value("QSpot/maxSearchAlbums",MAX_ALBUMS_DEFAULT).toInt();
        m_maxSearchAlbumsInput.setText(QString::number(m_maxSearchAlbums));

        m_maxSearchArtists = m_pAppSettings->value("QSpot/maxSearchArtists",MAX_ARTISTS_DEFAULT).toInt();
        m_maxSearchArtistsInput.setText(QString::number(m_maxSearchArtists));

        m_minBufferLen = m_pAppSettings->value("QSpot/minBufferLen",MIN_BUFFER_LEN_DEFAULT).toInt();
        m_MinBufferLenInput.setText(QString::number(m_minBufferLen));

        m_preBufferingActive = m_pAppSettings->value("QSpot/preBufferingActive",PRE_BUFFERING_DEFAULT).toInt();

        if(m_preBufferingActive)
            m_PreBufferNextTrackSelection.setChecked(true);
        else
            m_PreBufferNextTrackSelection.setChecked(false);

        m_hideCoverart = m_pAppSettings->value("QSpot/HideCoverart",HIDE_COVERART_DEFAULT).toInt();
        if(m_hideCoverart)
        { //set hide coverart checkbox
            m_HideCoverArt.setChecked(true);
        }
        else
        {
            m_HideCoverArt.setChecked(false);
        }

        m_prefBitrate = m_pAppSettings->value("QSpot/PrefBitrate",IDX_BITRATE_160k).toInt();
#ifdef Q_WS_MAEMO_5
        m_pPreferredBitrateSelector->setCurrentIndex(m_prefBitrate);
#else
        m_PreferredBitrateCombo.setCurrentIndex(m_prefBitrate);
#endif
        m_prefSyncBitrate = m_pAppSettings->value("QSpot/PrefSyncBitrate",IDX_BITRATE_160k).toInt();
#ifdef Q_WS_MAEMO_5
        m_pPreferredSyncBitrateSelector->setCurrentIndex(m_prefSyncBitrate);
#else
        m_PreferredSyncBitrateCombo.setCurrentIndex(m_prefSynxBitrate);
#endif

        m_allowMobileSync = m_pAppSettings->value("QSpot/allowMobileSync",ALLOW_MOBILE_SYNC_DEFAULT).toInt();

        if(m_allowMobileSync)
            m_AllowMobileSyncSelection.setChecked(true);
        else
            m_AllowMobileSyncSelection.setChecked(false);

        m_currUILang = m_UILangCodeList.indexOf(m_pAppSettings->value("QSpot/language",LANG_DEFAULT).toString());
#ifdef Q_WS_MAEMO_5
        m_pUILangSelector->setCurrentIndex(m_currUILang);
#else
        m_UILangCombo.setCurrentIndex(m_currUILang);
#endif

        m_currOrientation = m_pAppSettings->value("QSpot/displayOrientation",ORIENTATION_DEFAULT).toInt();
#ifdef Q_WS_MAEMO_5
        m_pOrientationSelector->setCurrentIndex(m_currOrientation);
#else
        m_OrientationCombo.setCurrentIndex(m_currOrientation);
#endif

        m_pauseOnHsetUnplugged = m_pAppSettings->value("QSpot/pauseOnHsetUnplugged",PAUSE_ON_HEADSET_UNPLUG_DEFAULT).toInt();
        if(m_pauseOnHsetUnplugged)
            m_PauseOnHSDisconnSelection.setChecked(true);
        else
            m_PauseOnHSDisconnSelection.setChecked(false);

        m_enableScrobbling = m_pAppSettings->value("QSpot/enableScrobbling",ENABLE_SCROBBLING_DEFAULT).toInt();

        if(m_enableScrobbling)
        {
            m_EnableScrobblingSelection.setChecked(true);
            m_LastFMUserIDInput.setEnabled(true);
            m_LastFMPasswordInput.setEnabled(true);
            m_strLastFMUserID = m_pAppSettings->value("QSpot/lastfmUid","").toString();
            m_strLastFMPassword = m_pAppSettings->value("QSpot/lastfmPwd","").toString();
            m_LastFMUserIDInput.setText(m_strLastFMUserID);
            m_LastFMPasswordInput.setText(m_strLastFMPassword);
        }
        else
        {
            m_EnableScrobblingSelection.setChecked(false);
            m_LastFMUserIDInput.setEnabled(false);
            m_LastFMPasswordInput.setEnabled(false);
            m_LastFMUserIDInput.setText("");
            m_LastFMPasswordInput.setText("");
        }

        m_strCacheDirPath = m_pAppSettings->value("QSpot/cacheLocation", CACHE_PATH_EMMC + "/" + CACHE_DIR_NAME_DEFAULT).toString();
        if(m_strCacheDirPath==CACHE_PATH_MMC + "/" + CACHE_DIR_NAME_DEFAULT)
        { //cache location set to memory (SD) card
#ifdef Q_WS_MAEMO_5
            m_pPathSelector->setCurrentIndex(1);
            m_PathSelBtn.setDisabled(false);
#endif
        }
        else
        { //cache location set to internal EMMC
#ifdef Q_WS_MAEMO_5
            m_pPathSelector->setCurrentIndex(0);
            if(!mmcPresent())
            { //disable value button in case a memory (SD) card is not present
                m_PathSelBtn.setDisabled(true);
            }
#endif
        }
    }
}

void SettingsDialog::saveSettings()
{
    //save (modified) settings

    bool settingsChanged = false;
    qint32 selBitrate, selUILang, selOrientation;
    bool cacheLocationChanged = false;

    if(m_pAppSettings)
    {
        if(m_maxSearchTracksInput.text().toInt()!=m_maxSearchTracks)
        {
            m_pAppSettings->setValue("QSpot/maxSearchTracks",m_maxSearchTracksInput.text().toInt());
            settingsChanged = true;
        }

        if(m_maxSearchAlbumsInput.text().toInt()!=m_maxSearchAlbums)
        {
            m_pAppSettings->setValue("QSpot/maxSearchAlbums",m_maxSearchAlbumsInput.text().toInt());
            settingsChanged = true;
        }

        if(m_maxSearchArtistsInput.text().toInt()!=m_maxSearchArtists)
        {
            m_pAppSettings->setValue("QSpot/maxSearchArtists",m_maxSearchArtistsInput.text().toInt());
            settingsChanged = true;
        }

        if(m_MinBufferLenInput.text().toInt()!=m_minBufferLen)
        {
            m_pAppSettings->setValue("QSpot/minBufferLen",m_MinBufferLenInput.text().toInt());
            settingsChanged = true;
        }

        if(m_PreBufferNextTrackSelection.isChecked() && !m_preBufferingActive)
        {
            m_pAppSettings->setValue("QSpot/preBufferingActive",1);
            settingsChanged = true;
        }
        else if(!m_PreBufferNextTrackSelection.isChecked() && m_preBufferingActive)
        {
            m_pAppSettings->setValue("QSpot/preBufferingActive",0);
            settingsChanged = true;
        }

        if(m_HideCoverArt.isChecked() && !m_hideCoverart)
        {
            m_pAppSettings->setValue("QSpot/HideCoverart",1);
            settingsChanged = true;
        }
        else if(!m_HideCoverArt.isChecked() && m_hideCoverart)
        { //credentials should not be saved, remove any existing values
            m_pAppSettings->setValue("QSpot/HideCoverart",0);
            settingsChanged = true;
        }


        //streaming bitrate selection
#ifdef Q_WS_MAEMO_5
        selBitrate = m_pPreferredBitrateSelector->currentIndex();
#else
        selBitrate = m_PreferredBitrateCombo.currentIndex();
#endif

        if(selBitrate!=m_prefBitrate)
        {
            m_pAppSettings->setValue("QSpot/PrefBitrate",selBitrate);
            settingsChanged = true;
        }

        //offline sync. bitrate selection
#ifdef Q_WS_MAEMO_5
        selBitrate = m_pPreferredSyncBitrateSelector->currentIndex();
#else
        selBitrate = m_PreferredSyncBitrateCombo.currentIndex();
#endif

        if(selBitrate!=m_prefSyncBitrate)
        {
            m_pAppSettings->setValue("QSpot/PrefSyncBitrate",selBitrate);
            settingsChanged = true;
        }

        if(m_AllowMobileSyncSelection.isChecked() && !m_allowMobileSync)
        {
            m_pAppSettings->setValue("QSpot/allowMobileSync",1);
            settingsChanged = true;
        }
        else if(!m_AllowMobileSyncSelection.isChecked() && m_allowMobileSync)
        {
            m_pAppSettings->setValue("QSpot/allowMobileSync",0);
            settingsChanged = true;
        }

#ifdef Q_WS_MAEMO_5
        selUILang = m_pUILangSelector->currentIndex();
#else
        selUILang = m_UILangCombo.currentIndex();
#endif

        if(selUILang!=m_currUILang)
        {
            m_pAppSettings->setValue("QSpot/language",m_UILangCodeList.at(selUILang));
#ifdef Q_WS_MAEMO_5
            QMaemo5InformationBox::information(this,tr("Please restart QSpot for language change to take effect!"));
#else
            Banner::information(this,tr("Please restart QSpot for language change to take effect!"));
#endif

        }

#ifdef Q_WS_MAEMO_5
        selOrientation = m_pOrientationSelector->currentIndex();
#else
        selOrientation = m_OrientationCombo.currentIndex();
#endif

        if(selOrientation!=m_currOrientation)
        {
            m_pAppSettings->setValue("QSpot/displayOrientation",selOrientation);
            settingsChanged = true;
        }

        if(m_PauseOnHSDisconnSelection.isChecked() && !m_pauseOnHsetUnplugged)
        {
            m_pAppSettings->setValue("QSpot/pauseOnHsetUnplugged",1);
            settingsChanged = true;
        }
        else if(!m_PauseOnHSDisconnSelection.isChecked() && m_pauseOnHsetUnplugged)
        {
            m_pAppSettings->setValue("QSpot/pauseOnHsetUnplugged",0);
            settingsChanged = true;
        }

        if(m_EnableScrobblingSelection.isChecked())
        {
            m_pAppSettings->setValue("QSpot/enableScrobbling",1);
            m_pAppSettings->setValue("QSpot/lastfmUid",m_LastFMUserIDInput.text());
            m_pAppSettings->setValue("QSpot/lastfmPwd",m_LastFMPasswordInput.text());
            settingsChanged = true;
        }
        else
        {
            m_pAppSettings->setValue("QSpot/enableScrobbling",0);
            //empty associated userID / password values
            m_pAppSettings->setValue("QSpot/lastfmUid","");
            m_pAppSettings->setValue("QSpot/lastfmPwd","");
            settingsChanged = true;
        }

#ifdef Q_WS_MAEMO_5
        if(m_pPathSelector->currentIndex()==0 && m_strCacheDirPath!=CACHE_PATH_EMMC+"/"+CACHE_DIR_NAME_DEFAULT)
        {
            m_pAppSettings->setValue("QSpot/cacheLocation",CACHE_PATH_EMMC+"/"+CACHE_DIR_NAME_DEFAULT);
            cacheLocationChanged = true;
        }
        else if(m_pPathSelector->currentIndex()==1 && m_strCacheDirPath!=CACHE_PATH_MMC+"/"+CACHE_DIR_NAME_DEFAULT)
        {
            m_pAppSettings->setValue("QSpot/cacheLocation",CACHE_PATH_MMC+"/"+CACHE_DIR_NAME_DEFAULT);
            cacheLocationChanged = true;
        }
#endif

        if(cacheLocationChanged)
        {
#ifdef Q_WS_MAEMO_5
            QMaemo5InformationBox::information(this,tr("Please restart QSpot for cache location change to take effect!"));
#else
            Banner::information(this,tr("Please restart QSpot for cache location change to take effect!"));
#endif
        }

        if(settingsChanged)
            emit settingsSaved();
    }
}

void SettingsDialog::OnScrobblingStateChanged(int newState)
{
    //scrobbling state has been changed (enabled / disabled)

    if(newState==Qt::Checked)
    {
        //scrobbling enabled
        m_LastFMUserIDInput.setEnabled(true);
        m_LastFMPasswordInput.setEnabled(true);
    }
    else
    {
        //scrobbling enabled
        m_LastFMUserIDInput.setEnabled(false);
        m_LastFMPasswordInput.setEnabled(false);
        m_LastFMUserIDInput.setText("");
        m_LastFMPasswordInput.setText("");
    }
}

void SettingsDialog::OnSaveSettingsBtn()
{
    //save current settings (if changed)
    saveSettings();
    hide(); //close / hide dialog
}

void SettingsDialog::OnOrientationChanged()
{
    //device orientation changed (e.g, screen rotate)

    QRect screenGeometry = QApplication::desktop()->screenGeometry(); // get current screen geometry

    m_MainLayout.removeWidget(&m_SaveSettingsBtn);
    if (screenGeometry.width() > screenGeometry.height())
    {
        //switch to landscape mode
        m_MainLayout.addWidget(&m_SaveSettingsBtn,1,1,1,1,Qt::AlignRight);
        setCapsLandscape();
    }
    else
    {
        //switch to portrait mode
         m_MainLayout.addWidget(&m_SaveSettingsBtn,1,0,1,1);
         setCapsPortrait();
    }
}

bool SettingsDialog::mmcPresent()
{
    //Returns true in case memory (SD) card is present; otherwiser returns false.
    //SD card is detected by checking for presence of /media/mmc1 in /etc/mtab.

   QFile mtabFile("/etc/mtab");
   QString strLine;

   if(!mtabFile.open(QIODevice::ReadOnly))
       return false;

   QTextStream input(&mtabFile);
   while (!input.atEnd())
   {
     strLine = input.readLine();
     if(strLine.contains("/media/mmc1")) //SD (MMC) card present and mounted
     {
         mtabFile.close();
         return true;
     }
   }

   mtabFile.close();
   return false;
}

void SettingsDialog::OnPathSelectionBtn()
{
    //user has clicked the cache / setting path selection button
}
