/**
 *  spotifysession.h: Spotify session class. Handles the session functions.
 */
#ifndef SPOTIFYSESSION_H
#define SPOTIFYSESSION_H

#include <libspotify/api.h>
#include <credmanager.h>
#include <QMetaType>
#include <QDebug>
#include <QDir>
#include <QList>
#include <QImage>
#include <QHash>
#include <QBuffer>
#include <QApplication>
#include <QTimer>
#include <QThread>
#include <QWaitCondition>
#include <QMutex>
#include "error.h"
#include "track.h"
#include "playlist.h"
#include "artistcatalogue.h"
#include "albumbrowser.h"
#include "search.h"
#include "user.h"
#include "link.h"
#include "artist.h"
#include "album.h"
#include "PlaybackThread.h"
#include "connectionmanager.h"

namespace Spotify
{

    class Session: public QObject
    {
        Q_OBJECT

    public:

            Session( QObject* parent = 0 );
            virtual ~Session();

            /**
             * Pointer to the Session object. Used to map
             * C-callbacks to C++ members.
             */
            static Spotify::Session* callingSession;

            // For mapping sp_playlist pointers to corresponding Spotify::Playlist pointers
            QHash< sp_playlist*, Spotify::Playlist* > m_playlistHash;

            void setCachePath(const QString& strCachePath); //set the path for the libspotify cache data (directory will be created automatically if not exists upon session initialisation)

            void setSettingsPath(const QString& strSettingsPath); //set the path for the libspotify settings data

            void init(); //initialise a new Spotify session

            void initAudio(); //init the audio / playback thread

            void setPreferredBitrate(const sp_bitrate& bitrate); //set preferred audio streaming bitrate

            void setPreferredSyncBitrate(const sp_bitrate& bitrate, bool resyncTracks = false); //set preferred offline synchronisation bitrate (resync existing tracks in case resyncTracks is true)

            void setMinBufferSize(qint32 secs); //set the size of the internal FIFO queue used to hold audio data

            void setPreBufferingActive(bool active); //set whether track pre-buffering is active (i.e., buffer next track data in advance)

            bool getPreBufferingActive(); //returns current pre-buffering state

            void setNextTrackStartPos(); //set the next track start position in the audio FIFO

            void setNewTrackStarted(bool started); //indicate start of new track audio buffer data (in playback thread FIFO)

            void setPlaybackPos(qint64 pos);

            qint64 getPlaybackPos();

            void setBufferingPos(qint32 pos);

            qint32 getBufferingPos();

            void setPlaying(bool playing);

            void resetPlaybackMembers();  //reset the playback related instance members (and removes any pending events from the playback thread)

            bool isOfflineMode(); //returns true in case logged in offline; false in case we are currently online and connected to Spotify access point.

            QString getRememberedUser(); //Returns remembered (logged-in) UserID for the session

            QString getCurrentUser(); //Returns the currently username for the currently logged-in user

            bool getOfflineTimeLeft(qint32& days, qint32& hours, qint32& minutes); //Returns time until the offline keystore expires and user is required to go online (days, hours, and minutes

            Spotify::Playlist* getStarredPlaylist(); //returns (pointer) to the playlist containing user's starred tracks (or NULL in case error has occured in allocating the playlist).//left as ref. params). Returns true on success; false on error.

            void setAllowMobileSync(bool allow); //Set whether offline syncing is allowed over mobile (e.g., 2G/3G) connections

            void setOfflineMode(bool offline, bool userRequested); //Toggle between offline / online playback mode (according to offline param). In case state change is requested by user,
                                                                   //userRequested parameter is set to true; otherwise false (requested by system i.e., due to changed online connectivity state).


            /**
             * Fetches the currently logged in user.
             * @return The logged in user (or NULL if not logged in).
             */
            Spotify::User* getUser() const { return m_currentUser; }

            /**
             * Handles login of specified user.
             * @param username Username
             * @param password Password
             * @param storeCredentals Whether user credentials should be remembered
             */
            void login( QString username, QString password, bool storeCredentials = false);

            /**
             * Reconnects / re-logins current user to Spotify. Used in case network connection etc. is temporarily lost, and reconnect is required
             * Returns SP_ERROR_OK on success; otherwise error code is set.
             */
            Spotify::Error relogin();

            /**
             * Load a track
             */
            Spotify::Error load( Spotify::Track* track );

            /**
             * Start playback of a Spotify::Track
             * @param track Track to play.
             * @param preBuffer Track should be buffered only (at end of current FIFO)
             */
            Spotify::Error play( Spotify::Track* track, bool preBuffer = false);

            /**
             * Seek to a given position in the stream.
             * @param pos Position to seek to (in msecs).
             */
            void seek( qint32 pos );

            /**
             * Stop playback.
             */
            void stop();

            /**
             * Toggle music delivery (i.e., buffering) pause /resume from Spotify.
             */
            void pauseResumeBuffering();

            /**
             * Toggle playback pause / resume (e.g, stop / continue writing data to PulseAudio).
             */
            void pauseResumePlayback();


            /**
             * Search
             *
             */
            Spotify::Error search( const QString& query, const int trackOffset, const int trackCount,
                                   const int albumOffset,const int albumCount, const int artistOffset,
                                   const int artistCount, void* userData );


            /**
             * Request to load image (identified by imageID) from the Spotify backend
             * @param imageID ID of the image to be loaded.
             * @param Pointer to Spotify::ImageContainer instance for storing the loaded image
             * @return bool
             */
            bool loadImage(const byte* imageID, Spotify::ImageContainer* pContainer);

            /**
             * Add a new playlist to the playlist container by name.
             * @param name Name of the new playlist.
             * @return Error
             */
            bool addPlaylist( QString name );

            /**
             * Add a new playlist to the playlist container by link.
             * @param lin Link to the existing playlist.
             * @return Error code.
             */
            bool addPlaylist( Spotify::Link link );

            /**
             * Remove a playlist.
             * @param playlist The index of the playlist to remove.
             * @return Error code.
             */
            bool removePlaylist( const int index );

            /**
             * Move a playlist
             * @param index The index of the playlist to move.
             * @param position The position to move the playlist to.
             * @return Error code.
             */
            bool movePlaylist( const int index, int position );

            /**

             * Set whether the specified playlist should be synchronised for offline usage.
             * @param pPlaylist playlist to be made available / unavailable in offline mode
             * @param offline true in case playlist should be made available in offline mode; otherwise set to false
             */
            void setPlaylistOfflineMode(Spotify::Playlist* pPlaylist, bool offline);


            /**
             * Browse an artist.
             * @param artist Artist to be browsed. The artist metadata does not have to be loaded.
             * @return TRUE on success, otherwise FALSE
             */
            bool browse( Spotify::Artist* artist );

            /**
             * Browse an album
             * @param album Album to be browsed. The album metadata does not have to be loaded.
             * @param pCallerID Pointer to identifier of of the caller (requester)
             * @return TRUE on success, otherwise FALSE.
             */
            bool browse( Spotify::Album* album, qint32* pCallerID);


            //libspotify callbacks

            // Artist Browsing callbacks:
            static void SP_CALLCONV artistBrowseCompleteCallback( sp_artistbrowse* ab, void* userdata );

            // Album Browsing callbacks:
            static void SP_CALLCONV albumBrowseCompleteCallback( sp_albumbrowse* ab, void* userdata );

            // Playlist Container callbacks:
            static void SP_CALLCONV playlistAddedCallback( sp_playlistcontainer* pc, sp_playlist* playlist,
                                               int position, void* userdata );

            static void SP_CALLCONV playlistRemovedCallback( sp_playlistcontainer* pc, sp_playlist* playlist,
                                                 int position, void* userdata );


            static void SP_CALLCONV playlistMovedCallback( sp_playlistcontainer* pc, sp_playlist* playlist,
                                               int oldPosition, int newPosition, void* userdata );


            static void SP_CALLCONV playListContainerLoadedCallback(sp_playlistcontainer *pc, void* userdata);

            // Image callbacks:
            static void SP_CALLCONV imageLoadedCallback( sp_image* image, void* userdata );

            // Search callbacks:
            static void SP_CALLCONV searchCompleteCallback( sp_search* search, void* userdata );

            // Session callbacks:
            static void SP_CALLCONV loggedInCallback( sp_session* session, sp_error error );
            static void SP_CALLCONV loggedOutCallback( sp_session* session );
            static void SP_CALLCONV metadataUpdatedCallback( sp_session* session );
            static void SP_CALLCONV connectionErrorCallback( sp_session* session, sp_error error );
            static void SP_CALLCONV  messageToUserCallback( sp_session* session, const char* message );
            static void SP_CALLCONV notifyMainThreadCallback( sp_session* session );
            static int SP_CALLCONV musicDeliveryCallback( sp_session* session, const sp_audioformat* format,
                                               const void* frames, int num_frames );
            static void SP_CALLCONV playTokenLostCallback( sp_session* session );
            static void SP_CALLCONV logMessageCallback( sp_session* session, const char* message );
            static void SP_CALLCONV endOfTrackCallback( sp_session* session );
            static void SP_CALLCONV streamingError( sp_session* session, sp_error error );
            static void SP_CALLCONV offlineStatusUpdated( sp_session* session);
            static void SP_CALLCONV offlineSynchronistationError( sp_session* session, sp_error error);

            /**
              * Forget the currently logged-in user's credentials (in case stored by libspotify)
             */
            void forgetCurrentUser();

            /**
             * Log the current user out of Spotify.
             */
            void logout();

        private slots:

            void updateMetadata(); //metadata has been updated (for e.g., playlists)
            void searchDeletedSlot( QObject* parent );
            void artistCatalogueDeletedSlot( QObject* parent );
            void albumBrowserDeletedSlot( QObject* parent );
            void OnProcessThreadEvents();
            void OnRootListContainerLoaded(sp_playlistcontainer* pc);
            void OnPlayListAdded(sp_playlist* playlist , qint32 position);
            void OnPlayListRemoved(sp_playlist* playlist, qint32 position);
            void OnImageLoaded(sp_image* image, void* userdata);
            void OnMusicDelivery(int num_frames, const sp_audioformat* format, QByteArray buffer);
            void OnSearchComplete(sp_search* search);
            void OnAlbumBrowserReady(sp_albumbrowse* ab, qint32*);
            void OnArtistBrowseComplete( sp_artistbrowse* ab );
            void OnResumeBuffering();
            void OnPlaybackPositionUpdated(qint64 duration);
            void OnPlaybackFinished();
            void OnPlayingState(bool state);
            void OnOfflineStatusUpdated();
            void OnOnlineStateChanged(bool online);

        signals:
            /**
             * Signal emitted when an error has occured.
             * @param error Error object indicating the error.
             */
            void error( Spotify::Error error );

            /**
             * Signal emitted when a user is logged in.
             * @param error Error object indicating whether the login was successfull or not.
             */
            void loggedIn( Spotify::Error error );

            /**
             * Signal emitted when a user is logged out.
             */
            void loggedOut();

            /**
             * Signal emitted when meta data has been updated.
             */
            void metadataUpdated();

            /**
             * Signal emitted when a search has completed. The receiver will have
             * ownership of all Spotify::Track pointers and is responsible for deleting the data
             * when these are no longer needed.
             * @param tracks List of Spotify::Track pointers.
             * @param userdata Pointer to an arbitrary structure. Can be used to identify the owner
             * of the search.
             */
            void searchComplete( Spotify::Search* search );

            void artistCatalogueReady( Spotify::ArtistCatalogue* catalogue );

            void albumBrowserReady( Spotify::AlbumBrowser*, qint32 );
            /**
             * Signal emitted when a message to the user has been received.
             */
            void messageToUser( QString message );

            /**
             * Signal emitted when cover art has been received.
             * @param image QImage representation of the cover art.
             * @param callerID ID of intended receiver of result.
             */
            void coverArtReady( QImage image, qint32 callerID );

            /**
             * Signal emitted when the root playlist container has been loaded (internal usage)
             */

            void rootListContainerLoaded(sp_playlistcontainer*);

            /**
             * Signal emitted when a playlist has been added / loaded to playlist container at specified position (internal usage)
             */

            void onPlaylistAdded(sp_playlist*, qint32);

            /**
             * Signal emitted when a playlist has been removed from playlist container at specified position (internal usage)
             */

            void onPlaylistRemoved(sp_playlist*, qint32);

            /**
             * Signal emitted when a playlist has been renamed (internal usage)
             */

            void onPlaylistRenamed(sp_playlist*);

            /**
             * Signal emitted when tracks have been added to a playlist (internal usage)
             */

            void onTracksAdded(sp_playlist*, sp_track* const*, int, int);

             /**
             * Signal emitted when image data has been loaded (e.g., album coverart)
             */
            void onImageLoaded(sp_image*, void*); 

            /**
            * Signal emitted when offline status has been updated
            */
            void onOfflineStatusUpdated();

            /**
            * Signal emitted when offline synchronisation error status have been updated (parameter contains error message)
            */
            void offlineSyncError(const QString&);

            /**
             * Signal emitted when playlist containers have been received. The Spotify::Session object
             * owns the Spotify::Playlist data and the Spotify::Track data within and will delete
             * these when destructed.
             * @param playlists List of Spotify::Playlist pointers.
             */
            void playlistContainersReady( QList<Spotify::Playlist*> );

	      /**
             * Signal emitted when track data for the playlist is ready.
             */
            void playlistTracksReady(Spotify::Playlist*);

            /**
             * Signal emitted when a new playlist has been added. The Spotify::Session object
             * owns the Spotify::Playlist data and the Spotify::Track data within and will delete
             * these when destructed.
             * @playlist Playlist added.
             * @position The index of the new playlist.
             */
            void playlistAdded( Spotify::Playlist* playlist, qint32 position );


            /**
             * Signal emitted when a playlist has been moved.
             * @param oldPosition The old index of the playlist.
             * @param newPosition The new index of the playlist.
             */
            void playlistMoved( int oldPosition, int newPosition );

            /**
             * Signal emitted when a playlist has been removed.
             * @param position The index of the playlist that has been removed.
             */
            void playlistRemoved( qint32 position );

            /**
             * Signal emitted when playback has finished.
             */
            void playbackFinished();


            /**
             * Signal emitted to indicate current playback state (playing (true) / paused (false))
             */

            void playing(bool);

            /**
             * Signal emitted when the current buffering amount has been updated.
             * @param bufferedAmout The amount buffered of the currently loaded track (in milliseconds).
             */
            void trackBufferingAmountUpdated( quint32 bufferedAmount );

            /**
             * Signal emitted when the current playback position in the track has been updated.
             * @param pos The current playback position in the track (in milliseconds).
             */
            void playbackPositionUpdated( qint64 pos );


            /**
             * Signal emitted when the playtoken has been lost.
             */
            void playTokenLost();

            void onNotifyMainThread();

            /**
             * Signal emitted when metadata has been updated for the specified playlist
             */
            void playlistMetadataUpdated(sp_playlist*);

            void onMusicDelivery(int, const sp_audioformat*, QByteArray);

            void onSearchComplete(sp_search*);

            void onAlbumBrowserReady(sp_albumbrowse*, qint32*);

            void onArtistBrowseComplete(sp_artistbrowse*);

            void bufferNextTrack();

            /**
             * Signal emitted when offline sync status updated and sync is in progress (first param indicates if it is new sync operation; second parameter gives percentage completed)
             */
            void offlineSyncStatusUpdated(bool, qint32);

            /**
             * Signal emitted when current offline state has changed, e.g., online -> offline (boolean specifies new state; true = offline mode active).
             */
            void offlineModeStateChanged(bool);


        private:

            Spotify::Error m_error;
            Spotify::Track* m_currentTrack;

            ConnectionManager* m_pConnMan; //connection mananger instance (get current online status etc.)

            sp_session_config m_config;
            QString m_strCachePath; //libspotify cache path
            QString m_strSettingsPath; //libspotify settings path
            sp_session* m_session; //current session instance
            sp_playlistcontainer* m_playlistcontainer;
            Spotify::User* m_currentUser;
            bool m_userReqLogout; //set to true in case logout process has been initiated by user
            sp_connection_rules m_connRules; //currently set connection rules
            sp_connectionstate m_connState; //current session connection status

            QByteArray m_audioFragmentBuffer; //temporary buffer that holds enough samples (i.e., one sec of audio); the buffer is then appended to main playback thread FIFO.
            qint32 m_framesConsumed; //number of samples currently held in the temporary audio buffer

            bool m_allPlaylistsLoaded; //TRUE in case metadata for all playlists in the root container has been loaded; false otherwise.

            quint32 m_bufferedAmount; //current buffered amount (of the currently loaded track)
            qint64 m_playbackPos; //current playback position for the currently loaded track

            bool m_buffering; //true in case we are currently buffering
            bool m_ProcessEvents; //set to true when events in the internal libspotify threads should be processed
            bool m_preBufActive; //true if track pre-buffering is enabled
            QTimer *m_pProcEvtTimer;
            bool m_offlineSyncing;  //set to true in case offline sync is in progress; false otherwise
            bool m_offlineMode;  //true in case offline mode is active; false otherwise
            bool m_offlineModeForced; //true in case offline mode has been forced by the user (i.e., not automatically); false otherwise

            /**
             * Audio playback thread
             */
            PlaybackThread* m_pAudioThread;

            /**
             * Update the current session connection status
             * @return void
             */
             void updateConnectionState();

            /**
             * Set the connection rule specified to state (on / off)
             * @param rule Rule to set / un-set
             * @param state Rule state (true = set / false = un-set)
             * @return void
             */
            void setConnectionRule(sp_connection_rules rule, bool state);

            /**
             * Set  the current type of connection (e.g., wifi / 3G etc.)
             * @param type Current connection type
             * @return void
             */
            void setConnectionType(sp_connection_type type);

            /**
             *  Set connection type / rules
             * @return void
             */
            void setNetworkState();

            //sets connection type / rules

            // For maintaing a sorted (by index) list of Spotify::Playlist pointers
            QList< Spotify::Playlist* > m_playlistList;

            //playlist instance holding user's starred tracks
            Spotify::Playlist* m_pStarredPlaylist;

            // For mapping sp_search pointers to Spotify::Search pointers
            QHash< sp_search*, Spotify::Search* > m_searchHash;

            // For mapping sp_artistbrowse pointers to Spotify::ArtistCatalogue pointers:
            QHash< sp_artistbrowse*, Spotify::ArtistCatalogue* > m_artistBrowseHash;

            // For mapping sp_albumbrowse pointers to Spotify::AlbumBrowser pointers:
            QHash< sp_albumbrowse*, Spotify::AlbumBrowser* > m_albumBrowseHash;

            //methods to be invoked by corresponding callbacks

            void artistBrowseCompleteWrapper( sp_artistbrowse* ab, void* userdata );

            void albumBrowseCompleteWrapper( sp_albumbrowse* ab, void* userdata );


            void playlistAddedWrapper( sp_playlistcontainer* pc, sp_playlist* playlist, int position,
                                       void* userdata );


            void playlistRemovedWrapper( sp_playlistcontainer* pc, sp_playlist* playlist,
                                         int position, void* userdata );


            void playlistMovedWrapper( sp_playlistcontainer* pc, sp_playlist* playlist,
                                       int oldPosition, int newPosition, void* userdata );


            void playListContainerLoadedWrapper(sp_playlistcontainer *pc, void* userdata);


            void imageLoadedWrapper( sp_image* image, void* userdata );


            void searchCompleteWrapper( sp_search* search, void* userdata );

            void loggedInWrapper( Spotify::Error error );

            void loggedOutWrapper();

            void metadataUpdatedWrapper();

            void connectionErrorWrapper( Spotify::Error error );

            void messageToUserWrapper( QString message );

            void notifyMainThreadWrapper();

            int musicDeliveryWrapper( const sp_audioformat* format, const void* frames, int num_frames );

            void playTokenLostWrapper( sp_session* session );

            void logMessageWrapper( QString message );

            void endOfTrackWrapper();

            void streamingErrorWrapper(sp_error error);

            void offlineStatusUpdatedWrapper();

            void offlineSynchronisationErrorWrapper(sp_error error);
    };
}



Q_DECLARE_METATYPE( Spotify::Playlist* );
Q_DECLARE_METATYPE( Spotify::Track* );
Q_DECLARE_METATYPE( Spotify::Search* );
#endif // SPOTIFYSESSION_H
