//ArtistView.cpp: Implementation of the ArtistView class.

#include "artistview.h"

ArtistView::ArtistView(Spotify::Session* pSession, QWidget* pParent)
    : QDialog(pParent)
{
    // set up the view UI etc

   // setAttribute(Qt::WA_Maemo5StackedWindow); //set flag that window is part of window stack

    m_pSession = pSession;

    setWindowTitle(tr("Browse Artist"));

    m_BiographyDlg.setWindowTitle(tr("Artist Biography"));

    //setup signal / slot connections
    connect(m_pSession,SIGNAL(artistCatalogueReady(Spotify::ArtistCatalogue*)),SLOT(OnArtistCatalogueReady(Spotify::ArtistCatalogue*)));
    connect(&m_BiographyBtn,SIGNAL(clicked()),SLOT(OnArtistBiographyBtn()));

    m_pAlbumListView = new QListView(this);
    m_pAlbumListViewDelegate = new ListViewDelegate(this);
    m_pAlbumListViewDelegate->setHeaderFontPointSize(15);
    m_pAlbumListViewDelegate->setSubHeaderFontPointSize(14);
    m_pAlbumListModel = new QStandardItemModel(this);

    //connect view to item delegate and model
    m_pAlbumListView->setModel(m_pAlbumListModel);
    m_pAlbumListView->setItemDelegate(m_pAlbumListViewDelegate);
    connect(m_pAlbumListView,SIGNAL(activated(QModelIndex)),SLOT(OnAlbumActivated(QModelIndex)));

    m_pPortraitContainer = new Spotify::ImageContainer(this);
    connect(m_pPortraitContainer,SIGNAL(loaded()),SLOT(OnArtistPortraitLoaded()));

    QFont captionFont;
    captionFont.setFamily(QApplication::font().family());
    captionFont.setPointSize(19);
    m_ArtistCaption.setFont(captionFont);
    m_ArtistCaption.setWordWrap(true);

    m_ArtistPortrait.setFixedSize(ARTIST_PORTRAIT_WIDTH, ARTIST_PORTRAIT_HEIGHT);

    m_BiographyBtn.setText(tr("Biography"));

    m_pGridLayout = new QGridLayout();

    m_pGridLayout->addWidget(&m_ArtistPortrait,0,0,3,1);
    m_pGridLayout->addWidget(&m_ArtistCaption,0,1,1,1);
    m_pGridLayout->addWidget(&m_BiographyBtn,1,1,1,1);
    m_pGridLayout->addWidget(m_pAlbumListView,2,1,1,1);

    setLayout(m_pGridLayout);

}

ArtistView::~ArtistView()
{
}

void ArtistView::setArtist(Spotify::Artist* pArtist)
{
    //set current artist
    bool status;

    if(m_pSession && pArtist)
    {
        m_BiographyBtn.setVisible(false);

       // m_ArtistCaption.clear();
        //m_ArtistCaption.setVisible(false);

        m_ArtistPortrait.clear();

        m_pAlbumListModel->clear(); //clear model rows

        status = m_pSession->browse(pArtist); //request to browse specified artist details

        if(!status)
        {
            //error occured
            QMaemo5InformationBox::information(this,tr("Fetching of artist details failed!"));
        }
        else
        {
            //enable progress indication, browse call succeeded
            QMaemo5InformationBox::information(this,tr("Loading ") + pArtist->getName());
            m_ArtistCaption.setText(pArtist->getName());
            setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true);
        }
    }
}

void ArtistView::OnArtistCatalogueReady(Spotify::ArtistCatalogue* pArtistCatalogue)
{
    //artist catalogue request ready

    Spotify::Error error;

    if(pArtistCatalogue)
    {
        error = sp_artistbrowse_error(pArtistCatalogue->getArtistBrowse());
        if(error.isError())
        {
            //error occured (artist browse request failed)
            QMaemo5InformationBox::information(this,tr("Fetching of artist details failed: ") + error.toString());
        }
        else
        {
            //artist browse reques successful
            m_pArtistCatalogue = pArtistCatalogue;
            m_pArtistCatalogue->load(); //load artist information
            if(m_pArtistCatalogue->getBiography()!="")
                m_BiographyBtn.setVisible(true); //artist biography available
            if(m_pArtistCatalogue->getPortraits().size()>0)
            { //request to load artist picture
                m_pSession->loadImage(m_pArtistCatalogue->getPortraits().at(0),m_pPortraitContainer); //request artist portrait
            }
            //add artist albums to listview
            addAlbums();
        }
    }

    setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false); //hide progress indicator
}

void ArtistView::addAlbums()
{
    //add artist albums to the listview

    QPixmap iconMap;

    iconMap.load("/usr/share/images/icon-type-album.png"); //load album type icon from file

    QIcon albumTypeIcon(iconMap);

    int currAlbumType = -1;

    qSort(m_pArtistCatalogue->getAlbums()->begin(),m_pArtistCatalogue->getAlbums()->end(),albumsByTypeYearDesc); //sort albums first by type, then by year

    for(int pos = 0; pos < m_pArtistCatalogue->getAlbums()->size(); pos++)
    {
        Spotify::Album* currAlbum = m_pArtistCatalogue->getAlbums()->at(pos);
        if(currAlbum)
        {
            if(currAlbum->isAvailable())
            { //album is available
                if(currAlbumType!=currAlbum->type())
                {
                    //new type, add header item to listview
                    currAlbumType = currAlbum->type();
                    QStandardItem* headerItem = new QStandardItem();
                    headerItem->setEditable(false);
                    headerItem->setEnabled(false);
                    headerItem->setData(true,ListViewDelegate::headerItemRole); //item is special header item
                    headerItem->setData(currAlbum->getCategory(),ListViewDelegate::headerRole);
                    headerItem->setData(QColor(0,100,0),ListViewDelegate::headerItemColorRole); //set header item color to green
                    headerItem->setData(18,ListViewDelegate::headerItemSizeRole); //set font size used for header item
                    m_pAlbumListModel->appendRow(headerItem);
                }
                QStandardItem* albumItem = new QStandardItem();
                albumItem->setEditable(false);
                albumItem->setData(albumTypeIcon,ListViewDelegate::iconRole);
                albumItem->setData(currAlbum->getName(),ListViewDelegate::headerRole);
                if(currAlbum->getArtist())
                {
                    albumItem->setData(currAlbum->getArtist()->getName(),ListViewDelegate::subHeaderLeftJustifyRole);
                }
                if(currAlbum->getYear()!=0) //assign year only if not zero
                    albumItem->setData(QString::number(currAlbum->getYear()),ListViewDelegate::subHeaderRightJustifyRole);
                albumItem->setData(pos,ListViewDelegate::customDataRole); //assign position in internal album list
                m_pAlbumListModel->appendRow(albumItem);
            }
        }
    }

    if(m_pAlbumListModel->rowCount()>0)
    { //scroll to top, no current / selected row by default
        m_pAlbumListView->scrollToTop();
        QModelIndex index = m_pAlbumListModel->index(-1,-1);
        m_pAlbumListView->setCurrentIndex(index);
    }
}

void ArtistView::OnAlbumActivated(QModelIndex albumIndex)
{
    //an album has been activated (selected) from the listview
   Spotify::Album* pAlbum = m_pArtistCatalogue->getAlbums()->at(albumIndex.data(ListViewDelegate::customDataRole).toInt());

   if(pAlbum)
   {
       emit showAlbumView(pAlbum); //request to show the album view dialog
   }
}

void ArtistView::OnArtistBiographyBtn()
{
    //display artist biography

    if(m_pArtistCatalogue)
    {
        m_BiographyDlg.setText(m_pArtistCatalogue->getBiography());
        m_BiographyDlg.showMaximized();
    }
}

void ArtistView::OnArtistPortraitLoaded()
{ //Album portrait ready / loaded

    if(m_pPortraitContainer)
        m_ArtistPortrait.setPixmap(QPixmap::fromImage(m_pPortraitContainer->getImage()).scaled(ARTIST_PORTRAIT_WIDTH,ARTIST_PORTRAIT_HEIGHT));

}
