import QtQuick 1.0
import "components"
import "scripts/createobject.js" as ObjectCreator
import "scripts/dailymotion.js" as DM
import "scripts/videolistscripts.js" as Scripts

Page {
    id: root

    property string groupId
    property bool isMember: false
    property string videoFeed

    function setGroup(id) {
        groupId = id;
        videoFeed = "https://api.dailymotion.com/group/" + id + "/videos?limit=30&family_filter=" + Settings.safeSearch.toString() + "&fields=" + _DM_VIDEO_FIELDS;
        DM.getVideos(videoFeed);
        checkIfMember();
    }

    function checkIfMember() {
        var i = 0;
        while ((!isMember) && (i < dailymotionGroupModel.count)) {
            if (dailymotionGroupModel.get(i).id == groupId) {
                isMember = true;
            }
            i++;
        }
    }

    function goToVideoInfo(video) {
        pageStack.push(Qt.resolvedUrl("DMInfoPage.qml"));
        pageStack.currentPage.setVideo(video);
    }


    function selectAll() {
        Scripts.selectAll()
    }

    function selectNone() {
        Scripts.selectNone();
    }

    function addVideosToFavourites() {
        DM.addVideosToFavourites();
    }

    function deleteVideosFromFavourites() {
        DM.deleteVideosFromFavourites();
    }

    function showPlaylistDialog() {
        DM.showPlaylistDialog();
    }

    function addVideosToPlaybackQueue() {
        DM.addVideosToPlaybackQueue();
    }

    function addVideosToDownloadQueue(convertToAudio) {
        DM.addVideosToDownloadQueue(convertToAudio);
    }

    function openOrCloseMenu() {
        if (videoList.checkList.length === 0) {
            if (menu1.status == DialogStatus.Closed) {
                menu1.open();
            }
            else {
                menu1.close();
            }
        }
        else {
            if (menu2.status == DialogStatus.Closed) {
                menu2.open();
            }
            else {
                menu2.close();
            }
        }
    }

    tools: Menu {

        menuItems: [
            MenuItem {
                text: isMember ? qsTr("Leave this group") : qsTr("Join this group")
                onClicked: isMember ? Dailymotion.leaveGroup(groupId) : Dailymotion.joinGroup(groupId)
                visible: Dailymotion.userSignedIn
            },

            MenuItem {
                text: videoList.checkList.length === 0 ? qsTr("Select all") : qsTr("Select none")
                closeMenuWhenClicked: false
                enabled: videoList.count > 0
                onClicked: videoList.checkList.length === 0 ? selectAll() : selectNone()
            },

            MenuItem {
                text: qsTr("Add to favourites")
                visible: Dailymotion.userSignedIn
                enabled: videoList.checkList.length > 0
                onClicked: addVideosToFavourites()
            },

            MenuItem {
                text: qsTr("Add to playlist")
                visible: Dailymotion.userSignedIn
                enabled: videoList.checkList.length > 0
                onClicked: showPlaylistDialog()
            },

            MenuItem {
                text: qsTr("Play")
                visible: Settings.mediaPlayer == "cutetubePlayer"
                enabled: videoList.checkList.length > 0
                onClicked: addVideosToPlaybackQueue()
            },

            MenuItem {
                text: qsTr("Download")
                enabled: videoList.checkList.length > 0
                onClicked: addVideosToDownloadQueue(false)
            },

            MenuItem {
                text: qsTr("Download (audio)")
                enabled: videoList.checkList.length > 0
                onClicked: addVideosToDownloadQueue(true)
            }
        ]
    }

    Connections {
        target: Dailymotion
        onGroupJoined: {
            if (pageStack.currentPage == root) {
                isMember = true;
                messages.displayMessage(qsTr("You have joined this group"));
            }
        }
        onGroupLeft: {
            if (pageStack.currentPage == root) {
                isMember = false;
                messages.displayMessage(qsTr("You have left this group"));
            }
        }
    }

    ListView {
        id: videoList

        property variant checkList : []

        anchors { fill: parent; topMargin: titleArea.height }
        maximumFlickVelocity: Settings.maximumFlickVelocity
        highlightRangeMode: ListView.NoHighlightRange
        cacheBuffer: 2500
        clip: true
        onAtYEndChanged: if ((atYEnd) && (videoListModel.moreResults) && (!videoListModel.loading)) DM.getVideos(videoFeed);
        footer: Item {
            id: footer

            width: videoList.width
            height: 100
            visible: videoListModel.loading
            opacity: footer.visible ? 1 : 0

            BusyDialog {
                anchors.centerIn: footer
                opacity: footer.opacity
            }
        }

        model: ListModel {
            id: videoListModel

            property bool loading : false
            property bool error: false
            property bool moreResults: false
            property int page: 1

            function reset() {
                videoListModel.clear();
                videoListModel.page = 1;
                videoListModel.moreResults = false;
                videoListModel.error = false;
                DM.getVideos(videoFeed);
            }
        }

        delegate: DMListDelegate {
            id: delegate

            function addOrRemoveFromCheckList() {
                var cl = videoList.checkList;
                if (!delegate.checked) {
                    cl.push(index);
                }
                else {
                    for (var i = 0; i < cl.length; i++) {
                        if (cl[i] == index) {
                            cl.splice(i, 1);
                        }
                    }
                }
                videoList.checkList = cl;
            }

            checked: Scripts.indexInCheckList(index)
            onClicked: {
                if (videoList.checkList.length === 0) {
                    goToVideoInfo(ObjectCreator.cloneObject(videoListModel.get(index)));
                }
                else {
                    addOrRemoveFromCheckList();
                }
            }
            onPressAndHold: addOrRemoveFromCheckList()
            onPlayClicked: if (videoList.checkList.length === 0) playVideos([ObjectCreator.cloneObject(videoListModel.get(index))]);
        }

        ScrollBar {}
    }

    Label {
        id: noResultsText

        anchors.centerIn: videoList
        font.pixelSize: _LARGE_FONT_SIZE
        font.bold: true
        color: "#4d4d4d"
        horizontalAlignment: Text.AlignHCenter
        verticalAlignment: Text.AlignVCenter
        text: qsTr("No videos found")
        visible: (!videoListModel.loading) && (videoListModel.count == 0)
    }
}
