#include "downloadmanager.h"

#include <QtNetwork/QNetworkRequest>
#include <QtNetwork/QNetworkReply>
#include <QRegExp>
#include <QDebug>
#include <QMap>
#include <QList>
#include <QUrl>

DownloadManager::DownloadManager(QObject *parent) :
    QObject(parent), downloading(false) {
    dlMap["720p"] = 22;
    dlMap["480p"] = 35;
    dlMap["360p"] = 34;
    dlMap["High quality"] = 18;
    dlMap["Mobile"] = 5;

    connect(this, SIGNAL(gotVideoUrl(QByteArray)), this, SLOT(performDownload(QByteArray)));
}

void DownloadManager::setNetworkAccessManager(QNetworkAccessManager *manager) {
    nam = manager;
}

void DownloadManager::setDownloadQuality(const QString &quality) {
    downloadFormat = dlMap.value(quality, 18);
}

void DownloadManager::getVideoUrl(const QString &playerUrl) {
    downloadReply = nam->get(QNetworkRequest(QUrl(playerUrl)));
    connect(downloadReply, SIGNAL(finished()), this, SLOT(parseVideoPage()));
}

void DownloadManager::parseVideoPage() {
    if (downloadReply->error()) {
        emit statusChanged(tr("Failed"));
        return;
    }
    QMap<int, QByteArray> formats;
    QByteArray response = downloadReply->readAll();
    response = response.simplified().replace(QByteArray(" "), QByteArray(""));
    int pos = response.indexOf("\",\"fmt_url_map\":\"") + 17;
    int pos2 = response.indexOf('\"', pos);
    response = response.mid(pos, pos2 - pos);
    QList<QByteArray> parts = response.split('|');
    int key = parts.first().toInt();
    for (int i = 1; i < parts.length(); i++) {
        QByteArray part = parts[i];
        QList<QByteArray> keyAndValue = part.split(',');
        QByteArray url = keyAndValue.first().replace(QByteArray("\\/"), QByteArray("/")).replace(QByteArray("\\u0026"), QByteArray("&"));
        formats[key] = url;
        key = keyAndValue.last().toInt();
    }
    QList<int> flist;
    flist << 22 << 35 << 34 << 18 << 5;
    QByteArray videoUrl = "";
    QString quality;
    int index = flist.indexOf(downloadFormat);
    while ((videoUrl == "") && (index < flist.size())) {
        videoUrl = formats.value(flist.at(index), "");
        quality = dlMap.key(flist.at(index));
        index++;
    }
    if (videoUrl == "") {
        emit statusChanged(tr("Failed"));
    }
    else {
        emit gotVideoUrl(videoUrl);
        emit qualityChanged(quality);
    }
}

void DownloadManager::pauseDownload() {
    downloadReply->abort();
}

void DownloadManager::cancelDownload() {
    downloadReply->abort();
    output.remove(output.fileName());
    emit downloadCancelled();
}

void DownloadManager::startDownload(const QString &filePath, const QString &url) {
    downloading = true;
    output.setFileName(filePath + ".partial");
    if (output.exists()) {
        //        qDebug() << "File exists";
        if (!output.open(QIODevice::Append)) {
            //            qDebug() << "No write permissions";
            downloading = false;
            return;                 // skip this download
        }
    }
    else if (!output.open(QIODevice::WriteOnly)) {
        qDebug() << "No write permissions";
        downloading = false;
        return;                 // skip this download
    }

    if (url.contains("youtube")) {
        // It's a YouTube video, so we must get the URL from the web page

        getVideoUrl(url);
    }
    else {
        // It's a pR0n video :D
        performDownload(url.toAscii());
    }
}

void DownloadManager::performDownload(const QByteArray &videoUrl) {
    //    qDebug() << QUrl::fromEncoded(videoUrl);
    QNetworkRequest request(QUrl::fromEncoded(videoUrl));

    if (output.size() > 0) {
        request.setRawHeader("Range", "bytes=" + QByteArray::number(output.size()) + "-"); // Set 'Range' header if resuming a download
    }

    downloadReply = nam->get(request);
    emit statusChanged(tr("Downloading"));
    downloadTime.start();
    connect(downloadReply, SIGNAL(downloadProgress(qint64, qint64)), this, SLOT(updateProgress(qint64,qint64)));
    connect(downloadReply, SIGNAL(finished()), this, SLOT(downloadFinished()));
    connect(downloadReply, SIGNAL(readyRead()), this, SLOT(downloadReadyRead()));
}

void DownloadManager::updateProgress(qint64 bytesReceived, qint64 bytesTotal) {
    double speed = bytesReceived * 1000.0 / downloadTime.elapsed();
    QString unit;
    if (speed < 1024) {
        unit = "bytes/sec";
    } else if (speed < 1024*1024) {
        speed /= 1024;
        unit = "kB/s";
    } else {
        speed /= 1024*1024;
        unit = "MB/s";
    }

    emit progressChanged(bytesReceived, bytesTotal, QString::fromLatin1("%1 %2")
                         .arg(speed, 3, 'f', 1).arg(unit));
}

void DownloadManager::downloadFinished() {
    QUrl redirect = downloadReply->attribute(QNetworkRequest::RedirectionTargetAttribute).toUrl();
    if (!redirect.isEmpty()) {
        performDownload(redirect.toEncoded()); // Follow redirect :P
    }
    else {
        output.close();
        downloading = false;
        QString status;
        if (downloadReply->error()) {
            if (downloadReply->error() == QNetworkReply::OperationCanceledError) {
                status = tr("Paused");
            }
            else {
                status = tr("Failed");
            }
            emit statusChanged(status);
        }
        else {
            QString filename = output.fileName().left(output.fileName().lastIndexOf("."));
            int num = 1;
            bool fileSaved = output.rename(filename);
            while ((!fileSaved) && (num < 10)) {
                if (num == 1) {
                    filename = filename.insert(filename.lastIndexOf("."), "(" + QByteArray::number(num) + ")");
                }
                else {
                    filename = filename.replace(filename.lastIndexOf("(" + QByteArray::number(num - 1) + ")"), 3, "(" + QByteArray::number(num) + ")");
                }
                //                qDebug() << filename;
                fileSaved = output.rename(filename);
                num++;
            }
            emit downloadCompleted(filename);
        }
    }
}

void DownloadManager::downloadReadyRead() {
    output.write(downloadReply->readAll());
}
