// Copyright 2010 Jochen Becher
//
// This file is part of MovieSchedule.
//
// MovieSchedule is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// MovieSchedule is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with MovieSchedule.  If not, see <http://www.gnu.org/licenses/>.

#include "connectivitymanager.h"

#include <QDBusMessage>
#include <QDBusInterface>
#ifdef MAEMO_SDK
#include <icd/dbus_api.h>
#endif

ConnectivityManager::ConnectivityManager(QObject *parent)
    : QObject(parent),
    _dbus_connection(QDBusConnection::systemBus()),
#ifdef MAEMO_SDK
    _interface(new QDBusInterface(ICD_DBUS_API_INTERFACE,
                                  ICD_DBUS_API_PATH,
                                  ICD_DBUS_API_INTERFACE,
                                  _dbus_connection))
#else
    _interface(0)
#endif
{
#ifdef MAEMO_SDK
    _dbus_connection.connect(ICD_DBUS_API_INTERFACE, ICD_DBUS_API_PATH, ICD_DBUS_API_INTERFACE,
                             ICD_DBUS_API_STATISTICS_SIG, this, SLOT(StatisticsSentResult(const QDBusMessage&)));
    _dbus_connection.connect(ICD_DBUS_API_INTERFACE, ICD_DBUS_API_PATH, ICD_DBUS_API_INTERFACE,
                             ICD_DBUS_API_STATE_SIG, this, SLOT(ChangeState(const QDBusMessage&)));
#endif
}

ConnectivityManager::~ConnectivityManager()
{
#ifdef MAEMO_SDK
    _dbus_connection.disconnect(ICD_DBUS_API_INTERFACE, ICD_DBUS_API_PATH, ICD_DBUS_API_INTERFACE,
                                ICD_DBUS_API_STATE_SIG, this, SLOT(ChangeState(const QDBusMessage&)));
    _dbus_connection.disconnect(ICD_DBUS_API_INTERFACE, ICD_DBUS_API_PATH, ICD_DBUS_API_INTERFACE,
                                ICD_DBUS_API_STATISTICS_SIG, this, SLOT(StatisticsSentResult(const QDBusMessage&)));
#endif
}

void ConnectivityManager::Connect()
{
#ifdef MAEMO_SDK
    _interface->call(ICD_DBUS_API_CONNECT_REQ,
                     QVariant((unsigned int) ICD_CONNECTION_FLAG_USER_EVENT));
#endif
    UpdateConnectionState();
}

void ConnectivityManager::Disconnect()
{
#ifdef MAEMO_SDK
    _interface->call(ICD_DBUS_API_DISCONNECT_REQ,
                     QVariant((unsigned int) ICD_CONNECTION_FLAG_USER_EVENT));
#endif
}

void ConnectivityManager::UpdateConnectionState()
{
#ifdef MAEMO_SDK
    _interface->call(ICD_DBUS_API_STATE_REQ);
#endif
    emit Connected();
}

void ConnectivityManager::StatisticsSentResult(const QDBusMessage &rep)
{
    Q_UNUSED(rep);
}

void ConnectivityManager::ChangeState(const QDBusMessage &rep)
{
#ifdef MAEMO_SDK
    QString connection_error = rep.arguments().value(6).value<QString>();

    if (!connection_error.isEmpty()) {
        emit Error();
        return;
    }

    unsigned int connection_status = rep.arguments().value(7).value<unsigned int>();

    switch (connection_status) {
    case ICD_STATE_DISCONNECTED:
        emit Disconnected();
        break;
    case ICD_STATE_CONNECTING:
        break;
    case ICD_STATE_CONNECTED:
        emit Connected();
        break;
    case ICD_STATE_DISCONNECTING:
        break;
    case ICD_STATE_LIMITED_CONN_ENABLED:
        break;
    case ICD_STATE_LIMITED_CONN_DISABLED:
        break;
    case ICD_STATE_SEARCH_START:
        break;
    case ICD_STATE_SEARCH_STOP:
        break;
    case ICD_STATE_INTERNAL_ADDRESS_ACQUIRED:
        break;
    }
#else
    Q_UNUSED(rep);
#endif
}
