// Copyright 2010 Jochen Becher
//
// This file is part of MovieSchedule.
//
// MovieSchedule is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// MovieSchedule is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with MovieSchedule.  If not, see <http://www.gnu.org/licenses/>.

#include "gpsclient.h"

#ifdef QT_MOBILITY_LOCATION
#include <QGeoPositionInfoSource>
#endif

#ifdef LIBLOCATION
#include <QTimer>
#endif

#include <QUrl>
#include <QNetworkAccessManager>
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QMutexLocker>
#include <iostream>

#ifdef LIBLOCATION
static void changed(LocationGPSDevice *device, gpointer userdata)
{
    Q_UNUSED(device);
    ((GpsClient *) userdata)->GpsChanged();
}

static void connected(LocationGPSDevice *device, gpointer userdata)
{
    Q_UNUSED(device);
    ((GpsClient *) userdata)->GpsConnected();
}

static void disconnected(LocationGPSDevice *device, gpointer userdata)
{
    Q_UNUSED(device);
    ((GpsClient *) userdata)->GpsDisconnected();
}
#endif

GpsClient::GpsClient()
    :
#ifdef QT_MOBILITY_LOCATION
    _geo_position_info_source(QGeoPositionInfoSource::createDefaultSource(this)),
#endif
#ifdef LIBLOCATION
    _location_gpsd_control(0),
    _location_gps_device(0),
    _time_out_timer(new QTimer(this)),
#endif
    _network(new QNetworkAccessManager(this)),
    _search_task_id(INVALID_SEARCH_TASK_ID)
{
    {
        QMutexLocker locker(&_next_search_task_id_mutex);
        _search_task_id = _next_search_task_id++;
    }
#ifdef QT_MOBILITY_LOCATION
    connect(_geo_position_info_source, SIGNAL(positionUpdated(QGeoPositionInfo)), this, SLOT(GeoPositionUpdated(const QGeoPositionInfo &)));
    connect(_geo_position_info_source, SIGNAL(updateTimeout()), this, SLOT(GeoPositionTimedOut()));
#endif
#ifdef LIBLOCATION
    connect(_time_out_timer, SIGNAL(timeout()), this, SLOT(GpsTimedOut()));
#endif
    connect(_network, SIGNAL(finished(QNetworkReply*)), this, SLOT(ReplyFinished(QNetworkReply*)));
}

GpsClient::~GpsClient()
{
#ifdef LIBLOCATION
    if (_location_gps_device != 0)
        g_object_unref(_location_gps_device);
    if (_location_gpsd_control != 0)
        location_gpsd_control_stop(_location_gpsd_control);
#endif
}

void GpsClient::SearchLocation()
{
    _semaphore.Activate(GetSearchTaskId());
#ifdef QT_MOBILITY_LOCATION
    _geo_position_info_source->requestUpdate(30 * 1000);
    emit SearchStarted(GetSearchTaskId());
#elif defined(LIBLOCATION)
    _location_gpsd_control = location_gpsd_control_get_default();
    if (_location_gpsd_control != 0) {
        location_gpsd_control_start(_location_gpsd_control);
    }
    emit SearchStarted(GetSearchTaskId());
    _location_gps_device = (LocationGPSDevice *) g_object_new(LOCATION_TYPE_GPS_DEVICE, NULL);
    if (_location_gps_device != 0) {
        location_gps_device_reset_last_known(_location_gps_device);
        g_signal_connect(_location_gps_device, "changed", G_CALLBACK(changed), this);
        g_signal_connect(_location_gps_device, "connected", G_CALLBACK(connected), this);
        g_signal_connect(_location_gps_device, "disconnected", G_CALLBACK(disconnected), this);
    }
    _time_out_timer->start(30 * 1000);
#else
    emit SearchStarted(GetSearchTaskId());
    emit SearchError(GetSearchTaskId());
    emit SearchFinished(GetSearchTaskId(), false);
    deleteLater();
#endif
}

void GpsClient::CancelAllRunningSearchs()
{
    _semaphore.CancelAll();
}

#ifdef QT_MOBILITY_LOCATION
void GpsClient::GeoPositionUpdated(const QGeoPositionInfo &geo_position_info)
{
    if (geo_position_info.coordinate().isValid()) {
        //std::cout << "longitude " << geo_position_info.coordinate().longitude()
        //        << ", latitude " << geo_position_info.coordinate().latitude()
        //        << ", altitude " << geo_position_info.coordinate().altitude()
        //        << std::endl;
        SearchTown(QString("%1").arg(geo_position_info.coordinate().longitude()),
                   QString("%1").arg(geo_position_info.coordinate().latitude()));
        emit SearchForTownStarted(GetSearchTaskId());
    } else {
        //std::cout << "invalid coordinate received" << std::endl;
        emit SearchError(GetSearchTaskId());
        emit SearchFinished(GetSearchTaskId(), false);
        deleteLater();
    }
}

void GpsClient::GeoPositionTimedOut()
{
    //std::cout << "time-out" << std::endl;
    emit SearchError(GetSearchTaskId());
    emit SearchFinished(GetSearchTaskId(), false);
    deleteLater();
}
#endif

#ifdef LIBLOCATION
void GpsClient::GpsConnected()
{
    //std::cout << "connected" << std::endl;
}

void GpsClient::GpsChanged()
{
    if (_location_gps_device->status == LOCATION_GPS_DEVICE_STATUS_FIX
        && _location_gps_device->fix != 0
        && (_location_gps_device->fix->fields & LOCATION_GPS_DEVICE_LATLONG_SET) != 0)
    {
        //std::cout << "longitude " << _location_gps_device->fix->longitude
        //        << ", latitude " << _location_gps_device->fix->latitude
        //        << ", altitude " << _location_gps_device->fix->altitude
        //        << ", eph " << _location_gps_device->fix->eph
        //        << std::endl;
        if (_location_gps_device->fix->eph != LOCATION_GPS_DEVICE_NAN
            && _location_gps_device->fix->eph <= (20 * 1000 * 100)) // 20km
        {
            SearchTown(QString("%1").arg(_location_gps_device->fix->longitude),
                       QString("%1").arg(_location_gps_device->fix->latitude));
            emit SearchForTownStarted(GetSearchTaskId());
        } else {
            //std::cout << "waiting for better accuracy" << std::endl;
        }
    } else {
        //std::cout << "waiting for location" << std::endl;
    }
}

void GpsClient::GpsDisconnected()
{
    //std::cout << "disconnected" << std::endl;
}

void GpsClient::GpsTimedOut()
{
    //std::cout << "time-out" << std::endl;
    emit SearchError(GetSearchTaskId());
    emit SearchFinished(GetSearchTaskId(), false);
    deleteLater();
}
#endif

void GpsClient::SearchTown(const QString &longitude, const QString &latitude)
{
    // TODO: try to fetch a unique city name, at least with country code.
    // http://code.google.com/intl/en/apis/maps/documentation/geocoding/index.html#ReverseGeocoding
    QUrl url("http://maps.google.com/maps/geo");
    url.addEncodedQueryItem("q", QUrl::toPercentEncoding(latitude + "," + longitude));
    url.addEncodedQueryItem("output", QUrl::toPercentEncoding("xml"));
    _network->get(QNetworkRequest(url));
}

void GpsClient::ReplyFinished(QNetworkReply *network_reply)
{
    if (!network_reply->error()) {
        QString data = QString::fromUtf8(network_reply->readAll());
        int start = data.indexOf("<LocalityName>");
        if (start >= 0) {
            int end = data.indexOf("</LocalityName>", start);
            QString town = data.mid(start + 14, end - start - 14);
            if (!town.isEmpty()) {
                //std::cout << "Found town " << qPrintable(town) << std::endl;
                emit TownUpdate(GetSearchTaskId(), town);
                emit SearchFinished(GetSearchTaskId(), true);
                deleteLater();
            } else {
                //std::cout << "No town found in " << qPrintable(data) << std::endl;
                emit SearchError(GetSearchTaskId());
                emit SearchFinished(GetSearchTaskId(), false);
                deleteLater();
            }
        } else {
            //std::cout << "No <LocalityName> found in " << qPrintable(data) << std::endl;
            emit SearchError(GetSearchTaskId());
            emit SearchFinished(GetSearchTaskId(), false);
            deleteLater();
        }
    }
    network_reply->deleteLater();
}

void GpsClient::NetworkError(QNetworkReply::NetworkError)
{
    //std::cout << "Network error" << std::endl;
    emit SearchError(GetSearchTaskId());
    emit SearchFinished(GetSearchTaskId(), false);
    deleteLater();
}

QMutex GpsClient::_next_search_task_id_mutex;
int GpsClient::_next_search_task_id = 1;
SearchClientSemaphore GpsClient::_semaphore;
