// Copyright 2010 Jochen Becher
//
// This file is part of MovieSchedule.
//
// MovieSchedule is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// MovieSchedule is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with MovieSchedule.  If not, see <http://www.gnu.org/licenses/>.

#include "moviecontroller.h"

#include "data/movie.h"
#include "data/cinemaschedule.h"
#include "control/actioncontroller.h"
#include "control/itemmodelsortclient.h"
#include "ui/moviewindow.h"
#include "ui/movieschedulemodel.h"
#include "ui/contextdialog.h"
#include "ui/uiutils.h"
#include "searchclients/movieschedulesearchclient.h"
#include "utils/assertedlocker.h"
#include "utils/asynccall.h"

#include <QSortFilterProxyModel>
#include <iostream>

static const char *MSG_NO_MOVIE_SCHEDULE_FOUND = QT_TRANSLATE_NOOP("MovieController", "No schedule found for %1.");
static const char *MSG_MOVIE_SCHEDULE_ERROR = QT_TRANSLATE_NOOP("MovieController", "Error on fetching movie schedule.");

MovieController::MovieController(MovieWindow *movie_window, CinemaSchedule *cinema_schedule,
                                 ActionController *action_controller, ItemModelSortController *sort_controller,
                                 QThread *search_worker)
                                     : QObject(0),
                                     _movie_window(movie_window),
                                     _cinema_schedule(cinema_schedule),
                                     _action_controller(action_controller),
                                     _sort_controller(sort_controller),
                                     _search_worker(search_worker),
                                     _current_search_task_id(MovieScheduleSearchClient::INVALID_SEARCH_TASK_ID),
                                     _movie_schedule_model(0),
                                     _movie_schedule_proxy_model(new QSortFilterProxyModel(this))
{
    connect(_movie_window, SIGNAL(ScheduleEntrySelected(ScheduleEntryKey)), this, SLOT(ScheduleEntrySelected(ScheduleEntryKey)));
}

void MovieController::ShowMovie(MovieKey movie_key)
{
    CancelSearch();
    AssertedReadLocker locker(_cinema_schedule->GetLock());
    _movie_key = movie_key;
    const Movie *movie = ((const CinemaSchedule *) _cinema_schedule)->FindMovie(movie_key);
    if (movie != 0) {
        SetModel(0);
        _movie_window->SetMovieName(movie->GetName());
        _movie_window->SetMovieScheduleModel(_movie_schedule_proxy_model);
        _movie_window->show();
        MovieScheduleSearchClient *client = new MovieScheduleSearchClient(_cinema_schedule);
        _current_search_task_id = client->GetSearchTaskId();
        connect(client, SIGNAL(SearchStarted(int)), this, SLOT(SearchStarted(int)));
        connect(client, SIGNAL(Reply(int, bool)), this, SLOT(Reply(int, bool)));
        connect(client, SIGNAL(SearchFinished(int, bool)), this, SLOT(SearchFinished(int, bool)));
        connect(client, SIGNAL(Error(int)), this, SLOT(Error(int)));
        client->moveToThread(_search_worker);
        CallAsync(client, &MovieScheduleSearchClient::SearchSchedule, movie->GetKey(), movie->GetTheatersUrl());
        // search client deletes itself
    }
}

void MovieController::Cancel()
{
    _movie_window->hide();
    CancelSearch();
}

void MovieController::CancelSearch()
{
    SetModel(0);
    AssertedWriteLocker locker(_cinema_schedule->GetLock());
    _current_search_task_id = MovieScheduleSearchClient::INVALID_SEARCH_TASK_ID;
    MovieScheduleSearchClient::CancelAllRunningSearchs();
}

void MovieController::ScheduleEntrySelected(ScheduleEntryKey schedule_entry_key)
{
    ContextDialog *dialog = new ContextDialog(_cinema_schedule, _movie_window);
    connect(dialog, SIGNAL(AddToCalendar(ScheduleEntryKey)), _action_controller, SLOT(AddToCalendar(ScheduleEntryKey)));
    connect(dialog, SIGNAL(CallTheaterByPhone(CinemaKey)), _action_controller, SLOT(CallTheaterByPhone(CinemaKey)));
    connect(dialog, SIGNAL(FindRouteToTheater(CinemaKey)), _action_controller, SLOT(FindRouteToTheater(CinemaKey)));
    connect(dialog, SIGNAL(SearchMovieInWeb(MovieKey)), _action_controller, SLOT(SearchMovieInWeb(MovieKey)));
    connect(dialog, SIGNAL(SearchTheaterInWeb(CinemaKey)), _action_controller, SLOT(SearchTheaterInWeb(CinemaKey)));
    dialog->Show(schedule_entry_key);
    // dialog deletes itself
}

void MovieController::SearchStarted(int search_task_id)
{
    if (search_task_id != _current_search_task_id) {
        return;
    }
    _movie_window->SetBusy(true);
}

void MovieController::Reply(int search_task_id, bool intermediate)
{
    if (search_task_id != _current_search_task_id) {
        return;
    }
    Sort(intermediate, SLOT(SortFinished(QAbstractItemModel*,int,bool)));
}

void MovieController::Error(int search_task_id)
{
    if (search_task_id != _current_search_task_id) {
        return;
    }
    Sort(false, SLOT(SortErrorFinished(QAbstractItemModel*,int,bool)));
}

void MovieController::SearchFinished(int search_task_id, bool success)
{
    Q_UNUSED(success);
    if (search_task_id != _current_search_task_id) {
        return;
    }
    _movie_window->SetBusy(false);
}

void MovieController::Sort(bool intermediate, const char *slot)
{
    MovieScheduleModel *movie_schedule_model = new MovieScheduleModel(_cinema_schedule, _movie_key, this);
    movie_schedule_model->Update();
    QSortFilterProxyModel *sort_model = new QSortFilterProxyModel(this);
    sort_model->setSortCaseSensitivity(Qt::CaseInsensitive);
    sort_model->setSortRole(MovieScheduleModel::SortRole);
    sort_model->setDynamicSortFilter(false);
    sort_model->setSourceModel(movie_schedule_model);
    ItemModelSortClient *sort_client = new ItemModelSortClient(_sort_controller, this);
    connect(sort_client, SIGNAL(SortFinished(QAbstractItemModel*,int,bool)), this, slot);
    sort_client->Sort(sort_model, _current_search_task_id, intermediate);
    // proxy deletes itself
}

void MovieController::SortFinished(QAbstractItemModel *model, int search_task_id, bool intermediate)
{
    if (search_task_id != _current_search_task_id) {
        return;
    }
    SetModel(model);
    if (!intermediate) {
        if (_movie_schedule_model->rowCount() == 0) {
            _movie_window->SetError(tr(MSG_NO_MOVIE_SCHEDULE_FOUND).arg(_movie_key.GetName()));
        }
    }
}

void MovieController::SortErrorFinished(QAbstractItemModel *model, int search_task_id, bool intermediate)
{
    Q_UNUSED(intermediate);
    if (search_task_id != _current_search_task_id) {
        return;
    }
    SetModel(model);
    if (_movie_schedule_model->rowCount() == 0) {
        _movie_window->SetError(tr(MSG_MOVIE_SCHEDULE_ERROR));
    } else {
        UiUtils::ShowError(tr(MSG_MOVIE_SCHEDULE_ERROR));
    }
}

void MovieController::SetModel(QAbstractItemModel *model)
{
    delete _movie_schedule_proxy_model->sourceModel();
    _movie_schedule_proxy_model->setSourceModel(model);
    delete _movie_schedule_model;
    if (model != 0) {
        _movie_schedule_model = (MovieScheduleModel *) ((QSortFilterProxyModel *) model)->sourceModel();
    } else {
        _movie_schedule_model = 0;
    }
}
