// Copyright 2010 Jochen Becher
//
// This file is part of MovieSchedule.
//
// MovieSchedule is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// MovieSchedule is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with MovieSchedule.  If not, see <http://www.gnu.org/licenses/>.

#include "actioncontroller.h"

#include "data/cinemaschedule.h"
#include "data/scheduleentry.h"
#include "data/cinema.h"
#include "data/movie.h"
#include "ui/uiutils.h"
#include "utils/calendar.h"
#include "utils/timeutils.h"
#include "utils/assertedlocker.h"

#include <QtDBus>
#include <QUrl>
#include <QDesktopServices>
#include <iostream>

ActionController::ActionController(const CinemaSchedule *cinema_schedule) :
        QObject(0),
        _cinema_schedule(cinema_schedule)
{
}

void ActionController::AddToCalendar(ScheduleEntryKey schedule_entry_key)
{
#ifdef MAEMO_SDK
    ScheduleEntry schedule_entry;
    Movie movie;
    Cinema cinema;
    {
        AssertedReadLocker locker(_cinema_schedule->GetLock());
        schedule_entry = _cinema_schedule->FindScheduleEntry(schedule_entry_key);
        if (schedule_entry.IsValid()) {
            movie = *schedule_entry.GetMovie();
            cinema = *schedule_entry.GetCinema();
        }
    }
    if (schedule_entry.IsValid()) {
        Calendar calendar(this);
        QDateTime start_time;
        start_time.setDate(schedule_entry.GetDate());
        start_time.setTime(schedule_entry.GetStartTime());
        int minutes = movie.GetDuration().hour() * 60
                      + movie.GetDuration().minute();
        minutes = ((minutes + 14) / 15) * 15;
        QDateTime end_time = start_time.addSecs(minutes * 60);
        QString location = cinema.GetName();
        QString description;
        if (!cinema.GetAddress().isEmpty()) {
            if (!location.isEmpty()) {
                location += ", ";
            }
            location += cinema.GetAddress();
        }
        if (!cinema.GetTelephone().isEmpty()) {
            if (!description.isEmpty()) {
                description += "\n";
            }
            description += cinema.GetTelephone();
        }
        if (!calendar.AddEvent(movie.GetName(), start_time, end_time,
                               location, description)) {
            ShowError(tr("Adding calendar event failed."));
            return;
        }
        UiUtils::ShowInformation(
                tr("%1 (%2 %3) added to your calendar.")
                .arg(movie.GetName())
                .arg(TimeUtils::ToDateString(schedule_entry.GetDate()))
                .arg(TimeUtils::ToTimeString(schedule_entry.GetStartTime())));
    } else {
        ShowError(tr("Adding calendar event failed."));
    }
#else
    Q_UNUSED(schedule_entry_key);
    ShowError(tr("Adding calendar event failed."));
#endif
}

void ActionController::CallTheaterByPhone(CinemaKey cinema_key)
{
    Cinema cinema;
    {
        AssertedReadLocker locker(_cinema_schedule->GetLock());
        const Cinema *cinema_p = _cinema_schedule->FindCinema(cinema_key);
        if (cinema_p != 0) {
            cinema = *cinema_p;
        }
    }
    if (cinema.IsValid()) {
        if (!cinema.GetTelephone().isEmpty()) {
            if (!QDBusConnection::systemBus().isConnected()) {
                ShowError(tr("Unable to call theater by phone: communication bus not available."));
                return;
            }
            QDBusInterface iface("com.nokia.csd.Call", "/com/nokia/csd/call", "com.nokia.csd.Call", QDBusConnection::systemBus());
            if (!iface.isValid()) {
                ShowError(tr("Unable to call theater by phone: phone service not available."));
                return;
            }
            QString phone_number = TrimPhoneNumber(cinema.GetTelephone());
            QDBusReply<QDBusObjectPath> reply = iface.call("CreateWith", phone_number, 0);
            if (!reply.isValid()) {
                std::cout << qPrintable(reply.error().message()) << ": <" << qPrintable(phone_number) << ">" << std::endl;
                ShowError(tr("Unable to call theater by phone: call failed."));
                return;
            }
        }
    }
}

void ActionController::FindRouteToTheater(CinemaKey cinema_key)
{
    Cinema cinema;
    {
        AssertedReadLocker locker(_cinema_schedule->GetLock());
        const Cinema *cinema_p = _cinema_schedule->FindCinema(cinema_key);
        if (cinema_p != 0) {
            cinema = *cinema_p;
        }
    }
    if (cinema.IsValid()) {
        // TODO implement find route to theater
    }
}

void ActionController::SearchTheaterInWeb(CinemaKey cinema_key)
{
    QString key;
    {
        AssertedReadLocker locker(_cinema_schedule->GetLock());
        const Cinema *cinema = _cinema_schedule->FindCinema(cinema_key);
        if (cinema != 0) {
            key = cinema->GetName() + " " + cinema->GetAddress();
        }
    }
    if (!key.isEmpty()) {
        SearchInWeb(key);
    }
}

void ActionController::SearchMovieInWeb(MovieKey movie_key)
{
    QString key;
    {
        AssertedReadLocker locker(_cinema_schedule->GetLock());
        const Movie *movie = _cinema_schedule->FindMovie(movie_key);
        if (movie != 0) {
            key = movie->GetName();
        }
    }
    if (!key.isEmpty()) {
        SearchInWeb(key);
    }
}

void ActionController::ContactAuthor()
{
    QUrl url("mailto::Jochen Becher <j.becher@ovi.com>", QUrl::TolerantMode);
    url.addQueryItem("subject", "Application 'MovieSchedule' for Nokia N900");
    QDesktopServices::openUrl(url);
}

QString ActionController::TrimPhoneNumber(const QString &phone_number)
{
    QString trimmed;
    bool plus_is_valid = true;
    foreach (QChar ch, phone_number) {
        switch (ch.toAscii()) {
        case ',':
        case '.':
        case '(':
        case ')':
        case '-':
        case ' ':
        case '\t':
        case '/':
            // trim character
            break;
        case 'p':
        case 'w':
        case 'x':
            trimmed += ch.toUpper();
            break;
        case '+':
            if (plus_is_valid) {
                trimmed += ch;
            }
            break;
        default:
            trimmed += ch;
        }
        // TODO plus_is_valid stays true as long as we are parsing a number suppression prefix (e.g. *31#)
        plus_is_valid = false;
    }
    return trimmed;
}

void ActionController::SearchInWeb(const QString &key)
{
    QUrl url("http://www.google.com/search");
    url.addQueryItem("q", key);
    if (!QDesktopServices::openUrl(url)) {
        ShowError(tr("Unable to search in web: browser service not available."));
        return;
    }
}

void ActionController::ShowError(const QString &msg)
{
    UiUtils::ShowError(msg);
}
