// Copyright 2010 Jochen Becher
//
// This file is part of MovieSchedule.
//
// MovieSchedule is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// MovieSchedule is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with MovieSchedule.  If not, see <http://www.gnu.org/licenses/>.

#include "abstractsearchclient.h"

#include <QMutexLocker>
#include <iostream>

AbstractSearchClient::AbstractSearchClient(QObject *parent)
    : QObject(parent),
    _network(new QNetworkAccessManager(this)),
    _search_task_id(INVALID_SEARCH_TASK_ID),
    _start(0)
{
    connect(_network, SIGNAL(finished(QNetworkReply *)),
             this, SLOT(ReplyFinished(QNetworkReply*)));
    {
        QMutexLocker locker(&_next_search_task_id_mutex);
        _search_task_id = _next_search_task_id++;
    }
}

AbstractSearchClient::~AbstractSearchClient()
{
}

void AbstractSearchClient::Search(const QUrl &url, int start)
{
    _start = start;
    if (start == 0) {
        emit SearchStarted(_search_task_id);
    }
    QNetworkRequest request;
    QUrl xurl(url);    
    if (_start != 0) {
        xurl.addQueryItem("start", QString::number(_start));
    }
    //std::cout << "URL: " << qPrintable(QString(xurl.toEncoded())) << std::endl;
    request.setUrl(xurl);
    //request.setRawHeader("User-Agent", "Mozilla/5.0 (X11; U; Linux i686; de; rv:1.9.1.9) Gecko/20100401 Ubuntu/9.10 (karmic) Firefox/3.5.9");
    request.setRawHeader("User-Agent", "Mozilla/5.0");
    //request.setRawHeader("Accept", "text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8");
    request.setRawHeader("Accept", "application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8");
    request.setRawHeader("Accept-Language", "en-gb;q=1.0,en;q=0.9,de-de;q=0.5,de;q=0.3");
    //request.setRawHeader("Accept-Encoding", "gzip,deflate");
    request.setRawHeader("Accept-Charset", "ISO-8859-1,utf-8;q=0.7,*;q=0.7");
    request.setRawHeader("Keep-Alive", "300");
    request.setRawHeader("Connection", "keep-alive");
    request.setRawHeader("Cache-Control", "max-age=0");
    QNetworkReply *reply = _network->get(request);
    connect(reply, SIGNAL(downloadProgress(qint64,qint64)), this, SLOT(DownloadProgress(qint64,qint64)));
    connect(reply, SIGNAL(error(QNetworkReply::NetworkError)), this, SLOT(NetworkError(QNetworkReply::NetworkError)));
}

void AbstractSearchClient::DownloadProgress(qint64 a,qint64 b)
{
    //std::cout << "Search Progress of " << qPrintable(objectName()) << " - " << a << ", " << b << std::endl;
    //sleep(1);
    emit Progress(_search_task_id, a, b);
}

void AbstractSearchClient::NetworkError(QNetworkReply::NetworkError error)
{
    emit SearchFinished(_search_task_id, false);
    std::cout << "ERROR: " << error << std::endl;
    emit Error(_search_task_id);
    sender()->deleteLater();
    deleteLater();
}

QMutex AbstractSearchClient::_next_search_task_id_mutex;
int AbstractSearchClient::_next_search_task_id = 1;
