// Copyright 2010 Jochen Becher
//
// This file is part of MovieSchedule.
//
// MovieSchedule is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// MovieSchedule is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with MovieSchedule.  If not, see <http://www.gnu.org/licenses/>.

#include "cinemaschedule.h"

#include "cinema.h"
#include "movie.h"

CinemaSchedule::CinemaSchedule()
    : _lock(),
    _all_cinemas_loaded(false),
    _all_movies_loaded(false)
{
}

void CinemaSchedule::Clear()
{
    _lock.AssertLockedForWrite();
    _schedules.clear();
    foreach (Movie *movie, _movies.values()) {
        delete movie;
    }
    _movies.clear();
    _all_movies_loaded = false;
    foreach (Cinema *cinema, _cinemas.values()) {
        delete cinema;
    }
    _cinemas.clear();
    _all_cinemas_loaded = false;
}

Cinema *CinemaSchedule::FindCinema(const CinemaKey &key)
{
    _lock.AssertLockedForWrite();
    if (!_cinemas.contains(key)) {
        return 0;
    }
    return _cinemas[key];
}

const Cinema *CinemaSchedule::FindCinema(const CinemaKey &key) const
{
    _lock.AssertLockedForRead();
    if (!_cinemas.contains(key)) {
        return 0;
    }
    return _cinemas[key];
}

Cinema *CinemaSchedule::AddCinema(const CinemaKey &key)
{
    _lock.AssertLockedForWrite();
    Cinema *cinema = 0;
    if (_cinemas.contains(key))
    {
        cinema = _cinemas[key];
        if (cinema != 0) {
            return cinema;
        }
    }
    cinema = new Cinema();
    cinema->SetName(key.GetName());
    cinema->SetAddress(key.GetAddress());
    _cinemas[key] = cinema;
    return cinema;
}

Movie *CinemaSchedule::FindMovie(const MovieKey &key)
{
    _lock.AssertLockedForWrite();
    if (!_movies.contains(key)) {
        return 0;
    }
    return _movies[key];
}

const Movie *CinemaSchedule::FindMovie(const MovieKey &key) const
{
    _lock.AssertLockedForRead();
    if (!_movies.contains(key)) {
        return 0;
    }
    return _movies[key];
}

Movie *CinemaSchedule::AddMovie(const MovieKey &key)
{
    _lock.AssertLockedForWrite();
    Movie *movie = 0;
    if (_movies.contains(key)){
        movie = _movies[key];
        if (movie != 0) {
            return movie;
        }
    }
    movie = new Movie();
    movie->SetName(key.GetName());
    _movies[key] = movie;
    return movie;
}

const CinemaSchedule::ScheduleDates CinemaSchedule::GetScheduleDates() const
{
    _lock.AssertLockedForRead();
    ScheduleDates result;
    Q_FOREACH(const ScheduleEntry &entry, _schedules) {
        result |= entry.GetDate();
    }
    return result;
}

const CinemaSchedule::ScheduleDates CinemaSchedule::GetScheduleDates(const Cinema *cinema) const
{
    _lock.AssertLockedForRead();
    ScheduleDates result;
    Q_FOREACH(const ScheduleEntry &entry, _schedules) {
        if (entry.GetCinema() == cinema) {
            result |= entry.GetDate();
        }
    }
    return result;
}

const CinemaSchedule::ScheduleDates CinemaSchedule::GetScheduleDates(const Movie *movie) const
{
    _lock.AssertLockedForRead();
    ScheduleDates result;
    Q_FOREACH(const ScheduleEntry &entry, _schedules) {
        if (entry.GetMovie() == movie) {
            result |= entry.GetDate();
        }
    }
    return result;
}

ScheduleEntry CinemaSchedule::FindScheduleEntry(const ScheduleEntryKey &schedule_entry_key) const
{
    _lock.AssertLockedForRead();
    const Cinema *cinema = FindCinema(schedule_entry_key.GetCinemaKey());
    const Movie *movie = FindMovie(schedule_entry_key.GetMovieKey());
    ScheduleEntry schedule_entry;
    if (cinema != 0 && movie != 0) {
        Q_FOREACH(const ScheduleEntry &entry, _schedules) {
            if (entry.GetCinema() == cinema && entry.GetMovie() == movie
                && entry.GetStartTime() == schedule_entry_key.GetStartTime()
                && entry.GetDate() == schedule_entry_key.GetDate()) {
                schedule_entry = entry;
                break;
            }
        }
    }
    return schedule_entry;
}

const CinemaSchedule::Schedules CinemaSchedule::GetSchedules(const Cinema *cinema) const
{
    _lock.AssertLockedForRead();
    Schedules result;
    Q_FOREACH(const ScheduleEntry &entry, _schedules) {
        if (entry.GetCinema() == cinema) {
            result |= entry;
        }
    }
    return result;
}

const CinemaSchedule::Schedules CinemaSchedule::GetSchedules(const Movie *movie) const
{
    _lock.AssertLockedForRead();
    Schedules result;
    Q_FOREACH(const ScheduleEntry &entry, _schedules) {
        if (entry.GetMovie() == movie) {
            result |= entry;
        }
    }
    return result;
}

const CinemaSchedule::ScheduleKeys CinemaSchedule::GetScheduleKeys(const CinemaKey &cinema_key) const
{
    _lock.AssertLockedForRead();
    ScheduleKeys result;
    const Cinema *cinema = FindCinema(cinema_key);
    if (cinema != 0) {
        Q_FOREACH(const ScheduleEntry &entry, _schedules) {
            if (entry.GetCinema() == cinema) {
                result |= entry.GetKey();
            }
        }
    }
    return result;
}

const CinemaSchedule::ScheduleKeys CinemaSchedule::GetScheduleKeys(const MovieKey &movie_key) const
{
    _lock.AssertLockedForRead();
    ScheduleKeys result;
    const Movie *movie = FindMovie(movie_key);
    if (movie != 0) {
        Q_FOREACH(const ScheduleEntry &entry, _schedules) {
            if (entry.GetMovie() == movie) {
                result |= entry.GetKey();
            }
        }
    }
    return result;
}

void CinemaSchedule::AddSchedule(const Cinema *cinema, const Movie *movie, const QTime &start_time, const QDate &date)
{
    _lock.AssertLockedForWrite();
    _schedules |= ScheduleEntry(cinema, movie, start_time, date);
}
