/*
 *  Microfeed - Backend for accessing feed-based services (unstable providers)
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
 
#define _XOPEN_SOURCE
#define _BSD_SOURCE
#define _GNU_SOURCE

#include <microfeed-provider/microfeedfeed.h>
#include <microfeed-common/microfeedmisc.h>
#include <microfeed-common/microfeedprotocol.h>
#include <microfeed-provider/microfeedhttp.h>
#include <microfeed-provider/microfeedjson.h>

#include "twitter.h"

#include <stdio.h>
#include <string.h>

static void destroy_feed(MicrofeedFeed* feed, void* user_data);
static MicrofeedError* update_feed(MicrofeedFeed* feed, int user_initiated, void* user_data);
static MicrofeedError* modify_item(MicrofeedFeed *feed, MicrofeedItem *existing_item, MicrofeedItem *new_item, void *user_data);
static MicrofeedError* mark_item(MicrofeedFeed* feed, const char* uid, int mark_status, void* user_data);
static void complete_item_properties(MicrofeedFeed* feed, MicrofeedItem* item, void* user_data);

MicrofeedFeedCallbacks overview_feed_callbacks = {
	destroy_feed,
	update_feed,
	modify_item,
	mark_item,
	complete_item_properties
};

MicrofeedFeedCallbacks feed_callbacks = {
	destroy_feed,
	update_feed,
	NULL, /* modify_feed_item */
	mark_item,
	complete_item_properties
};

static void destroy_feed(MicrofeedFeed* feed, void* user_data) {
	char* url;

	url = (char*)user_data;
	free(url);
}

static MicrofeedError* update_feed(MicrofeedFeed* feed, int user_initiated, void* user_data) {
	MicrofeedError* error = NULL;
	char* site;
	const char* path;
	char* url;
	MicrofeedHttp* http;
	int rate_limit;
	time_t delta = 0;
	MicrofeedJson* json;
	MicrofeedFeedIterator* iterator;
	unsigned int i;
	MicrofeedJson* status;
	const char* id;
	const char* created_at;
	time_t timestamp;
	MicrofeedItem* item;
	const char* cs;
	char* s;

	site = microfeed_publisher_get_setting_value(microfeed_feed_get_publisher(feed), "service.url", "http://twitter.com");
	path = (const char*)user_data;
	url = microfeed_util_string_concatenate(site, path, NULL);
	free(site);

	http = microfeed_http_new();
	do_authentication(microfeed_feed_get_publisher(feed), http);
	if ((rate_limit = get_rate_limit(microfeed_feed_get_publisher(feed), http)) != -1 && rate_limit < 1) {
		error = microfeed_error_new(MICROFEED_ERROR_FROM_SERVICE, "Rate limit exceeded."); /* TODO: Change to MICROFEED_ERROR_SERVICE_RATE_LIMIT_EXCEEDED */
	} else if ((json = microfeed_http_get_json(http, url))) {
		if (microfeed_http_get_server_time(http)) {
			delta = microfeed_http_get_server_time(http) - microfeed_http_get_reply_start_time(http);
		}
		if (microfeed_json_get_type(json, NULL) == MICROFEED_JSON_TYPE_ARRAY) {
			iterator = microfeed_feed_iterate_timeline(feed, 0, 1);
			for (i = 0; i < microfeed_json_get_size(json); i++) {
				if ((status = microfeed_json_get_object_by_index(json, i)) &&
				    (id = microfeed_json_get_as_string(status, "id"))) {
					if (microfeed_feed_iterator_jump_and_remove_previous_items(iterator, id)) {
						microfeed_feed_iterator_next(iterator);
					} else {
						if ((created_at = microfeed_json_get_string(status, "created_at"))) {
							timestamp = microfeed_http_parse_date(http, created_at) - delta;
						} else {
							timestamp = time(NULL) - delta;
						}


			/* TODO: ? */
			delta = 0;

						item = microfeed_item_new_with_status(id, timestamp, MICROFEED_ITEM_STATUS_NEW);
						microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_CONTENT_TEXT, microfeed_json_get_string(status, "text"));
						microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_USER_NICK, microfeed_json_get_string_by_path(status, "user", "screen_name", NULL));
						if ((cs = microfeed_json_get_as_string_by_path(status, "user", "id", NULL))) {
							microfeed_item_set_property(item, ".twitter.id", cs);						
							s = microfeed_util_string_concatenate(MICROFEED_FEED_URI_USER_PREFIX, cs, NULL);
							microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_USER_FEED, s);
							free(s);
						}
						microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_USER_IMAGE, microfeed_json_get_string_by_path(status, "user", "profile_image_url", NULL));
						if (!microfeed_json_is_null(status, "in_reply_to_status_id")) {
							microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_REFERRED_ITEM, microfeed_json_get_as_string(status, "in_reply_to_status_id"));
						}

						microfeed_feed_replace_item(feed, item);
						microfeed_item_free(item);
					}
				} else {
					error = microfeed_error_new(MICROFEED_ERROR_COMMUNICATION_FAILED, "Invalid reply (expected JSON object with id in array).");
				}
			}
			microfeed_feed_iterator_free(iterator);
		} else if (!(error = check_if_error(json))) {
			error = microfeed_error_new(MICROFEED_ERROR_COMMUNICATION_FAILED, "Invalid reply (expected JSON array).");
		}
		microfeed_json_free(json);
	} else {
		error = microfeed_error_new(MICROFEED_ERROR_CONNECTION_FAILED, "Could not get the feed.");
	}

	microfeed_http_free(http);
	free(url);

	return error;
}

static MicrofeedError* modify_item(MicrofeedFeed *feed, MicrofeedItem *existing_item, MicrofeedItem *new_item, void *user_data) {
	const char* text;
	char* encoded_text;
	const char* referred;
	char* post_data;
	const char* source;
	char* site;
	char* url;
	MicrofeedHttp* http;
	MicrofeedJson* json;
	
	if (existing_item) {

		return microfeed_error_new(MICROFEED_ERROR_CANNOT_MODIFY_ITEM, "Cannot modify an existing item.");
	}
	if (!(text = microfeed_item_get_property(new_item, MICROFEED_ITEM_PROPERTY_NAME_CONTENT_TEXT))) {

		return microfeed_error_new(MICROFEED_ERROR_MISSING_PROPERTY, "Expected a content.text property.");
	}
	
	encoded_text = microfeed_util_string_percent_encoding_escape(text);
	if ((referred = microfeed_item_get_property(new_item, MICROFEED_ITEM_PROPERTY_NAME_REFERRED_ITEM))) {
		post_data = microfeed_util_string_concatenate("status=", encoded_text, "&in_reply_to_status_id=", referred, NULL);
	} else {
		post_data = microfeed_util_string_concatenate("status=", encoded_text, NULL);		
	}
	free(encoded_text);
	if ((source = microfeed_item_get_property(new_item, ".twitter.source"))) {
		post_data = microfeed_util_string_append(post_data, "&source=", source, NULL);
	}

	site = microfeed_publisher_get_setting_value(microfeed_feed_get_publisher(feed), "service.url", "http://twitter.com");
	url = microfeed_util_string_concatenate(site, "/statuses/update.json", NULL);
	free(site);
	
	http = microfeed_http_new();
	do_authentication(microfeed_feed_get_publisher(feed), http);
	if (!(json = microfeed_http_post_json(http, url, post_data))) {

		return microfeed_error_new(MICROFEED_ERROR_COMMUNICATION_FAILED, "Invalid reply when updating status.");
	}
	
	microfeed_http_free(http);
	microfeed_json_free(json);
	free(url);
	free(post_data);
	
	return NULL;
}

static MicrofeedError* mark_item(MicrofeedFeed* feed, const char* uid, int mark_status, void* user_data) {
	char* site;
	char* url;
	MicrofeedHttp* http;
	MicrofeedJson* json;

	site = microfeed_publisher_get_setting_value(microfeed_feed_get_publisher(feed), "service.url", "http://twitter.com");
	if (mark_status) {
		url = microfeed_util_string_concatenate(site, "/favorites/create/", uid, ".json", NULL);
	} else {
		url = microfeed_util_string_concatenate(site, "/favorites/destroy/", uid, ".json", NULL);
	}
	free(site);

	http = microfeed_http_new();
	do_authentication(microfeed_feed_get_publisher(feed), http);
	if (!(json = microfeed_http_post_json(http, url, ""))) {

		return microfeed_error_new(MICROFEED_ERROR_COMMUNICATION_FAILED, "Invalid reply when changing favorite status.");
	}
	
	microfeed_http_free(http);
	microfeed_json_free(json);
	free(url);
	
	return NULL;
}

static void complete_item_properties(MicrofeedFeed* feed, MicrofeedItem* item, void* user_data) {
	const char* cs;
	char* site;
	char* url;
	char* uri;
	MicrofeedFeed* comments_feed;
	
	if (strcmp(microfeed_item_get_uid(item), MICROFEED_ITEM_UID_FEED_METADATA)) {
		if ((cs = microfeed_item_get_property(item, MICROFEED_ITEM_PROPERTY_NAME_USER_NICK))) {
			site = microfeed_publisher_get_setting_value(microfeed_feed_get_publisher(feed), "web.url", "http://twitter.com");
			url = microfeed_util_string_concatenate(site, "/", cs, NULL);
			microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_USER_URL, url);
			free(site);
			free(url);
		}
		if ((cs = microfeed_item_get_property(item, MICROFEED_ITEM_PROPERTY_NAME_REFERRED_ITEM))) {
			uri = microfeed_util_string_concatenate(MICROFEED_FEED_URI_COMMENTS_PREFIX, cs, NULL);
			microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_REFERRED_FEED, uri);
			free(uri);
		}
		uri = microfeed_util_string_concatenate(MICROFEED_FEED_URI_COMMENTS_PREFIX, microfeed_item_get_uid(item), NULL);
		if ((comments_feed = microfeed_publisher_get_or_instantiate_feed(microfeed_feed_get_publisher(feed), uri, 0))) {
			microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_COMMENTS_FEED, uri);
			microfeed_item_set_property_from_integer(item, MICROFEED_ITEM_PROPERTY_NAME_COMMENTS_COUNT, microfeed_feed_get_item_count(comments_feed));
			microfeed_object_unref(comments_feed, MicrofeedFeed);
		}
		free(uri);
	}
}
