#! /usr/bin/env python
# -*- coding: UTF-8 -*-

import os, sys, time
import gobject, dbus, telepathy
import mafw_monitor
import ConfigParser

DEBUG       = False
CONFIG_PATH = '/home/user/.media-im-status-updater.ini'

TELEPATHY_ACCOUNT_PREFIX        = '/org/freedesktop/Telepathy/Account/'
TELEPATHY_ACCOUNT_MANAGER_PATH  = '/org/freedesktop/Telepathy/AccountManager'

def debug(message):
  if DEBUG and sys.stderr.isatty():
    sys.stderr.write("[%f] %s\n" % (time.time(), str(message)))

class Config:
  __path        = None

  __template    = None
  __accounts    = None
  __is_enabled  = None

  def __init__(self, path):
    self.__path = path

    self.__template   = u'Listening to "%t" by "%a" via Nokia N900'
    self.__accounts   = list()
    self.__is_enabled = False

  def is_enabled(self):
    return self.__is_enabled

  def template(self):
    return self.__template

  def accounts(self):
    return self.__accounts

  def set_enabled(self, is_enabled, save = True):
    self.__is_enabled = is_enabled

    if save:
      self.save()

  def set_template(self, template, save = True):
    self.__template = template

    if save:
      self.save()

  def set_accounts(self, accounts, save = True):
    self.__accounts = accounts

    if save:
      self.save()

  def load(self):
    if os.path.isfile(self.__path):
      parser = ConfigParser.RawConfigParser()
      parser.read(self.__path)

      try:
        self.__template   = parser.get('Config', 'Template')
        self.__accounts   = parser.get('Config', 'Accounts').split(',')
        self.__is_enabled = parser.get('Config', 'Enabled').lower() == 'true'
      except ConfigParser.NoOptionError:
        pass

  def save(self):
    parser = ConfigParser.RawConfigParser()
    parser.add_section('Config')
    parser.set('Config', 'Template',  self.__template)
    parser.set('Config', 'Accounts',  str.join(',', self.__accounts))
    parser.set('Config', 'Enabled',   str(self.__is_enabled))

    f = open(self.__path, 'w')
    parser.write(f)
    f.close()

class Service(mafw_monitor.MafwMonitor):
  __config        = None
  __session_bus   = None

  #   DBus object and interface to the Telepathy AccountManager for updating
  # presence status and accessing account information.
  __am            = None
  __am_iface      = None

  #   Current state of media daemon.
  __am_state      = None

  __metadata      = None
  __updated_at    = None

  #   This is where we backup what the user specifies status to be, so we can
  # restore it on pause/stop.
  __user_presences = None

  def __init__(self, bus, config):
    mafw_monitor.MafwMonitor.__init__(self, bus)

    self.__config       = config
    self.__session_bus  = bus

    self.__am           = self.__session_bus.get_object(telepathy.ACCOUNT_MANAGER,
                                                     TELEPATHY_ACCOUNT_MANAGER_PATH)
    self.__am_iface     = dbus.Interface(self.__am, dbus.PROPERTIES_IFACE)

    self.__timer        = gobject.timeout_add(2000, self.__timer_process)

  def on_metadata(self, key, value):
    if self.__metadata == None:
      return

    self.__metadata[key]  = value
    self.__updated_at     = time.time()

  def on_media_started(self):
    self.__am_state   = mafw_monitor.MAFW_STATE_STARTED
    #   This is to force the timer process to redisplay track information after
    # a pause.
    self.__updated_at = time.time()

  def on_media_paused(self):
    self.__am_state   = mafw_monitor.MAFW_STATE_PAUSED
    self.__reset_im_status()
    self.__user_presences   = None

  def on_media_stopped(self):
    self.__am_state   = mafw_monitor.MAFW_STATE_STOPPED
    self.__reset_im_status()
    self.__metadata         = None
    self.__user_presences   = None

  def on_media_transition(self):
    self.__am_state   = mafw_monitor.MAFW_STATE_TRANSITION
    self.__metadata = dict()

  def __timer_process(self):
    self.__config.load()

    if not self.__config.is_enabled():
      self.__reset_im_status()
      self.__user_presences = None
      return True

    if self.__am_state and self.__am_state == mafw_monitor.MAFW_STATE_PAUSED:
      return True

    #   This is how we debounce skipping many tracks, the track has to be
    # playing for at least 5 seconds before we do the update.
    if self.__metadata and self.__updated_at and time.time() - self.__updated_at > 5:
      self.__on_metadata_complete()
      self.__updated_at = None

    return True

  def __on_metadata_complete(self):
    message = self.__config.template()

    for key in ['title', 'artist']:
      try:
        value = self.__metadata[key]
      except KeyError:
        value = u'<unknown>'

      message = message.replace('%' + key[0], value)

    debug('INFO: ' + message)

    if self.__config.is_enabled():
      self.__update_im_status(message)

  def __update_account_presence(self, account, presence):
    ac        = self.__session_bus.get_object(telepathy.ACCOUNT_MANAGER, account)
    ac_iface  = dbus.Interface(ac, dbus.PROPERTIES_IFACE)

    debug('INFO: Updating presence information on account: %s - %s' %
          (ac_iface.Get(telepathy.ACCOUNT, 'NormalizedName'), presence))

    ac_iface.Set(telepathy.ACCOUNT, 'RequestedPresence',
                 dbus.Struct(presence, signature = 'uss'))

  def __update_im_status(self, message):
    if self.__user_presences == None:
      self.__user_presences = dict()

    for account in self.__am_iface.Get(telepathy.ACCOUNT_MANAGER, 'ValidAccounts'):
      ac_name = account.replace(TELEPATHY_ACCOUNT_PREFIX, '')

      # Skip this account if we're not configured to use it.
      if not self.__config.accounts().count(ac_name) > 0:
        continue

      #   Get current presence information so we can maintain the same
      # availability.
      ac          = self.__session_bus.get_object(telepathy.ACCOUNT_MANAGER, account)
      ac_iface    = dbus.Interface(ac, dbus.PROPERTIES_IFACE)
      up, ua, um  = ac_iface.Get(telepathy.ACCOUNT, 'RequestedPresence')

      #   If not already done for this application, backup the accounts' current 
      # presence information so we can revert later using '__reset_im_status'.
      if not self.__user_presences.has_key(ac_name):
        debug('INFO: Storing user defined message: %s' % um)
        self.__user_presences[ac_name] = (up, ua, um)

      self.__update_account_presence(account, (up, ua, message))

  def __reset_im_status(self):
    if not self.__user_presences:
      return

    for account in self.__am_iface.Get(telepathy.ACCOUNT_MANAGER, 'ValidAccounts'):
      ac_name   = account.replace(TELEPATHY_ACCOUNT_PREFIX, '')

      # Skip this account if we're not configured to use it.
      if not self.__config.accounts().count(ac_name) > 0:
        continue

      if not self.__user_presences.has_key(ac_name):
        continue

      self.__update_account_presence(account, self.__user_presences[ac_name])

if __name__ == '__main__':
  import gobject, dbus.glib

  DEBUG = True

  bus     = dbus.SessionBus()
  config  = Config(CONFIG_PATH)

  service = Service(bus, config)

  main = gobject.MainLoop()
  main.run()

