/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Geolocation.
 *
 * The Initial Developer of the Original Code is Mozilla Corporation
 * Portions created by the Initial Developer are Copyright (C) 2009
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *  Oleg Romashin <romaxa@gmail.com>  (Original Author)
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include <stdio.h>
#include "MaemoLocationPrompt.h"
#include "nsIClassInfo.h"
#include "nsDOMClassInfoID.h"
#include "nsIDOMClassInfo.h"
#include "nsIPrefService.h"
#include "nsIPrefBranch.h"
#include "nsIServiceManager.h"
#include "nsServiceManagerUtils.h"
#include "nsComponentManagerUtils.h"
#include "nsIGenericFactory.h"
#include "nsICategoryManager.h"
#include "nsIPromptService.h"
#include "nsEmbedCID.h"
#include "nsIDOMWindow.h"
#include "nsIPermissionManager.h"
#include "nsIDialogParamBlock.h"
#include "nsIStringBundle.h"
#include "nsIPrompt.h"

static const char kPromptURL[] = "chrome://global/content/commonDialog.xul";
static const char kSelectPromptURL[] = "chrome://global/content/selectDialog.xul";
static const char kQuestionIconClass[] = "question-icon";
static const char kAlertIconClass[] = "alert-icon";
static const char kWarningIconClass[] = "message-icon";
// We include question-icon for backwards compatibility
static const char kAuthenticationIconClass[] = "authentication-icon question-icon";

#define kCommonDialogsProperties "chrome://global/locale/commonDialogs.properties"
#define kGeoProperties "chrome://maemo-geolocation/locale/geolocation.properties"

/****************************************************************
 ************************* ParamBlock ***************************
 ****************************************************************/

class ParamBlock {

public:
  ParamBlock() {
    mBlock = 0;
  }
  ~ParamBlock() {
    NS_IF_RELEASE(mBlock);
  }
  nsresult Init() {
    return CallCreateInstance(NS_DIALOGPARAMBLOCK_CONTRACTID, &mBlock);
  }
  nsIDialogParamBlock * operator->() const { return mBlock; }
  operator nsIDialogParamBlock * const () { return mBlock; }

private:
  nsIDialogParamBlock *mBlock;
};


NS_IMPL_ISUPPORTS2(MaemoLocationPrompt, nsIGeolocationPrompt, nsPIPromptService)

MaemoLocationPrompt::MaemoLocationPrompt()
{
  mWatcher = do_GetService(NS_WINDOWWATCHER_CONTRACTID);
}

MaemoLocationPrompt::~MaemoLocationPrompt()
{
}

nsresult
MaemoLocationPrompt::GetLocaleString(const char *aKey, PRUnichar **aResult)
{
  nsresult rv;

  nsCOMPtr<nsIStringBundleService> stringService = do_GetService(NS_STRINGBUNDLE_CONTRACTID);
  nsCOMPtr<nsIStringBundle> stringBundle;

  rv = stringService->CreateBundle(kCommonDialogsProperties, getter_AddRefs(stringBundle));
  if (NS_FAILED(rv)) return NS_ERROR_FAILURE;

  rv = stringBundle->GetStringFromName(NS_ConvertASCIItoUTF16(aKey).get(), aResult);

  return rv;
}

nsresult
MaemoLocationPrompt::DoDialog(nsIDOMWindow *aParent,
                              nsIDialogParamBlock *aParamBlock, const char *aChromeURL)
{
  NS_ENSURE_ARG(aParamBlock);
  NS_ENSURE_ARG(aChromeURL);
  if (!mWatcher)
    return NS_ERROR_FAILURE;

  nsresult rv = NS_OK;
  nsCOMPtr<nsIDOMWindow> activeParent; // retain ownership for method lifetime
  if (!aParent) {
    mWatcher->GetActiveWindow(getter_AddRefs(activeParent));
    aParent = activeParent;
  }

  nsCOMPtr<nsISupports> arguments(do_QueryInterface(aParamBlock));
  nsCOMPtr<nsIDOMWindow> dialog;
  rv = mWatcher->OpenWindow(aParent, aChromeURL, "_blank",
                            "centerscreen,chrome,modal,titlebar", arguments,
                            getter_AddRefs(dialog));

  return rv;
}

NS_IMETHODIMP
MaemoLocationPrompt::ConfirmEx(nsIDOMWindow *parent,
                               const PRUnichar *dialogTitle, const PRUnichar *text,
                               PRUint32 buttonFlags, const PRUnichar *button0Title,
                               const PRUnichar *button1Title, const PRUnichar *button2Title,
                               const PRUnichar *checkMsg, PRBool *checkValue,
                               PRInt32 *buttonPressed)
{
  nsresult rv;
  nsString stringOwner;

  if (!dialogTitle) {
    rv = GetLocaleString("Confirm", getter_Copies(stringOwner));
    if (NS_FAILED(rv)) return NS_ERROR_FAILURE;
    dialogTitle = stringOwner.get();
  }

  ParamBlock block;
  rv = block.Init();
  if (NS_FAILED(rv))
    return rv;

  block->SetString(eDialogTitle, dialogTitle);
  block->SetString(eMsg, text);

  int buttonIDs[] = { eButton0Text, eButton1Text, eButton2Text };
  const PRUnichar* buttonStrings[] = { button0Title, button1Title, button2Title };

#define BUTTON_DEFAULT_MASK 0x03000000

  block->SetInt(eDefaultButton, (buttonFlags & BUTTON_DEFAULT_MASK) >> 24);
  block->SetInt(eDelayButtonEnable, buttonFlags & nsIPromptService::BUTTON_DELAY_ENABLE);

  PRInt32 numberButtons = 0;
  for (int i = 0; i < 3; i++) { 

    nsString buttonTextStr;
    const PRUnichar* buttonText = 0;
    switch (buttonFlags & 0xff) {
      case nsIPromptService::BUTTON_TITLE_OK:
        GetLocaleString("OK", getter_Copies(buttonTextStr));
        break;
      case nsIPromptService::BUTTON_TITLE_CANCEL:
        GetLocaleString("Cancel", getter_Copies(buttonTextStr));
        break;
      case nsIPromptService::BUTTON_TITLE_YES:
        GetLocaleString("Yes", getter_Copies(buttonTextStr));
        break;
      case nsIPromptService::BUTTON_TITLE_NO:
        GetLocaleString("No", getter_Copies(buttonTextStr));
        break;
      case nsIPromptService::BUTTON_TITLE_SAVE:
        GetLocaleString("Save", getter_Copies(buttonTextStr));
        break;
      case nsIPromptService::BUTTON_TITLE_DONT_SAVE:
        GetLocaleString("DontSave", getter_Copies(buttonTextStr));
        break;
      case nsIPromptService::BUTTON_TITLE_REVERT:
        GetLocaleString("Revert", getter_Copies(buttonTextStr));
        break;
      case nsIPromptService::BUTTON_TITLE_IS_STRING:
        buttonText = buttonStrings[i];
        break;
    }
    if (!buttonText && !buttonTextStr.IsEmpty())
      buttonText = buttonTextStr.get();

    if (buttonText) {
      block->SetString(buttonIDs[i], buttonText);
      ++numberButtons;
    }
    buttonFlags >>= 8;
  }
  block->SetInt(eNumberButtons, numberButtons);

  block->SetString(eIconClass, NS_ConvertASCIItoUTF16(kQuestionIconClass).get());

  if (checkMsg && checkValue) {
    block->SetString(eCheckboxMsg, checkMsg);
    // since we're setting a PRInt32, we have to sanitize the PRBool first.
    // (myBool != PR_FALSE) is guaranteed to return either 1 or 0.
    block->SetInt(eCheckboxState, *checkValue != PR_FALSE);
  }

  /* perform the dialog */

  rv = DoDialog(parent, block, kPromptURL);
  if (NS_FAILED(rv))
    return rv;

  /* get back output parameters */

  if (buttonPressed)
    block->GetInt(eButtonPressed, buttonPressed);

  if (checkMsg && checkValue) {
    // GetInt returns a PRInt32; we need to sanitize it into PRBool
    PRInt32 tempValue;
    block->GetInt(eCheckboxState, &tempValue);
    *checkValue = (tempValue == 1);
  }

  return rv;
}


NS_IMETHODIMP
MaemoLocationPrompt::Prompt(nsIGeolocationRequest *request)
{
  nsresult rv;
  nsCOMPtr<nsIURI> uri;
  request->GetRequestingURI(getter_AddRefs(uri));
  NS_ENSURE_TRUE(uri, NS_ERROR_FAILURE);

  nsCOMPtr<nsIPermissionManager> pm = do_GetService("@mozilla.org/permissionmanager;1", &rv);
  NS_ENSURE_SUCCESS(rv, rv);

  PRUint32 type = nsIPermissionManager::UNKNOWN_ACTION;
  rv = pm->TestExactPermission(uri, "geo", &type);
  NS_ENSURE_SUCCESS(rv, rv);

  if (type == nsIPermissionManager::ALLOW_ACTION) {
    request->Allow();
    return NS_OK;
  }
  if (type == nsIPermissionManager::DENY_ACTION) {
    request->Cancel();
    return NS_OK;
  }

  nsCString spec;
  uri->GetSpec(spec);
  nsCOMPtr<nsIDOMWindow> win;
  request->GetRequestingWindow(getter_AddRefs(win));

  nsString title, text, shareBt, dontShareBt, checkMsg;
  nsCOMPtr<nsIStringBundleService> stringService = do_GetService(NS_STRINGBUNDLE_CONTRACTID);
  nsCOMPtr<nsIStringBundle> stringBundle;

  rv = stringService->CreateBundle(kGeoProperties, getter_AddRefs(stringBundle));
  if (NS_FAILED(rv)) return NS_ERROR_FAILURE;

  rv = stringBundle->GetStringFromName(NS_LITERAL_STRING("geolocation.titleRequest").get(), getter_Copies(title));
  rv = stringBundle->GetStringFromName(NS_LITERAL_STRING("geolocation.share").get(), getter_Copies(shareBt));
  rv = stringBundle->GetStringFromName(NS_LITERAL_STRING("geolocation.dontShare").get(), getter_Copies(dontShareBt));
  rv = stringBundle->GetStringFromName(NS_LITERAL_STRING("geolocation.remember").get(), getter_Copies(checkMsg));
  NS_ConvertUTF8toUTF16 location(spec);
  const PRUnichar *formatStrings[] = { location.get() };
  rv = stringBundle->FormatStringFromName(NS_LITERAL_STRING("geolocation.siteWantsToKnow").get(),
                                          formatStrings, 1, getter_Copies(text));

  PRBool checkState = PR_FALSE;
  PRBool retval = PR_FALSE;
#if 0
  // Browser UI does not display ConfirmCheck properly :(
  nsCOMPtr<nsIPromptService> ps(do_GetService(NS_PROMPTSERVICE_CONTRACTID));
  NS_ENSURE_TRUE(ps, NS_ERROR_FAILURE);
  rv = ps->ConfirmCheck(win, title.get(), text.get(), checkMsg.get(), &checkState, &retval);
#else
  PRInt32 buttonPressed = 1;
  rv = ConfirmEx(win, title.get(), text.get(),
                 (nsIPrompt::BUTTON_DELAY_ENABLE) +
                 (nsIPrompt::BUTTON_POS_1_DEFAULT) +
                 (nsIPrompt::BUTTON_TITLE_IS_STRING * nsIPrompt::BUTTON_POS_0) +
                 (nsIPrompt::BUTTON_TITLE_IS_STRING * nsIPrompt::BUTTON_POS_1),
                 shareBt.get(), dontShareBt.get(), nsnull,
                 checkMsg.get(), &checkState, &buttonPressed);
  retval = (buttonPressed == 0);
#endif
  NS_ENSURE_SUCCESS(rv, rv);
  if (!retval) {
    request->Cancel();
    return NS_OK;
  }

  if (pm && checkState)
    pm->Add(uri, "geo", nsIPermissionManager::ALLOW_ACTION);

  request->Allow();
  return NS_OK;
}

