
#include <iostream>
#include <string>

#include <CMulticalendar.h>
#include <CCalendar.h>
#include <CTodo.h>
#include <CAlarm.h>

#include <QStringList>
#include <QDateTime>

#include <CalendarErrors.h>

#include <main.cpp>
#include <functions.h>

using namespace std;


void usage_import(){
    cout << "Usage:" << endl;
    cout << "\t--operation import --calendar CAL_NAME --file FILE_NAME [--clear] [--verify]"<< endl;
    cout << endl;
    cout << "\t--clear\t\tClears the calendar before performing import" << endl;
    cout << "\t--verify\tVerifies the input calendar before performing import" << endl;
}

void usage_export(){
    cout << "Usage:" << endl;
    cout << "\t--operation export --calendar CAL_NAME [--file FILE_NAME] [--quiet]" << endl;
    cout << "\tDefaults to CAL_NAME.ics if no FILE_NAME given" << endl;
}

void usage_tasks_clear_done(){
    cout << "Usage:" << endl;
    cout << "\t--operation " << ARG_OP_VAL_TASKS_CLEAR_DONE.toLocal8Bit().constData()
            << " --calendar CAL_NAME" << endl;
           cout << "This will delete any of the tasks that have been completed." << endl;
}

void usage_tasks_bump_today() {
    cout << "Usage:" << endl;
    cout << "\t--operation " << ARG_OP_VAL_TASKS_BUMP_DAY.toLocal8Bit().constData()
            << " --calendar CAL_NAME [--days DAYS]" << endl;
           cout << "This will move any tasks that were due within DAYS ago to today." << endl;
           cout << "DAYS defaults to -1 (yesterday) if not provided." << endl;
}

void print_cals(CMulticalendar &mcal_instance){
    /*
     * Prints the list of calendars to std out
     */
    vector <CCalendar *> cals = mcal_instance.getListCalFromMc();

    cout << "Available calendars(" << cals.size() << "):" << endl;

    for (unsigned int i=0; i<cals.size(); i++){
        CCalendar *calendar = cals.at(i);
        unsigned int cal_id = calendar->getCalendarId();

        cout << calendar->getCalendarName() << " (id=" << cal_id << ")" << endl;
    }
    mcal_instance.releaseListCalendars(cals);

}

unsigned int parse_args(QStringList &args){
    //returns an int with the operation id to perform

    //get first instance of "calendar"
    //QString opstring = ARG_OPERATION_LONG;

    int a = args.indexOf(ARG_OPERATION_LONG);

    //no operation specified
    if(a == -1){
        return OPERATION_UNKNOWN;
    }
    cout << "index of operation is:" << a << endl;
    a++; //get next argument

    cout << "args.size is:" << args.size() << endl;
    if(args.size() < (a + 1)){
        return OPERATION_UNKNOWN;
    }

    cout << "operation is: " << qPrintable(args.at(a)) << endl;
//    cout << "operation is: " << args.at(a).toAscii().constData() << endl;

    return arg_to_code(args.at(a));
}

unsigned int arg_to_code(QString arg){

    if(arg == ARG_OP_VAL_EXPORT){
        return OPERATION_EXPORT;
    }

    if(arg == ARG_OP_VAL_IMPORT){
        return OPERATION_IMPORT;
    }

    if(arg == ARG_OP_VAL_LIST){
        return OPERATION_LIST;
    }

    if(arg == ARG_OP_VAL_TASKS_CLEAR_DONE){
        return OPERATION_TASKS_CLEAR_DONE;
    }

    if(arg == ARG_OP_VAL_TASKS_BUMP_DAY){
        return OPERATION_TASKS_BUMP_DAY;
    }

    return OPERATION_UNKNOWN;

}

bool parse_calendar(QStringList &args, QString &calendar){

    int a = args.indexOf(ARG_CAL_NAME);

    if(a == -1){
        return false;
    }

    a++;

    if(args.size() < (a + 1)){
        return false;
    }

    calendar = args.at(a);
    return true;
}



bool parse_cmd_arg(const QStringList args, const QString arg, int& pos){
    /*
     * Gets the index of the given command line argument provided
     *
     * Returns:
     *    - true if the argument exists
     *    - false otherwise
     */

    pos = args.indexOf(arg);

    if(pos == -1){
        return false;
    } else {
        return true;
    }
}

bool parse_cmd_arg(const QStringList args, const QString arg){

    int x = -2;

    if(parse_cmd_arg(args, arg, x)){
        return true;
    } else {
        return false;
    }

}



bool parse_cmd_arg(const QStringList args, const QString arg, QString& value){
    /*
     * Gets the value for an argument given on the command line
     *
     * Returns:
     *     - true if the argument exists and a following value was supplied
     *     - false otherwise
     */

    int a = -1;

    if(parse_cmd_arg(args,arg,a)){

        if(args.size() > ++a){
            value = args.at(a);
            return true;
        }
    }

    return false;
}

bool parse_cmd_arg_calendar(const QStringList args, QString & calendar){
    if(parse_cmd_arg(args, ARG_CAL_NAME, calendar)){
        return true;
   } else {
       cout << "Failed to parse the calendar name" << endl;
       return false;
   }
}

bool parse_cmd_arg_filename(const QStringList args, QString & filename){
    if(parse_cmd_arg(args, ARG_FILE_NAME, filename)){
        return true;
   } else {
       cout << "Failed to parse the file name" << endl;
       return false;
   }
}

bool parse_cmd_arg_days(const QStringList args, int & days){
    QString days_arg;
    if(parse_cmd_arg(args, ARG_DAYS, days_arg)){
        days = atoi(days_arg.toLocal8Bit().constData());
        if(days == 0 || days == INT_MAX || days == INT_MIN){
            return false;
        } else {
            days = abs(days);
            return true;
        }
   } else {
       cout << "--days argument requires a value" << endl;
       return false;
   }
}

bool get_calendar_info(string calendar, int & iCalendarId){
    /*
     * Verifies the calendar name and gets the API id
     *
     * Returns:
     *    - true if valid calendar
     *    - false if not a valid calendar
     *
     */

    if(!mcalendar->checkCalendarNameExists(calendar)){
        return false;
    }
    int errcode = 0;
    CCalendar *ccalendar = mcalendar->getCalendarByName(calendar, errcode);
    //cout << "get_calendar_info: errcode:" << errcode << endl;
    int cal_id = ccalendar->getCalendarId();
    //cout << "get_calendar_info: calendar id is: " << cal_id << endl;
    iCalendarId = cal_id;
    return true;

}

bool get_calendar_from_cmd(const QStringList args, QString & calendar, int & iCalendarId){
    /*
     * Parses the command line for a calendar. Verifies the name and gets the id from the API
     *
     */
    if(!parse_cmd_arg_calendar(args, calendar)){
        return false;
    }

    return get_calendar_info(calendar.toAscii().constData(),iCalendarId);

}


bool do_operation(const QStringList args, const QString operation){
    /*
     * Does the operation given
     *
     */
    bool result = false;

    switch (arg_to_code(operation)){
    case OPERATION_LIST:
        result = do_operation_list();
        break;

    case OPERATION_IMPORT:
        result = do_operation_import(args);
        break;

    case OPERATION_EXPORT:
        result = do_operation_export(args);
        break;

    case OPERATION_TASKS_CLEAR_DONE:
        result = do_operation_tasks_clear_done(args);
        break;

    case OPERATION_TASKS_BUMP_DAY:
        result = do_operation_tasks_bump_today(args);
        break;

    case OPERATION_UNKNOWN:
        cout << "Either the operation is unknown or not specified!" << endl;
        break;
    default:
        cout << "Could not determine operation" << endl;
    }

    return result;

}

bool do_operation_list(){
    CMulticalendar *mcalendar = CMulticalendar::MCInstance();
    print_cals(*mcalendar);
    return true;
}

bool do_operation_import(const QStringList args){
    QString calendar, filename;
    int calendar_id;

    if(!get_calendar_from_cmd(args, calendar, calendar_id)){
        cout << "That is not a valid calendar name" << endl;
        usage_import();
        return false;
    } else {
        cout << "Will import calendar with id:" << calendar_id << endl;
    }

    if(!parse_cmd_arg_filename(args, filename)){
        cout << "file was not specified:" << filename.toAscii().constData() << endl;
        usage_import();
        return false;
    }

    string eventId;
    unsigned int iCountTask = 0, iCountEvent = 0, iCountJournal = 0;
    int iCountDupeTask = 0, iCountDupeEvent = 0, iCountDupeJournal = 0;
    int errcode = 0;

    //mcalendar->setAutocommitOff();//does import override commit settings?
    //mcalendar->clearCalendar(calendar_id, errcode);
    bool result = false;

    if(parse_cmd_arg(args,ARG_VERIFY)){
        cout << "Verifying input file..." << endl;

        result = mcalendar->getICSFileInfo(
                filename.toStdString(),
                iCountEvent,
                iCountTask,
                iCountJournal,
                errcode);


        if(!result) {
            cout << "Failed to parse the input calendar";
            return false;
        } else {
            cout << "Input file verified." << endl;
            cout << "The following items will be imported:" << endl;
            cout << "Number of tasks-->" << iCountTask << endl;
            cout << "Number of events->" << iCountEvent << endl;
            cout << "Number of notes-->:" << iCountJournal << endl;
        }
    }


    /*
     * Test to see if a clear operation was requested
     *
     * It makes sense to clear the calendar if doing a whole
     * calendar import to try to catch "moved events". While iCal
     * has a concept of 'id', this may not always be implemented
     * in a consistent way by a calendar exporter/provider. This
     * clear option provides a work-around. Also, it makes sense
     * to NOT clear by default if the user likes to import just a single
     * new event (or subset of events) into the calenar.
     */
    if(parse_cmd_arg(args,ARG_CLEAR)){
        /*
         * First, clear the calendar contents. While the import API automatically ignores
         * duplicates, if the external calendar has any appointments that have "moved"
         * then there is no logic to figure that out. Clearing the calendar first
         * is a way to handle potentially "moved" appointments.
         */
        mcalendar->clearCalendar(calendar_id, errcode);

        if(errcode == CALENDAR_OPERATION_SUCCESSFUL){
            cout << "Clearing the calendar was successful." << endl;
        } else {
            cout << "The calendar could not be cleared." << endl;
            cout << "Please contact the developer with this error code: " << errcode << endl;
        }

    }

    cout << "Importing calendar..." << endl;
    result = mcalendar->importIcsFileData(
            filename.toStdString(),
            calendar_id,
            eventId,
            iCountDupeTask,
            iCountDupeEvent,
            iCountDupeJournal,
            errcode);

    if(result){
        switch (errcode){
        case CALENDAR_ENTRY_DUPLICATED:
            cout << "Duplicates were found and updated." << endl;
            cout << "Duplicate tasks-->" << iCountDupeTask << endl;
            cout << "Duplicate events->" << iCountDupeEvent << endl;
            cout << "Duplicate notes-->:" << iCountDupeJournal << endl;

            break;

        case CALENDAR_OPERATION_SUCCESSFUL:
            cout << "Import successful." << endl;

            break;

        default:
            cout << "WARN: api result was true, but the errorcode was not SUCCESS. It was " << errcode << endl;
        }

        return true;
    } else {
        cout << "An error occurred during import" << endl;
        cout << "errcode:" << errcode;

        return false;
    }

}

bool do_operation_export(const QStringList args){

    QString calendar, filename;
    int calendar_id;

    if(!get_calendar_from_cmd(args, calendar, calendar_id)){
        cout << "That is not a valid calendar name" << endl;
        usage_export();
        return false;
    } else {
        //cout << "will export calendar with id:" << calendar_id << endl;
        log("will export calendar with id:", 1, false );
        log(QString::number(calendar_id).toLocal8Bit().constData(),1,true);

    }

    if(!parse_cmd_arg_filename(args, filename)){
        //cout << "file was not specified:" << filename.toAscii().constData() << endl;
        log("file was not specified:",1,false);
        log(filename.toAscii().constData(),1,true);
        filename = calendar.append(".ics");
    }

    int errcode = 0;
    int result = 99;
    //cout << "calendar will be exported to:" << filename.toAscii().constData() << endl;
    log("calendar will be exported to:",1,false);
    log(filename.toAscii().constData(),1,true);
    result = mcalendar->exportCalendarEntries(calendar_id, filename.toAscii().constData(),errcode);

    if(errcode == CALENDAR_OPERATION_SUCCESSFUL){
        //cout << "The calendar was successfully exported." << endl;
        log("The calendar was successfully exported.",1,true);
        return true;
    } else {
        switch (errcode){
        case CALENDAR_FILE_ERROR:
            cout << "Failed to save the file." << endl;
            break;
        }

        return false;
    }

}

bool do_operation_tasks_clear_done(const QStringList args){

    QString calendar;
    int calendar_id;

    if(!get_calendar_from_cmd(args, calendar, calendar_id)){
        cout << "That is not a valid calendar name" << endl;
        usage_tasks_clear_done();
        return false;
    } else {
        cout << "Deleting completed tasks in calendar with id:" << calendar_id << endl;
    }

    int errcode = 0;

    CCalendar *cal = mcalendar->getCalendarById(calendar_id, errcode);
    if (errcode != CALENDAR_OPERATION_SUCCESSFUL){
        cout << "Failed to get the calendar" << endl;
        return false;
    }

    vector<CTodo *> tasks = cal->getTodos(errcode);
    if (errcode != CALENDAR_OPERATION_SUCCESSFUL){
        cout << "Failed to get the tasks" << endl;
        return false;
    }

    cout << "Total number of tasks(" << tasks.size() << "):" << endl;
    int cerror = 0, cokay = 0;
    for (unsigned int i=0; i<tasks.size(); i++){
        CTodo *task = tasks.at(i);

        if(task->getStatus() == 1){
            cal->deleteTodo(task->getId(),errcode);
            if(errcode != CALENDAR_OPERATION_SUCCESSFUL){
                cerror++;
            } else {
                cokay++;
            }
        }

    }

    cout << "Successful deletions: " << cokay << endl;
    cout << "Failed deletions    : " << cerror << endl;

    //release pointers
    for (unsigned int i=0; i<tasks.size(); i++){
        delete tasks.at(i);
        tasks.at(i) = 0;

    }

    if(cerror == 0){
        return true;
    } else {
        return false;
    }

}

bool do_operation_tasks_bump_today(const QStringList args){
    QString calendar;
    int calendar_id;

    if(!get_calendar_from_cmd(args, calendar, calendar_id)){
        cout << "That is not a valid calendar name" << endl;
        usage_tasks_bump_today();
        return false;
    } else {
        cout << "Bumping previous tasks to today in calendar with id:" << calendar_id << endl;
    }

    int errcode = 0;

    CCalendar *cal = mcalendar->getCalendarById(calendar_id, errcode);
    if (errcode != CALENDAR_OPERATION_SUCCESSFUL){
        cout << "Failed to get the calendar" << endl;
        return false;
    }

    vector<CTodo *> tasks = cal->getTodos(errcode);
    if (errcode != CALENDAR_OPERATION_SUCCESSFUL){
        cout << "Failed to get the tasks" << endl;
        return false;
    }
    int daysback;

    if(parse_cmd_arg(args, ARG_DAYS)){
        if( parse_cmd_arg_days(args, daysback)){

            daysback *= -1;

        } else {
            cout << "Failed to parse the days." << endl;
            return false;
        }
    } else {
        //default to yesterday
        daysback = -1;
    }

    QDate qd_daysback = QDateTime::currentDateTime().addDays(daysback).date();

    //qdt_yesterday.addDays(-1);
    cout << "Day to consider starting is:" <<
            qd_daysback.toString().toLocal8Bit().constData() << endl;


    int cerror = 0, cokay = 0;
    for (unsigned int i=0; i<tasks.size(); i++){

        CTodo *task = tasks.at(i);

        //if the task has not been completed yet, check if it was due yesterday
        if(task->getStatus() == 0){

            QDateTime qdt_due;
            qdt_due.setTime_t(task->getDue());

            QDate qd_due = qdt_due.date();

            if(qd_due >= qd_daysback && qd_due < QDateTime::currentDateTime().date() ){
                cout << "Due:" << task->getDue() << "\t" <<
                        qdt_due.toString().toLocal8Bit().constData() << "\t" <<
                        task->getSummary() << endl;
                //qdt_due = qdt_due.addDays(1);
                qdt_due = QDateTime::currentDateTime();

                /*
                 * Need to test the default calendar behavior first
                 *
                if(parse_cmd_arg(args,ARG_ALARMS)){
                    //bump the alarm, too
                    CAlarm *alarm = task->getAlarm();
                    if(alarm){
                        cout << "Trigger:" << alarm->getTrigger() << endl;
                        cout << "Duration:" << alarm->getDuration() << endl;
                        cout << "TimeBefore" << alarm->getTimeBefore() << endl;
                        cout << "" << alarm->getRepeat() << endl;
                    }
                }
                */

                task->setDue(qdt_due.toTime_t());
                mcalendar->modifyTodo(task, calendar_id, errcode);
                //errcode = 500;
                if(errcode != CALENDAR_OPERATION_SUCCESSFUL){
                    cerror++;
                } else {
                    cokay++;
                }
            }

        }

    }

    cout << "Successful updates: " << cokay << endl;
    cout << "Failed updates    : " << cerror << endl;

    //release pointers
    for (unsigned int i=0; i<tasks.size(); i++){
        delete tasks.at(i);
        tasks.at(i) = 0;

    }

    if(cerror == 0){
        return true;
    } else {
        return false;
    }
}
