/*
 * tangle-wrapper-actor.h
 *
 * This file is part of Tangle Toolkit - A graphical actor library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-wrapper-actor.h"
#include <string.h>
#include <gobject/gvaluecollector.h>

/**
 * SECTION:tangle-wrapper-actor
 * @Short_description: 	A #TangleActor that wraps any ClutterActor inside
 * @Title: TangleWrapperActor
 */

static void clutter_scriptable_iface_init(ClutterScriptableIface *iface);

G_DEFINE_TYPE_WITH_CODE(TangleWrapperActor, tangle_wrapper_actor, TANGLE_TYPE_ACTOR,
                        G_IMPLEMENT_INTERFACE(CLUTTER_TYPE_SCRIPTABLE, clutter_scriptable_iface_init););

enum {
	PROP_0,
	PROP_WRAPPED
};

struct _TangleWrapperActorPrivate {
	ClutterActor* wrapped;
};

ClutterActor* tangle_wrapper_actor_new(ClutterActor* wrapped) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_WRAPPER_ACTOR, "wrapped", wrapped, NULL));
}

ClutterActor* tangle_wrapper_actor_get_wrapped(TangleWrapperActor* wrapper_actor) {

	return wrapper_actor->priv->wrapped;
}

void tangle_wrapper_actor_get(TangleWrapperActor* wrapper_actor, const gchar* first_property_name, ...) {
	const gchar* property_name;
	va_list args;
	GParamSpec* param_spec;
	GObject* object;
	GValue value;
	gchar* error;
	
	va_start(args, first_property_name);
	for (property_name = first_property_name; property_name; property_name = va_arg(args, const gchar*)) {
		if ((param_spec = g_object_class_find_property(G_OBJECT_GET_CLASS(G_OBJECT(wrapper_actor)), property_name))) {
			object = G_OBJECT(wrapper_actor);
		} else if ((param_spec = g_object_class_find_property(G_OBJECT_GET_CLASS(G_OBJECT(wrapper_actor->priv->wrapped)), property_name))) {
			object = G_OBJECT(wrapper_actor->priv->wrapped);
		} else {
			g_warning("%s: Neither wrapper nor wrapped actor has property named '%s'.", G_STRLOC, property_name);
			break;
		}

		g_object_get_property(object, property_name, &value);

		error = NULL;
		G_VALUE_LCOPY(&value, args, 0, &error);

		g_value_unset(&value);

		if (error) {
			g_warning("%s: %s", G_STRLOC, error);
			g_free(error);			
			break;
		}
	}
	va_end(args);	
}

void tangle_wrapper_actor_set(TangleWrapperActor* wrapper_actor, const gchar* first_property_name, ...) {
	const gchar* property_name;
	va_list args;
	GParamSpec* param_spec;
	GObject* object;
	GValue value;
	gchar* error;
	
	va_start(args, first_property_name);
	for (property_name = first_property_name; property_name; property_name = va_arg(args, const gchar*)) {
		if ((param_spec = g_object_class_find_property(G_OBJECT_GET_CLASS(G_OBJECT(wrapper_actor)), property_name))) {
			object = G_OBJECT(wrapper_actor);
		} else if ((param_spec = g_object_class_find_property(G_OBJECT_GET_CLASS(G_OBJECT(wrapper_actor->priv->wrapped)), property_name))) {
			object = G_OBJECT(wrapper_actor->priv->wrapped);
		} else {
			g_warning("%s: Neither wrapper nor wrapped actor has property named '%s'.", G_STRLOC, property_name);
			break;
		}

		g_value_init(&value, G_PARAM_SPEC_VALUE_TYPE(param_spec));		
		error = NULL;
		G_VALUE_COLLECT(&value, args, 0, &error);
		if (error) {
			g_warning("%s: %s", G_STRLOC, error);
			g_free(error);
			break;
		}
		
		g_object_set_property(object, property_name, &value);

		g_value_unset(&value);
	}
	va_end(args);
}

static void tangle_wrapper_actor_get_preferred_width(TangleActor* actor, gfloat for_height, gboolean interacting, gfloat* min_width_p, gfloat* natural_width_p, gfloat* max_width_p) {
	TangleWrapperActor* wrapper_actor;
	
	wrapper_actor = TANGLE_WRAPPER_ACTOR(actor);
	if (TANGLE_IS_ACTOR(wrapper_actor->priv->wrapped)) {
		tangle_actor_get_preferred_width(TANGLE_ACTOR(wrapper_actor->priv->wrapped), for_height, interacting, min_width_p, natural_width_p, max_width_p);
	} else {
		clutter_actor_get_preferred_width(wrapper_actor->priv->wrapped, for_height, min_width_p, natural_width_p);
		if (max_width_p) {
			*max_width_p = 0;
		}
	}
}

static void tangle_wrapper_actor_get_preferred_height(TangleActor* actor, gfloat for_width, gboolean interacting, gfloat* min_height_p, gfloat* natural_height_p, gfloat* max_height_p) {
	TangleWrapperActor* wrapper_actor;
	
	wrapper_actor = TANGLE_WRAPPER_ACTOR(actor);
	if (TANGLE_IS_ACTOR(wrapper_actor->priv->wrapped)) {
		tangle_actor_get_preferred_height(TANGLE_ACTOR(wrapper_actor->priv->wrapped), for_width, interacting, min_height_p, natural_height_p, max_height_p);
	} else {
		clutter_actor_get_preferred_height(TANGLE_WRAPPER_ACTOR(actor)->priv->wrapped, for_width, min_height_p, natural_height_p);
		if (max_height_p) {
			*max_height_p = 0;
		}
	}
}

static void tangle_wrapper_actor_handle_ancestor_event(TangleActor* actor, TangleEvent* event) {
	TangleWrapperActor* wrapper_actor;
	
	TANGLE_ACTOR_CLASS(tangle_wrapper_actor_parent_class)->handle_ancestor_event(actor, event);
	
	wrapper_actor = TANGLE_WRAPPER_ACTOR(actor);
	if (TANGLE_IS_ACTOR(wrapper_actor->priv->wrapped)) {
		tangle_actor_handle_ancestor_event(TANGLE_ACTOR(wrapper_actor->priv->wrapped), event);
	}
}
	

static void tangle_wrapper_actor_allocate_wrapped(TangleWrapperActor* wrapper_actor, ClutterActor* wrapped, const ClutterActorBox* box, ClutterAllocationFlags flags) {
	clutter_actor_allocate(wrapped, box, flags);
}

static void tangle_wrapper_actor_allocate(ClutterActor* actor, const ClutterActorBox* box, ClutterAllocationFlags flags) {
	TangleWrapperActor* wrapper_actor;
	ClutterActorBox actor_box;
	TangleWrapperActorClass* wrapper_actor_class;
	
	wrapper_actor = TANGLE_WRAPPER_ACTOR(actor);
	
	CLUTTER_ACTOR_CLASS(tangle_wrapper_actor_parent_class)->allocate(actor, box, flags);

	actor_box.x1 = actor_box.y1 = 0.0;
	actor_box.x2 = box->x2 - box->x1;
	actor_box.y2 = box->y2 - box->y1;
	
	TANGLE_WRAPPER_ACTOR_GET_CLASS(wrapper_actor)->allocate_wrapped(wrapper_actor, wrapper_actor->priv->wrapped, &actor_box, flags);
}

static void tangle_wrapper_actor_paint(ClutterActor* actor) {
	gdouble scale_x;
	gdouble scale_y;

	if (CLUTTER_ACTOR_IS_MAPPED(TANGLE_WRAPPER_ACTOR(actor)->priv->wrapped)) {
		cogl_push_matrix();

		g_object_get(actor, "transition-scale-x", &scale_x, "transition-scale-y", &scale_y, NULL);
		cogl_scale(scale_x, scale_y, 0);

		clutter_actor_paint(TANGLE_WRAPPER_ACTOR(actor)->priv->wrapped);

		cogl_pop_matrix();
	}
}

static void tangle_wrapper_actor_pick(ClutterActor* actor, const ClutterColor* color) {
	CLUTTER_ACTOR_CLASS(tangle_wrapper_actor_parent_class)->pick(actor, color);
	
	tangle_wrapper_actor_paint(actor);
}

static void tangle_wrapper_actor_map(ClutterActor* actor) {
	CLUTTER_ACTOR_CLASS(tangle_wrapper_actor_parent_class)->map(actor);

	g_return_if_fail(TANGLE_WRAPPER_ACTOR(actor)->priv->wrapped);

	clutter_actor_map(TANGLE_WRAPPER_ACTOR(actor)->priv->wrapped);
}

static void tangle_wrapper_actor_unmap(ClutterActor* actor) {
	CLUTTER_ACTOR_CLASS(tangle_wrapper_actor_parent_class)->unmap(actor);

	g_return_if_fail(TANGLE_WRAPPER_ACTOR(actor)->priv->wrapped);

	clutter_actor_unmap(TANGLE_WRAPPER_ACTOR(actor)->priv->wrapped);
}

static void tangle_wrapper_actor_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleWrapperActor* wrapper_actor;
	
	wrapper_actor = TANGLE_WRAPPER_ACTOR(object);

	switch (prop_id) {
		case PROP_WRAPPED:
			wrapper_actor->priv->wrapped = CLUTTER_ACTOR(g_value_get_object(value));
			g_object_ref_sink(wrapper_actor->priv->wrapped);
			clutter_actor_set_parent(wrapper_actor->priv->wrapped, CLUTTER_ACTOR(wrapper_actor));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_wrapper_actor_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleWrapperActor* wrapper_actor;

	wrapper_actor = TANGLE_WRAPPER_ACTOR(object);

        switch (prop_id) {
		case PROP_WRAPPED:
			g_value_set_object(value, wrapper_actor->priv->wrapped);
			break;
		default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_wrapper_actor_dispose(GObject* object) {
	TangleWrapperActor* wrapper_actor;
	
	wrapper_actor = TANGLE_WRAPPER_ACTOR(object);
	
	if (wrapper_actor->priv->wrapped) {
		clutter_actor_destroy(wrapper_actor->priv->wrapped);
		g_object_unref(wrapper_actor->priv->wrapped);
		wrapper_actor->priv->wrapped = NULL;
	}

	G_OBJECT_CLASS(tangle_wrapper_actor_parent_class)->dispose(object);
}

static void tangle_wrapper_actor_class_init(TangleWrapperActorClass* wrapper_actor_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(wrapper_actor_class);
	ClutterActorClass* clutter_actor_class = CLUTTER_ACTOR_CLASS(wrapper_actor_class);
	TangleActorClass* actor_class = TANGLE_ACTOR_CLASS(wrapper_actor_class);

	gobject_class->dispose = tangle_wrapper_actor_dispose;
	gobject_class->set_property = tangle_wrapper_actor_set_property;
	gobject_class->get_property = tangle_wrapper_actor_get_property;

	clutter_actor_class->allocate = tangle_wrapper_actor_allocate;
	clutter_actor_class->paint = tangle_wrapper_actor_paint;
	clutter_actor_class->pick = tangle_wrapper_actor_pick;
	clutter_actor_class->map = tangle_wrapper_actor_map;
	clutter_actor_class->unmap = tangle_wrapper_actor_unmap;
	
	actor_class->get_preferred_width = tangle_wrapper_actor_get_preferred_width;
	actor_class->get_preferred_height = tangle_wrapper_actor_get_preferred_height;
	actor_class->handle_ancestor_event = tangle_wrapper_actor_handle_ancestor_event;

	wrapper_actor_class->allocate_wrapped = tangle_wrapper_actor_allocate_wrapped;

	/**
	 * TangleWrapperActor:actor:
	 */
	g_object_class_install_property(gobject_class, PROP_WRAPPED,
	                                g_param_spec_object("wrapped",
	                                                    "Wrapped",
	                                                    "The wrapped ClutterActor",
	                                                    CLUTTER_TYPE_ACTOR,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	g_type_class_add_private(gobject_class, sizeof(TangleWrapperActorPrivate));
}

static void tangle_wrapper_actor_init(TangleWrapperActor* wrapper_actor) {
	wrapper_actor->priv = G_TYPE_INSTANCE_GET_PRIVATE(wrapper_actor, TANGLE_TYPE_WRAPPER_ACTOR, TangleWrapperActorPrivate);
}

static void tangle_wrapper_actor_set_custom_property(ClutterScriptable* scriptable, ClutterScript* script,  const gchar* name, const GValue* value) {
	TangleWrapperActor* wrapper_actor;
	GParamSpec* param_spec;
	GObject* object = NULL;

	wrapper_actor = TANGLE_WRAPPER_ACTOR(scriptable);

	if ((param_spec = g_object_class_find_property(G_OBJECT_GET_CLASS(G_OBJECT(wrapper_actor)), name))) {
		object = G_OBJECT(wrapper_actor);
	} else if ((param_spec = g_object_class_find_property(G_OBJECT_GET_CLASS(G_OBJECT(wrapper_actor->priv->wrapped)), name))) {
		object = G_OBJECT(wrapper_actor->priv->wrapped);
	} else {
		g_warning("%s: Neither wrapper nor wrapped actor has property named '%s'.", G_STRLOC, name);
	}
	if (object) {
		g_object_set_property(object, name, value);
	}
}

static void clutter_scriptable_iface_init(ClutterScriptableIface *iface) {
	iface->set_custom_property = tangle_wrapper_actor_set_custom_property;
}
