/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef MICROFEEDFEED_H
#define MICROFEEDFEED_H

/**
 * @addtogroup provider libmicrofeed-provider - Modules that are used in the provider side
 * @{
 * @addtogroup MicrofeedFeed
 *
 * A feed with caching support and automatical updates.
 *
 * MicrofeedFeed is thread-aware, because it implements internal lock.
 * Usually an MicrofeedFeed object is accessed simultaneously from multiple threads, for example from
 * a feed update thread and from the main thread responding to DBus messages. When one thread is
 * accessing the object, others are waiting.
 *
 * Note that MicrofeedFeed launches a new thread for every update and similar operation that may take
 * some time. Thus, you have to call #microfeed_thread_init at the very first and
 * #microfeed_thread_cleanup at the very last in your application.
 * 
 * @{
 */

/**
 * Opaque data type representing Microfeed feed.
 */
typedef struct _MicrofeedFeed MicrofeedFeed;

/**
 * Opaque data type representing Microfeed feed iterator that iterates over the items of the feed.
 */
typedef struct _MicrofeedFeedIterator MicrofeedFeedIterator;

/** @} */

#include <microfeed-provider/microfeedpublisher.h>
#include <microfeed-common/microfeedstore.h>
#include <microfeed-common/microfeeditem.h>
#include <microfeed-common/microfeedmisc.h>
#include <microfeed-provider/microfeederror.h>

/**
 * @addtogroup MicrofeedFeed
 * @{
 */

typedef void (*MicrofeedFeedDestroyCallback)(MicrofeedFeed* feed, void* user_data);

/**
 * This function is called when a subscriber wants to update the feed.
 * 
 * The actual network operations are happening here. The function is called in a separate thread, so it
 * may also block. For pull-type services, typically one update cycle is implemented. For
 * push-type services, this function initializes the connection and waits for the events indefinitely.
 * However, the function must check peridiocally if the subscriber count (#microfeed_feed_get_subscriber_count)
 * has dropped to zero and in that case close the connection and return.
 *
 * The update thread has one reference to the feed meaning that the feed is not freed before the thread
 * finishes. The reference count is incremented and decremented automatically.
 *
 * If this function returns non-NULL error object, a signal to subscribers is sent.
 * 
 * @param feed Instantiated MicrofeedFeed.
 * @param user_intiatied Boolean value telling if the update is requested by a subscriber (true), or periodical event (false).
 * @param user_data The pointer given when the feed has been instantiated.
 * @return Error or NULL, if the operation succeeded.
 */
typedef MicrofeedError* (*MicrofeedFeedUpdateCallback)(MicrofeedFeed* feed, int user_initiated, void* user_data);

/**
 * This function is called when a subscriber wants to add a new item, modify an existing item, or delete an existing item.
 *
 * If the existing item parameter is null, the new item parameter contains a new item to be added.
 * If the new item parameter is null, the existing item parameter contains an esiting item to be deleted.
 * In other case, the existing item parameter contains the current properties of a modified item and
 * the new item parameter contains the new properties for the existing item.
 * 
 * @param feed Instantiated MicrofeedFeed.
 * @param existing_item An existing item or NULL.
 * @param new_item A new item or NULL.
 * @param user_data The pointer given when the feed has been instantiated.
 * @return Error or NULL, if the operation succeeded.
 */
typedef MicrofeedError* (*MicrofeedFeedModifyItemCallback)(MicrofeedFeed* feed, MicrofeedItem* existing_item, MicrofeedItem* new_item, void* user_data);
typedef MicrofeedError* (*MicrofeedFeedMarkCallback)(MicrofeedFeed* feed, const char* uid, int mark_status, void* user_data);
typedef void (*MicrofeedFeedCompleteItemPropertiesCallback)(MicrofeedFeed* feed, MicrofeedItem* item, void* user_data);

typedef struct {
	MicrofeedFeedDestroyCallback destroy;
	MicrofeedFeedUpdateCallback update;
	MicrofeedFeedModifyItemCallback modify_item;
	MicrofeedFeedMarkCallback mark_item;
	MicrofeedFeedCompleteItemPropertiesCallback complete_item_properties;
} MicrofeedFeedCallbacks;

typedef enum {
	MICROFEED_FEED_PERMISSION_NONE = 0,
	MICROFEED_FEED_PERMISSION_ADD = 1 << 0
} MicrofeedFeedPermission;

MicrofeedFeed* microfeed_feed_new(MicrofeedPublisher* publisher, const char* uri, const char* name, MicrofeedFeedPermission feed_permission, int create, MicrofeedFeedCallbacks* callbacks, void* user_data);
MicrofeedFeed* microfeed_feed_new_virtual(MicrofeedFeed* master_feed, const char* uri, const char* name, MicrofeedFeedPermission feed_permission, int create, MicrofeedFeedCallbacks* callbacks, void* user_data);

const char* microfeed_feed_get_uri(MicrofeedFeed* feed);
const char* microfeed_feed_get_name(MicrofeedFeed* feed);
MicrofeedPublisher* microfeed_feed_get_publisher(MicrofeedFeed* feed);
void microfeed_feed_set_name(MicrofeedFeed* feed, const char* name);
MicrofeedItem* microfeed_feed_get_metadata_item(MicrofeedFeed* feed);
void microfeed_feed_send_metadata_item(MicrofeedFeed* feed, const char* destination);
void microfeed_feed_replace_item(MicrofeedFeed* feed, MicrofeedItem* item);
void microfeed_feed_remove_item(MicrofeedFeed* feed, const char* uid);
void microfeed_feed_remove_items(MicrofeedFeed* feed, const char* start_uid, const char* end_uid);
MicrofeedItem* microfeed_feed_get_item(MicrofeedFeed* feed, const char* uid);
MicrofeedFeedIterator* microfeed_feed_iterate(MicrofeedFeed* feed, const char* start_uid, int backwards);
MicrofeedFeedIterator* microfeed_feed_iterate_timeline(MicrofeedFeed* feed, time_t start_timestamp, int backwards);
void microfeed_feed_update(MicrofeedFeed* feed, const char* bus_name);
void microfeed_feed_republish(MicrofeedFeed* feed, const char* start_uid, const char* end_uid, unsigned int max_count, const char* bus_name);
MicrofeedError* microfeed_feed_set_item_status(MicrofeedFeed* feed, const char* uid, MicrofeedItemStatus status_to_set);
MicrofeedError* microfeed_feed_unset_item_status(MicrofeedFeed* feed, const char* uid, MicrofeedItemStatus status_to_unset);
int microfeed_feed_get_item_count(MicrofeedFeed* feed);
MicrofeedFeedPermission microfeed_feed_permission_from_string(const char* string);
char* microfeed_feed_permission_to_string(MicrofeedFeedPermission item_permission);
void microfeed_feed_add_data_property(MicrofeedFeed* feed, const char* property_name);
void microfeed_feed_add_virtual_feed_property(MicrofeedFeed* feed, const char* property_name, const char* feed_uri_prefix, int create);
long microfeed_feed_get_last_updated_time(MicrofeedFeed* feed);
void microfeed_feed_start_periodical_update(MicrofeedFeed* feed);

void microfeed_feed_iterator_free(MicrofeedFeedIterator* iterator);
MicrofeedFeed* microfeed_feed_iterator_get_feed(MicrofeedFeedIterator* iterator);
MicrofeedItem* microfeed_feed_iterator_get_item(MicrofeedFeedIterator* iterator);
void microfeed_feed_iterator_next(MicrofeedFeedIterator* iterator);
int microfeed_feed_iterator_jump_and_remove_previous_items(MicrofeedFeedIterator* iterator, const char* uid);

/**
 * @}
 * @}
 */

/* For internal use only. */
MicrofeedError* microfeed_feed_call_modify_item_callback(MicrofeedFeed* feed, MicrofeedItem* existing_item, MicrofeedItem* new_item);

#endif
