/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation, or under the terms of the GNU Lesser General
 *  Public License version 2.1 as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <microfeed-common/microfeedfileindextable.h>
#include <microfeed-common/microfeedmisc.h>

#include <string.h>

typedef uint32_t Size;

struct _MicrofeedFileIndexTable {
	MicrofeedFile* file;
	MicrofeedFileIndex index;
	MicrofeedFileIndexTableGetKeyFunction get_key;
	MicrofeedFileIndexTableCompareKeysFunction compare_keys;
	unsigned int delta;
	MicrofeedFileIndexTableIterator* iterators;
};

struct _MicrofeedFileIndexTableIterator {
	MicrofeedFileIndexTableIterator* previous;
	MicrofeedFileIndexTableIterator* next;
	
	MicrofeedFileIndexTable* table;
	int backwards;
	Size index;
	MicrofeedFileIndex current_index;
};

typedef struct {
	time_t last_updated;
	Size size;
	Size reserved;
	MicrofeedFileIndex indices[1];
} Data;

static int get_index_block(MicrofeedFileIndexTable* table, Data* data, const void* key, size_t key_size, Size* index_return, void** block_return);
static Data* remove_index(MicrofeedFileIndexTable* table, Data* data, Size index);
static int compare_named_index_keys(const void* key1, size_t key1_size, const void* key2, size_t key2_size);
static  void get_named_index_key(const void* block, size_t block_size, const void** key_return, size_t* key_size_return);
static void check_state(MicrofeedFileIndexTable* table);

MicrofeedFileIndexTable* microfeed_file_index_table_new_sorted(MicrofeedFile* file, MicrofeedFileIndex index, int create, MicrofeedFileIndexTableCompareKeysFunction compare_keys, MicrofeedFileIndexTableGetKeyFunction get_key) {

	return microfeed_file_index_table_new_sorted_with_delta(file, index, create, 16, compare_keys, get_key);
}

MicrofeedFileIndexTable* microfeed_file_index_table_new_sorted_with_delta(MicrofeedFile* file, MicrofeedFileIndex index, int create, unsigned int delta, MicrofeedFileIndexTableCompareKeysFunction compare_keys, MicrofeedFileIndexTableGetKeyFunction get_key) {
	MicrofeedFileIndexTable* table;
	Data* data;

	if (index == MICROFEED_FILE_INDEX_INVALID || !microfeed_file_get_block(file, index, Data)) {
		if (!create) {

			return NULL;
		} else {
			data = microfeed_file_allocate_block(file, sizeof(Data) - sizeof(MicrofeedFileIndex) + delta * sizeof(MicrofeedFileIndex), Data);
			if (index != MICROFEED_FILE_INDEX_INVALID && microfeed_file_get_index(file, data) != index) {
				microfeed_file_free_block(file, data);
			
				return NULL;
			} else if (index == MICROFEED_FILE_INDEX_INVALID) {
				index = microfeed_file_get_index(file, data);
			}
			data->last_updated = time(NULL);
			data->size = 0;
			data->reserved = delta;
		}
	}

	table = microfeed_memory_allocate(MicrofeedFileIndexTable);
	table->file = file;
	table->index = index;
	table->get_key = get_key;
	table->compare_keys = compare_keys;
	table->delta = delta;
	
	check_state(table);
	
	return table;
}

MicrofeedFileIndexTable* microfeed_file_index_table_new_named_indices(MicrofeedFile* file, MicrofeedFileIndex index, int create) {

	return microfeed_file_index_table_new_sorted(file, index, create, microfeed_file_index_table_compare_keys_direct, get_named_index_key);
}

void microfeed_file_index_table_free(MicrofeedFileIndexTable* table) {
	check_state(table);

	microfeed_assert(!table->iterators);
	
	microfeed_memory_free(table);	
}

MicrofeedFileIndex microfeed_file_index_table_get_file_index(MicrofeedFileIndexTable* table) {

	return table->index;
}

int microfeed_file_index_table_insert_index(MicrofeedFileIndexTable* table, MicrofeedFileIndex index) {
	
	return microfeed_file_index_table_insert_block(table, microfeed_file_get_block_impl(table->file, index));
}

int microfeed_file_index_table_insert_block(MicrofeedFileIndexTable* table, void* block) {
	int retvalue = 0;
	Data* data;
	MicrofeedFileIndex block_index;
	const void* key;
	size_t key_size;
	Size index;
	void* existing_block;
	MicrofeedFileIndexTableIterator* iterator;
	
	data = microfeed_file_get_block(table->file, table->index, Data);
	block_index = microfeed_file_get_index(table->file, block);
	table->get_key(block, microfeed_file_block_get_size(block), &key, &key_size);
	if (!get_index_block(table, data, key, key_size, &index, &existing_block)) {
		if (data->size == data->reserved) {
			data->reserved += table->delta;
			data = microfeed_file_resize_block(table->file, data, sizeof(Data) - sizeof(MicrofeedFileIndex) + data->reserved * sizeof(MicrofeedFileIndex), Data);
		}
		
		if (index < data->size) {
			memmove(data->indices + index + 1, data->indices + index,
			        (data->size - index) * sizeof(MicrofeedFileIndex));
		}
		data->size++;
		data->indices[index] = block_index;
		data->last_updated = time(NULL);

		retvalue = 1;
		
		for (iterator = table->iterators; iterator; iterator = iterator->next) {
			if (iterator->index <= index) {
				iterator->index++;
			}
		}
	}
	
	check_state(table);

	return retvalue;	
}

MicrofeedFileIndex microfeed_file_index_table_replace_index(MicrofeedFileIndexTable* table, MicrofeedFileIndex index) {
	MicrofeedFileIndex return_index = MICROFEED_FILE_INDEX_INVALID;
	void* block;
	
	if ((block = microfeed_file_index_table_replace_block_impl(table, microfeed_file_get_block_impl(table->file, index)))) {
		return_index = microfeed_file_get_index(table->file, block);
	}
	
	check_state(table);
	
	return return_index;
}
	
void* microfeed_file_index_table_replace_block_impl(MicrofeedFileIndexTable* table, void* block) {
	Data* data;
	MicrofeedFileIndex block_index;
	const void* key;
	size_t key_size;
	Size index;
	void* existing_block;
	
	data = microfeed_file_get_block(table->file, table->index, Data);
	block_index = microfeed_file_get_index(table->file, block);
	table->get_key(block, microfeed_file_block_get_size(block), &key, &key_size);
	if (!get_index_block(table, data, key, key_size, &index, &existing_block)) {
		if (data->size == data->reserved) {
			data->reserved += table->delta;
			data = microfeed_file_resize_block(table->file, data, sizeof(Data) - sizeof(MicrofeedFileIndex) + data->reserved * sizeof(MicrofeedFileIndex), Data);
		}
		
		if (index < data->size) {
			memmove(data->indices + index + 1, data->indices + index,
			        (data->size - index) * sizeof(MicrofeedFileIndex));
		}
		data->size++;
	}
	data->indices[index] = block_index;
	data->last_updated = time(NULL);
	
	check_state(table);
	
	return existing_block;
}

MicrofeedFileIndex microfeed_file_index_table_update_index(MicrofeedFileIndexTable* table, MicrofeedFileIndex index, const void* new_data, size_t new_data_size) {
	MicrofeedFileIndex return_index = MICROFEED_FILE_INDEX_INVALID;
	void* block;
	
	if ((block = microfeed_file_index_table_update_block_impl(table, microfeed_file_get_block_impl(table->file, index), new_data, new_data_size))) {
		return_index = microfeed_file_get_index(table->file, block);
	}
	
	check_state(table);
	
	return return_index;
}

void* microfeed_file_index_table_update_block_impl(MicrofeedFileIndexTable* table, void* block, const void* new_data, size_t new_data_size) {
	Data* data;
	MicrofeedFileIndex block_index;
	const void* existing_key;
	size_t existing_key_size;
	const void* new_key;
	size_t new_key_size;
	Size existing_index;
	Size new_index;
	MicrofeedFileIndex file_index;
	void* existing_block = NULL;
	MicrofeedFileIndexTableIterator* iterator;
	
	data = microfeed_file_get_block(table->file, table->index, Data);
	block_index = microfeed_file_get_index(table->file, block);
	table->get_key(block, microfeed_file_block_get_size(block), &existing_key, &existing_key_size);
	table->get_key(new_data, new_data_size, &new_key, &new_key_size);
	if (table->compare_keys(existing_key, existing_key_size, new_key, new_key_size)) {
		microfeed_assert(get_index_block(table, data, existing_key, existing_key_size, &existing_index, &existing_block));
		microfeed_assert(block == existing_block);
		if (get_index_block(table, data, new_key, new_key_size, &new_index, &existing_block)) {
			file_index = microfeed_file_get_index(table->file, existing_block);
			data = remove_index(table, data, existing_index);
			if (existing_index < new_index) {
				new_index--;
			}
			existing_block = microfeed_file_get_index(table->file, file_index);
		} else if (existing_index < new_index) {
			new_index--;
			memmove(data->indices + existing_index, data->indices + existing_index + 1,
				(new_index - existing_index) * sizeof(MicrofeedFileIndex));
			for (iterator = table->iterators; iterator; iterator = iterator->next) {
				if (iterator->index >= existing_index && iterator->index < new_index) {
					iterator->index--;
				}
			}
		} else {
			memmove(data->indices + new_index + 1, data->indices + new_index,
				(existing_index - new_index) * sizeof(MicrofeedFileIndex));				
			for (iterator = table->iterators; iterator; iterator = iterator->next) {
				if (iterator->index >= new_index && iterator->index < existing_index) {
					iterator->index++;
				}
			}
		}
		data->indices[new_index] = block_index;
		data->last_updated = time(NULL);
	}
	
	return existing_block;
}

int microfeed_file_index_table_remove_index(MicrofeedFileIndexTable* table, MicrofeedFileIndex index) {
	int retvalue = 0;
	Data* data;
	Size i;
	
	data = microfeed_file_get_block(table->file, table->index, Data);
	for (i = 0; i < data->size; i++) {
		if (data->indices[i] == index) {
			remove_index(table, data, i);
			retvalue = 1;
			break;
		}
	}
	
	check_state(table);
	
	return retvalue;
}

int microfeed_file_index_table_remove_block(MicrofeedFileIndexTable* table, void* block) {
	
	return microfeed_file_index_table_remove_index(table, microfeed_file_get_index(table->file, block));
}

MicrofeedFileIndex microfeed_file_index_table_remove_key(MicrofeedFileIndexTable* table, const void* key, size_t key_size) {
	MicrofeedFileIndex return_index = MICROFEED_FILE_INDEX_INVALID;
	Data* data;
	Size index;
	void* block;
	
	data = microfeed_file_get_block(table->file, table->index, Data);
	if (get_index_block(table, data, key, key_size, &index, &block)) {
		return_index = data->indices[index];
		remove_index(table, data, index);
	}
	
	check_state(table);
	
	return return_index;
}

void* microfeed_file_index_table_get_block_impl(MicrofeedFileIndexTable* table, const void* key, size_t key_size) {
	Data* data;
	Size index;
	void* block;
	
	data = microfeed_file_get_block(table->file, table->index, Data);
	get_index_block(table, data, key, key_size, &index, &block);
	
	return block;
}

MicrofeedFileIndex microfeed_file_index_table_get_index(MicrofeedFileIndexTable* table, const void* key, size_t key_size) {
	MicrofeedFileIndex return_index = MICROFEED_FILE_INDEX_INVALID;
	Data* data;
	Size index;
	void* block;
	
	data = microfeed_file_get_block(table->file, table->index, Data);
	if (get_index_block(table, data, key, key_size, &index, &block)) {
		return_index = data->indices[index];
	}
	
	return return_index;
}

MicrofeedFileIndexTableIterator* microfeed_file_index_table_iterate(MicrofeedFileIndexTable* table, const void* start_key, size_t start_key_size, int backwards) {
	MicrofeedFileIndexTableIterator* iterator;
	Data* data;
	Size index;
	void* block;
	
	data = microfeed_file_get_block(table->file, table->index, Data);

	iterator = microfeed_memory_allocate(MicrofeedFileIndexTableIterator);
	iterator->table = table;
	iterator->backwards = backwards;
	if (table->iterators) {
		table->iterators->previous = table->iterators;
		iterator->next = table->iterators;
	}
	table->iterators = iterator;
	iterator->current_index = MICROFEED_FILE_INDEX_INVALID;

	if (start_key) {
		if (get_index_block(table, data, start_key, start_key_size, &index, &block)) {
			iterator->index = index;
		} else if (backwards) {
			iterator->index = (Size)-1;
		} else {
			iterator->index = data->size;
		}	
	} else if (backwards) {
		iterator->index = data->size - 1;
	}
	
	return iterator;	
}

void microfeed_file_index_table_get_block_key(MicrofeedFileIndexTable* table, const void* block, size_t block_size, const void** key_return, size_t* key_size_return) {
	table->get_key(block, block_size, key_return, key_size_return);
}

int microfeed_file_index_table_compare_keys(MicrofeedFileIndexTable* table, const void* key1, size_t key1_size, const void* key2, size_t key2_size) {

	return table->compare_keys(key1, key1_size, key2, key2_size);
}


time_t microfeed_file_index_table_get_last_updated_time(MicrofeedFileIndexTable* table) {
	Data* data;
	
	data = microfeed_file_get_block(table->file, table->index, Data);

	return data->last_updated;
}

void microfeed_file_index_table_set_last_updated_time(MicrofeedFileIndexTable* table, time_t timestamp) {
	Data* data;
	
	data = microfeed_file_get_block(table->file, table->index, Data);
	data->last_updated = timestamp;
}

int microfeed_file_index_table_get_size(MicrofeedFileIndexTable* table) {
	Data* data;
	
	data = microfeed_file_get_block(table->file, table->index, Data);

	return data->size;
}

int microfeed_file_index_table_insert_named_index(MicrofeedFileIndexTable* table, const char* name, MicrofeedFileIndex index) {
	int retvalue = 1;
	size_t length;
	void* block;
	
	length = strlen(name);
	block = microfeed_file_allocate_block_impl(table->file, sizeof(MicrofeedFileIndex) + length);
	*((MicrofeedFileIndex*)block) = index;
	memcpy(block + sizeof(MicrofeedFileIndex), name, length);
	if (!microfeed_file_index_table_insert_block(table, block)) {
		microfeed_file_free_block(table->file, block);
		retvalue = 0;
	}
	
	return retvalue;
}

int microfeed_file_index_table_remove_named_index(MicrofeedFileIndexTable* table, const char* name) {
	int retvalue = 0;
	void* block;
	MicrofeedFileIndex index;
	
	if ((block = microfeed_file_index_table_get_block_impl(table, name, strlen(name)))) {
		index = microfeed_file_get_index(table->file, block);
		microfeed_file_free_block(table->file, block);
		microfeed_file_index_table_remove_index(table, index);
		retvalue = 1;
	}
	
	return retvalue;
}

MicrofeedFileIndex microfeed_file_index_table_get_named_index(MicrofeedFileIndexTable* table, const char* name) {
	MicrofeedFileIndex index = MICROFEED_FILE_INDEX_INVALID;
	void* block;
	
	if ((block = microfeed_file_index_table_get_block_impl(table, name, strlen(name)))) {
		index = *((MicrofeedFileIndex*)block);
	}
	
	return index;
}

void microfeed_file_index_table_iterator_free(MicrofeedFileIndexTableIterator* iterator) {
	if (iterator->next) {
		iterator->next->previous = iterator->previous;
	}
	if (iterator->previous) {
		iterator->previous->next = iterator->next;
	} else {
		iterator->table->iterators = iterator->next;
	}
	iterator->table = NULL;
	microfeed_memory_free(iterator);
}

MicrofeedFileIndexTable* microfeed_file_index_table_iterator_get_table(MicrofeedFileIndexTableIterator* iterator) {

	return iterator->table;
}


void* microfeed_file_index_table_iterator_get_block_impl(MicrofeedFileIndexTableIterator* iterator) {
	void* block = NULL;
	MicrofeedFileIndex index;

	if ((index = microfeed_file_index_table_iterator_get_index(iterator)) != MICROFEED_FILE_INDEX_INVALID) {
		block = microfeed_file_get_block_impl(iterator->table->file, index);
	}

	return block;
}

MicrofeedFileIndex microfeed_file_index_table_iterator_get_index(MicrofeedFileIndexTableIterator* iterator) {
	Data* data;
	
	data = microfeed_file_get_block(iterator->table->file, iterator->table->index, Data);
	if (iterator->current_index == MICROFEED_FILE_INDEX_INVALID && iterator->index != (Size)-1 && iterator->index < data->size) {
		iterator->current_index = data->indices[iterator->index];
	}
	
	return iterator->current_index;
}

void microfeed_file_index_table_iterator_next(MicrofeedFileIndexTableIterator* iterator) {
	Data* data;
	
	data = microfeed_file_get_block(iterator->table->file, iterator->table->index, Data);
	if (iterator->backwards) {
		if (iterator->index != (Size)-1) {
			iterator->index--;
		}
	} else {
		if (iterator->index < data->size || iterator->index == (Size)-1) {
			iterator->index++;
		}
	}
	iterator->current_index = MICROFEED_FILE_INDEX_INVALID;
}

int microfeed_file_index_table_compare_keys_direct(const void* key1, size_t key1_size, const void* key2, size_t key2_size) {
	int result;
	
	if (key1_size < key2_size) {
		if (!(result = memcmp(key1, key2, key1_size))) {
			result = -1;
		}
	} else if (key1_size > key2_size) {
		if (!(result = memcmp(key1, key2, key1_size))) {
			result = 1;
		}	
	} else {
		result = memcmp(key1, key2, key1_size);
	}
	
	return result;
}

void microfeed_file_index_table_get_key_direct(const void* block, size_t block_size, const void** key_return, size_t* key_size_return) {
	*key_return = block;
	*key_size_return = block_size;
}

static int get_index_block(MicrofeedFileIndexTable* table, Data* data, const void* key, size_t key_size, Size* index_return, void** block_return) {
	int retval = 0;
	void* block;
	const void* block_key;
	size_t block_key_size;
	int result;
	long int i, min, max;

	*block_return = NULL;
	if (data->size == 0) {
		*index_return = 0;
	} else if (block = microfeed_file_get_block_impl(table->file, data->indices[0]),
	           table->get_key(block, microfeed_file_block_get_size(block), &block_key, &block_key_size),
	           (result = table->compare_keys(key, key_size, block_key, block_key_size)) == 0) {
		*index_return = 0;
		*block_return = block;
		retval = 1;	
	} else if (result < 0) {
		*index_return = 0;
	} else if (data->size == 1) {
		*index_return = 1;
	} else if (block = microfeed_file_get_block_impl(table->file, data->indices[data->size - 1]),
	           table->get_key(block, microfeed_file_block_get_size(block), &block_key, &block_key_size),
	           (result = table->compare_keys(key, key_size, block_key, block_key_size)) == 0) {
		*index_return = data->size - 1;
		*block_return = block;
		retval = 1;
	} else if (result > 0) {
		*index_return = data->size;
	} else if (data->size == 2) {
		*index_return = data->size - 1;
	} else {
		min = i = 0;
		max = data->size - 1;

		while (min <= max) {
			i = (min + max) / 2;
			microfeed_assert((block = microfeed_file_get_block_impl(table->file, data->indices[i])));
			table->get_key(block, microfeed_file_block_get_size(block), &block_key, &block_key_size);
			if ((result = table->compare_keys(key, key_size, block_key, block_key_size)) == 0) {
				*block_return = block;
				retval = 1;
				break;
			} else if (result < 0) {
				max = i - 1;
			} else {
				i++;
				min = i;
			}
		}

		*index_return = i;
	}

	return retval;
}

static Data* remove_index(MicrofeedFileIndexTable* table, Data* data, Size index) {
	MicrofeedFileIndexTableIterator* iterator;

	if (index + 1 < data->size) {
		memmove(data->indices + index, data->indices + index + 1,
		        (data->size - index - 1) * sizeof(MicrofeedFileIndex));
	}

	data->size--;
	if (data->size + 2 * table->delta == data->reserved) {
		data->reserved -= table->delta;
		data = microfeed_file_resize_block(table->file, data, sizeof(Data) - sizeof(MicrofeedFileIndex) + data->reserved * sizeof(MicrofeedFileIndex), Data);
	}

	for (iterator = table->iterators; iterator; iterator = iterator->next) {
		if (iterator->index >= index) {
			iterator->index--;
		}

	}
	data->last_updated = time(NULL);

	return data;
}

static void get_named_index_key(const void* block, size_t block_size, const void** key_return, size_t* key_size_return) {
	*key_return = block + sizeof(MicrofeedFileIndex);
	*key_size_return = block_size - sizeof(MicrofeedFileIndex);
}

static void check_state(MicrofeedFileIndexTable* table) {
	Data* data;
	long int i;

	data = microfeed_file_get_block(table->file, table->index, Data);
	microfeed_assert(data->size <= data->reserved);
	for (i = 0; i < data->size; i++) {
		microfeed_assert(microfeed_file_get_block_impl(table->file, data->indices[i]));
	}
}
