/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef MICROFEEDDATABASE_H
#define MICROFEEDDATABASE_H

#include <sys/types.h>

/**
 * @addtogroup provider libmicrofeed-provider - Modules that are used in the provider side
 * @{
 * @addtogroup MicrofeedDatabase
 *
 * A database layer.
 *
 * This module is used internally by @link MicrofeedFeed MicrofeedFeed@endlink. It is not accessed directly
 * when developing a provider. However, it can be also utilized outside of the library implementation.
 *
 * @{
 */

/**
 * Opaque data type representing a database which contains actual database tables.
 */
typedef struct _MicrofeedDatabase MicrofeedDatabase;

/**
 * Opaque data type representing a database table.
 */
typedef struct _MicrofeedDatabaseTable MicrofeedDatabaseTable;

/**
 * Opaque data type representing an index of a database table.
 */
typedef struct _MicrofeedDatabaseIndex MicrofeedDatabaseIndex;

/**
 * Opaque data type representing database iterator that iterates over the keys of the database.
 */
typedef struct _MicrofeedDatabaseIterator MicrofeedDatabaseIterator;

/**
 * This function is called when a new item is added into a database.
 * 
 * The function must extract the index key from the given data and return it using the
 * key_return and key_size_return parameters.
 * 
 * @param data The data of the added item.
 * @param data_size The size of the data.
 * @param key_return [return] The index key of the item.
 * @param key_size_return [return] The size of the index key.
 */
typedef void (*MicrofeedDatabaseIndexFunction)(const void* data, size_t data_size, const void** key_return, size_t* key_size_return);

/**
 * This function is called when two keys in a database are compared by the database engine.
 * 
 * The function must return negative integer if key1 is less than key2, zero if key1 is equal than key2, and
 * positive integer if key1 is greater than key2.
 * 
 * @param key1 The first key to compare.
 * @param key1_size The size of the first key.
 * @param key2 The second key to compare.
 * @param key2_size The size of the second key.
 * @return Negative integer, zero or positive integer depending of the order of the keys.
 */
typedef int (*MicrofeedDatabaseCompareFunction)(const void* key1, size_t key1_size, const void* key2, size_t key2_size);

MicrofeedDatabase* microfeed_database_new(const char* name, const char* directory);
MicrofeedDatabaseTable* microfeed_database_get_table(MicrofeedDatabase* database, const char* name);
int microfeed_database_get_named_data(MicrofeedDatabase* database, const char* name, void** data_return, size_t* data_size_return);
int microfeed_database_set_named_data(MicrofeedDatabase* database, const char* name, void* data, size_t data_size);
void microfeed_database_remove_named_data(MicrofeedDatabase* database, const char* name);

MicrofeedDatabaseIndex* microfeed_database_table_get_index(MicrofeedDatabaseTable* database_table, const char* name, int create, MicrofeedDatabaseIndexFunction index_function, MicrofeedDatabaseCompareFunction compare_function);
const char* microfeed_database_table_get_name(MicrofeedDatabaseTable* database_table);
int microfeed_database_table_get_data_count(MicrofeedDatabaseTable* database_table);
time_t microfeed_database_table_get_last_updated_time(MicrofeedDatabaseTable* database_table);
void microfeed_database_table_set_to_be_removed(MicrofeedDatabaseTable* database_table);

MicrofeedDatabaseTable* microfeed_database_index_get_table(MicrofeedDatabaseIndex* database_index);
const char* microfeed_database_index_get_name(MicrofeedDatabaseIndex* database_index);
void microfeed_database_index_replace_data(MicrofeedDatabaseIndex* database_index, const void* data, size_t data_size);
void microfeed_database_index_replace_data_partial(MicrofeedDatabaseIndex* database_index, const void* data, size_t data_size);
int microfeed_database_index_get_data(MicrofeedDatabaseIndex* database_index, const void* key, size_t key_size, void** data_return, size_t* data_size_return);
int microfeed_database_index_get_data_partial(MicrofeedDatabaseIndex* database_index, const void* key, size_t key_size, void** data_return, size_t* data_size_return);
int microfeed_database_index_is_data(MicrofeedDatabaseIndex* database_index, const void* key, size_t key_size);
MicrofeedDatabaseIterator* microfeed_database_index_iterate(MicrofeedDatabaseIndex* database_index, const void* start_key, size_t start_key_size, int backwards);
void microfeed_database_index_remove_data(MicrofeedDatabaseIndex* database_index, const void* key, size_t key_size);
void microfeed_database_index_remove_data_range(MicrofeedDatabaseIndex* database_index, const void* start_key, const size_t start_key_size, const void* end_key, size_t end_key_size);

void microfeed_database_iterator_free(MicrofeedDatabaseIterator* iterator);
int microfeed_database_iterator_get(MicrofeedDatabaseIterator* iterator, void** data_return, size_t* data_size_return);
void microfeed_database_iterator_next(MicrofeedDatabaseIterator* iterator);

int microfeed_database_compare_keys_direct(const void* key1, size_t key1_size, const void* key2, size_t key2_size);

/**
 * @}
 * @}
 */

#endif
