/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation, or under the terms of the GNU Lesser General
 *  Public License version 2.1 as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
 
#include <microfeed-common/microfeedthread.h>
#include <microfeed-common/microfeedmisc.h>
#include <microfeed-common/microfeedstore.h>

#include <pthread.h>
#include <signal.h>

#include <stdio.h>

struct _MicrofeedThread {
	MicrofeedThread* next;
	MicrofeedThread* previous;

	unsigned int reference_count;
	MicrofeedMutex* mutex;
	unsigned long id;

	MicrofeedThreadPool* thread_pool;
	void* thread_implementation;
	MicrofeedThreadFunction function;
	void* in_data;
	void* out_data;
	MicrofeedThreadExitCallback exit_callback;
	void* user_data;
};

struct _MicrofeedThreadPool {
	unsigned int started_threads;
	unsigned int max_threads;
	MicrofeedThreadExitCallback exit_callback;
	void* user_data;
	MicrofeedMutex* mutex;
	MicrofeedStore* waiting_threads;
};

struct _MicrofeedMutex {
	void* mutex_implementation;
};

static void* default_thread_new(void* (*function)(void* data), void* data);
static void default_thread_free(void* thread_implementation);
static void* default_thread_get_current(void);
static void default_thread_send_signal(void* thread_implementation, int signal_number);
static void default_thread_join(void* thread_implementation);
static void* default_mutex_new(void);
static void default_mutex_free(void* mutex_implementation);
static void default_mutex_lock(void* mutex_implementation);
static void default_mutex_unlock(void* mutex_implementation);
static void* thread_function(void* data);

MicrofeedThreadFunctions functions = {
	default_thread_new,
	default_thread_free,
	default_thread_get_current,
	default_thread_send_signal,
	default_thread_join,
	default_mutex_new,
	default_mutex_free,
	default_mutex_lock,
	default_mutex_unlock
};

static MicrofeedThread* threads = NULL;
static MicrofeedMutex* threads_mutex = NULL;
static unsigned long thread_count = 0;

/**
 * Sets the functions that implements the thread and mutex operations.
 * 
 * @param thread_functions A structure containing new implementions.
 */
void microfeed_thread_set_functions(MicrofeedThreadFunctions* thread_functions) {
	functions = *thread_functions;
}

void microfeed_thread_init(void) {
	if (!threads_mutex) {
		threads_mutex = microfeed_mutex_new();
	}
	if (!threads) {
		threads = microfeed_thread_get_current();
	}
}

void microfeed_thread_cleanup(void) {
	void* current_thread_implementation;
	MicrofeedThread* thread;
	
	microfeed_mutex_lock(threads_mutex);

	current_thread_implementation = functions.thread_get_current();	

	do {
		for (thread = threads; thread; thread = thread->next) {
			if (thread->thread_implementation && thread->thread_implementation != current_thread_implementation) {
				break;
			}
		}
		if (thread) {
			microfeed_mutex_unlock(threads_mutex);

/* TODO: What to do (if anything) if a thread is still running? */
			sleep(1);
			
			microfeed_mutex_lock(threads_mutex);
		}
	} while (thread);
	
	microfeed_mutex_unlock(threads_mutex);
}

MicrofeedThread* microfeed_thread_new(MicrofeedThreadFunction function, void* data) {
	
	return microfeed_thread_new_with_exit_callback(function, data, NULL, NULL);
}

MicrofeedThread* microfeed_thread_new_with_exit_callback(MicrofeedThreadFunction function, void* data, MicrofeedThreadExitCallback exit_callback, void* user_data) {
	MicrofeedThread* thread;
	
	thread = microfeed_memory_allocate(MicrofeedThread);
	thread->reference_count = 2;
	thread->mutex = microfeed_mutex_new();
	thread->function = function;
	thread->in_data = data;
	thread->out_data= NULL;
	thread->exit_callback = exit_callback;
	thread->user_data = user_data;

	microfeed_mutex_lock(threads_mutex);
	
	thread->id = ++thread_count;

	thread->next = threads;
	if (thread->next) {
		thread->next->previous = thread;
	}
	threads = thread;
			
	microfeed_mutex_lock(thread->mutex);

	thread->thread_implementation = functions.thread_new(thread_function, thread);

	microfeed_mutex_unlock(thread->mutex);

	microfeed_mutex_unlock(threads_mutex);
	
	return thread;
}

void microfeed_thread_free(MicrofeedThread* thread) {
	if (thread) {
		microfeed_mutex_lock(threads_mutex);
	
		if (thread->previous) {
			thread->previous->next = thread->next;
		} else if (threads) {
			threads = thread->next;
		}
		if (thread->next) {
			thread->next->previous = thread->previous;
		}
	
		microfeed_mutex_unlock(threads_mutex);
	
		microfeed_mutex_free(thread->mutex);
		microfeed_memory_free(thread);
	}
}

MicrofeedThread* microfeed_thread_ref(MicrofeedThread* thread) {
	microfeed_mutex_lock(thread->mutex);

	thread->reference_count++;

	microfeed_mutex_unlock(thread->mutex);
	
	return thread;
}

void microfeed_thread_unref(MicrofeedThread* thread) {
	microfeed_mutex_lock(thread->mutex);

	thread->reference_count--;
	if (thread->reference_count == 0) {
		microfeed_thread_free(thread);
	} else {
		
		microfeed_mutex_unlock(thread->mutex);
	}
}

MicrofeedThread* microfeed_thread_get_current(void) {
	MicrofeedThread* thread = NULL;
	void* thread_implementation;
	
	microfeed_mutex_lock(threads_mutex);
	
	if ((thread_implementation = functions.thread_get_current())) {
		for (thread = threads; thread; thread = thread->next) {
			if (thread->thread_implementation == thread_implementation) {
				break;
			}
		}
	}
	if (!thread) {
		thread = microfeed_memory_allocate(MicrofeedThread);
		thread->reference_count = 1;
		thread->mutex = microfeed_mutex_new();
		thread->id = 0;
		thread->function = NULL;
		thread->in_data = NULL;
		thread->out_data= NULL;

		thread->next = threads;
		if (thread->next) {
			thread->next->previous = thread;
		}
		threads = thread;
		thread->thread_implementation = thread_implementation;
	}
	
	microfeed_mutex_unlock(threads_mutex);

	return thread;
}

void microfeed_thread_send_signal(MicrofeedThread* thread, int signal_number) {
	if (thread->thread_implementation) {
		functions.thread_send_signal(thread->thread_implementation, signal_number);
	}
}

unsigned long microfeed_thread_get_id(MicrofeedThread* thread) {
	
	return thread->id;	
}

MicrofeedThreadPool* microfeed_thread_pool_new(unsigned int maximum_thread_count) {

	return microfeed_thread_pool_new_with_exit_callback(maximum_thread_count, NULL, NULL);
}

/**
 * ...
 * 
 * An exit function is called when a <b>physical thread</b> exits.
 */
MicrofeedThreadPool* microfeed_thread_pool_new_with_exit_callback(unsigned int maximum_thread_count, MicrofeedThreadExitCallback exit_callback, void* user_data) {
	MicrofeedThreadPool* thread_pool;
	
	thread_pool = microfeed_memory_allocate(MicrofeedThreadPool);
	thread_pool->max_threads = maximum_thread_count;
	thread_pool->exit_callback = exit_callback;
	thread_pool->user_data = user_data;
	thread_pool->mutex = microfeed_mutex_new();
	thread_pool->waiting_threads = microfeed_store_new_unsorted_data(microfeed_store_compare_keys_direct, microfeed_store_get_key_direct);
	
	return thread_pool;
	
}

void microfeed_thread_pool_free(MicrofeedThreadPool* thread_pool) {
	microfeed_assert(thread_pool->started_threads == 0);
	
	microfeed_mutex_free(thread_pool->mutex);	
	microfeed_store_free(thread_pool->waiting_threads);
	microfeed_memory_free(thread_pool);
}

MicrofeedThread* microfeed_thread_pool_queue_thread(MicrofeedThreadPool* thread_pool, MicrofeedThreadFunction function, void* data) {

	return microfeed_thread_pool_queue_thread_with_exit_callback(thread_pool, function, data, NULL, NULL);
}

MicrofeedThread* microfeed_thread_pool_queue_thread_with_exit_callback(MicrofeedThreadPool* thread_pool, MicrofeedThreadFunction function, void* data, MicrofeedThreadExitCallback exit_callback, void* user_data) {
	MicrofeedThread* thread;
	
	thread = microfeed_memory_allocate(MicrofeedThread);
	thread->reference_count = 2;
	thread->mutex = microfeed_mutex_new();
	thread->thread_pool = thread_pool;
	thread->function = function;
	thread->in_data = data;
	thread->out_data= NULL;
	thread->exit_callback = exit_callback;
	thread->user_data = user_data;

	microfeed_mutex_lock(thread_pool->mutex);

	if (thread_pool->started_threads < thread_pool->max_threads) {
		thread_pool->started_threads++;
		
		microfeed_mutex_lock(threads_mutex);
	
		thread->next = threads;
		if (thread->next) {
			thread->next->previous = thread;
		}
		threads = thread;
			
		microfeed_mutex_lock(thread->mutex);
		thread->thread_implementation = functions.thread_new(thread_function, thread);
	
		microfeed_mutex_unlock(thread->mutex);
	
		microfeed_mutex_unlock(threads_mutex);
	} else {
		microfeed_store_insert(thread_pool->waiting_threads, thread);
	}		

	microfeed_mutex_unlock(thread_pool->mutex);
	
	return thread;
}

unsigned int microfeed_thread_pool_get_started_thread_count(MicrofeedThreadPool* thread_pool) {
	
	return thread_pool->started_threads;
}

unsigned int microfeed_thread_pool_get_waiting_thread_count(MicrofeedThreadPool* thread_pool) {
	
	return microfeed_store_get_size(thread_pool->waiting_threads);
}

void microfeed_thread_pool_set_maximum_thread_count(MicrofeedThreadPool* thread_pool, unsigned int maximum_thread_count) {
	
	thread_pool->max_threads = maximum_thread_count;
}


MicrofeedMutex* microfeed_mutex_new(void) {
	MicrofeedMutex* mutex;
	
	mutex = microfeed_memory_allocate(MicrofeedMutex);
	mutex->mutex_implementation = functions.mutex_new();
	
	return mutex;
}

void microfeed_mutex_free(MicrofeedMutex* mutex) {
	functions.mutex_free(mutex->mutex_implementation);
	microfeed_memory_free(mutex);
}

void microfeed_mutex_lock(MicrofeedMutex* mutex) {
	functions.mutex_lock(mutex->mutex_implementation);
}

void microfeed_mutex_unlock(MicrofeedMutex* mutex) {
	functions.mutex_unlock(mutex->mutex_implementation);
}

static void* default_thread_new(void* (*function)(void* data), void* data) {
	pthread_attr_t attr;
	pthread_t* thread_implementation;

	pthread_attr_init(&attr);
	pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);

	thread_implementation = microfeed_memory_allocate(pthread_t);
	pthread_create(thread_implementation, &attr, (void* (*)(void*))function, data);

	pthread_attr_destroy(&attr);

	return (void*)thread_implementation;
}

static void default_thread_free(void* thread_implementation) {
	microfeed_memory_free(thread_implementation);
}

static void* default_thread_get_current(void) {
	pthread_t self;
	MicrofeedThread* thread;
	pthread_t* thread_implementation;
	
	self = pthread_self();
	
	for (thread = threads; thread; thread = thread->next) {
		if (thread->thread_implementation && *((pthread_t*)thread->thread_implementation) == self) {
			break;
		}
	}
	if (thread) {
		thread_implementation = thread->thread_implementation;
	} else {
		thread_implementation = microfeed_memory_allocate(pthread_t);
		*thread_implementation = self;
	}
	
	return (void*)thread_implementation;
}

static void default_thread_send_signal(void* thread_implementation, int signal_number) {
	pthread_kill(*(pthread_t*)thread_implementation, signal_number);
}

static void default_thread_join(void* thread_implementation) {
	pthread_join(*(pthread_t*)thread_implementation, NULL);
}

static void* default_mutex_new(void) {
	pthread_mutex_t* mutex_implementation;
	
	mutex_implementation = microfeed_memory_allocate(pthread_mutex_t);
	pthread_mutex_init(mutex_implementation, NULL);
	
	return (void*)mutex_implementation;
}

static void default_mutex_free(void* mutex_implementation) {
	pthread_mutex_destroy((pthread_mutex_t*)mutex_implementation);
	microfeed_memory_free(mutex_implementation);
}

static void default_mutex_lock(void* mutex_implementation) {
	struct timespec abs_timeout;
	
	pthread_mutex_lock((pthread_mutex_t*)mutex_implementation);

/*	abs_timeout.tv_sec = time(NULL) + 120;
	abs_timeout.tv_nsec = 0;
	if (pthread_mutex_timedlock((pthread_mutex_t*)mutex_implementation, &abs_timeout)) {
		fprintf(stderr, "ERROR: Could not lock mutex: %p\n", mutex_implementation);
		abort();
	}
*/
}

static void default_mutex_unlock(void* mutex_implementation) {
	pthread_mutex_unlock((pthread_mutex_t*)mutex_implementation);
}

static void* thread_function(void* data) {
	MicrofeedThread* thread;
	MicrofeedThreadPool* thread_pool;
	void* thread_implementation;
	
	thread = (MicrofeedThread*)data;

	microfeed_mutex_lock(thread->mutex);

	thread_pool = thread->thread_pool;
	thread_implementation = thread->thread_implementation;

	microfeed_mutex_unlock(thread->mutex);

	do {
		thread->out_data = thread->function(thread->in_data);
	
		if (thread->exit_callback) {
			thread->exit_callback(thread, thread->user_data);
		}
	
		microfeed_mutex_lock(thread->mutex);
	
		thread->reference_count--;
		if (thread->reference_count == 0) {
			microfeed_thread_free(thread);
		} else {
			thread->thread_implementation = NULL;

			microfeed_mutex_unlock(thread->mutex);
		}

		thread = NULL;
		
		if (thread_pool) {
			microfeed_mutex_lock(thread_pool->mutex);
			if (thread_pool->started_threads <= thread_pool->max_threads && microfeed_store_get_size(thread_pool->waiting_threads) > 0) {
				thread = microfeed_store_remove_index(thread_pool->waiting_threads, 0, MicrofeedThread);
				thread->thread_implementation = thread_implementation;
			} else {
				thread_pool->started_threads--;
				if (thread_pool->exit_callback) {
					thread_pool->exit_callback(thread, thread_pool->user_data);
				}
			}
			
			microfeed_mutex_unlock(thread_pool->mutex);
		}
	} while (thread);
	
	functions.thread_free(thread_implementation);

	return NULL;	
}
